import torch
import torch.nn as nn
import torch.nn.functional as F
from layers.Transformer_EncDec import Encoder, EncoderLayer
from layers.SelfAttention_Family import FullAttention, AttentionLayer
from layers.Embed import DataEmbedding_inverted,PatchEmbedding
import numpy as np
class FlattenHead(nn.Module):
    def __init__(self, n_vars, nf, target_window, head_dropout=0):
        super().__init__()
        self.n_vars = n_vars
        self.flatten = nn.Flatten(start_dim=-2)
        self.linear = nn.Linear(nf, target_window)
        self.dropout = nn.Dropout(head_dropout)

    def forward(self, x):  # x: [bs x nvars x d_model x patch_num]
        x = self.flatten(x)
        x = self.linear(x)
        x = self.dropout(x)
        return x
class Model(nn.Module):
    def __init__(self, configs):
        super(Model, self).__init__()
        self.task_name = configs.task_name
        self.seq_len = configs.seq_len
        self.pred_len = configs.pred_len
        self.output_attention = configs.output_attention
        self.enc_embedding1 = DataEmbedding_inverted(configs.seq_len, configs.d_model, configs.embed, configs.freq,
                                                    configs.dropout)
        self.enc_embedding2 = DataEmbedding_inverted(configs.seq_len, configs.d_model, configs.embed, configs.freq,
                                                    configs.dropout)
        self.encoder2 = Encoder(
            [
                EncoderLayer(
                    AttentionLayer(
                        FullAttention(False, configs.factor, attention_dropout=configs.dropout,
                                      output_attention=configs.output_attention), configs.d_model, configs.n_heads),
                    configs.d_model,
                    configs.d_ff,
                    dropout=configs.dropout,
                    activation=configs.activation
                ) for l in range(configs.e_layers)
            ],
            norm_layer=torch.nn.LayerNorm(configs.d_model)
        )
        self.projection1 = nn.Linear(configs.d_model, configs.pred_len, bias=True)
        self.projection2 = nn.Linear(configs.d_model, configs.pred_len, bias=True)
        patch_len=16
        stride=8
        padding = stride
        self.patch_embedding1 = PatchEmbedding(
            configs.d_model, patch_len, stride, padding, configs.dropout)
        self.encoder1 = Encoder(
            [
                EncoderLayer(
                    AttentionLayer(
                        FullAttention(False, configs.factor, attention_dropout=configs.dropout,
                                      output_attention=configs.output_attention), configs.d_model, configs.n_heads),
                    configs.d_model,
                    configs.d_ff,
                    dropout=configs.dropout,
                    activation=configs.activation
                ) for l in range(configs.e_layers)
            ],
            norm_layer=torch.nn.LayerNorm(configs.d_model)
        )
        self.linear = nn.Linear(96+configs.c_out,96)

        # Prediction Head
        self.head_nf = configs.d_model * \
                       int((configs.seq_len - patch_len) / stride + 2)
        if self.task_name == 'long_term_forecast' or self.task_name == 'short_term_forecast':
            self.head = FlattenHead(configs.enc_in, self.head_nf, configs.pred_len,
                                    head_dropout=configs.dropout)

    def forecast(self, x_enc, x_mark_enc, x_dec, x_mark_dec):
        means = x_enc.mean(1, keepdim=True).detach()
        x_enc = x_enc - means
        stdev = torch.sqrt(torch.var(x_enc, dim=1, keepdim=True, unbiased=False) + 1e-5)
        x_enc /= stdev
        B, T, N = x_enc.shape
        x_enc1 = x_enc.permute(0, 2, 1)
        enc_out1, n_vars = self.patch_embedding1(x_enc1)
        enc_out1, attns = self.encoder1(enc_out1)
        #print(enc_out1.shape)
        enc_out1 = torch.reshape(
            enc_out1, (-1, n_vars, enc_out1.shape[-2], enc_out1.shape[-1]))
        #print(enc_out1.shape)
        enc_out1 = enc_out1.permute(0, 1, 3, 2)
        dec_out1 = self.head(enc_out1)  
        dec_out1 = dec_out1.permute(0, 2, 1)
        dec_out1 = dec_out1[:,:,:N]
        dec_out1 = dec_out1 * (stdev[:, 0, :].unsqueeze(1).repeat(1, self.pred_len, 1))
        dec_out1 = dec_out1 + (means[:, 0, :].unsqueeze(1).repeat(1, self.pred_len, 1))
        
        enc_out2 = self.enc_embedding2(x_enc, x_mark_enc)
        enc_out2, attns = self.encoder2(enc_out2, attn_mask=None)
        dec_out2 = self.projection2(enc_out2).permute(0, 2, 1)[:, :, :N]
        return dec_out1,dec_out2


    def forward(self, x_enc, x_mark_enc, x_dec, x_mark_dec, mask=None):
        dec_out1,dec_out2 = self.forecast(x_enc, x_mark_enc, x_dec, x_mark_dec)
        if self.training:
            return dec_out1[:, -self.pred_len:, :],dec_out2[:, -self.pred_len:, :]
        else:
            return dec_out1[:, -self.pred_len:, :],dec_out2[:, -self.pred_len:, :],dec_out1[:, -self.pred_len:, :]+dec_out2[:, -self.pred_len:, :] 

