#PACKAGES -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

using Optim, Plots, DelimitedFiles, LinearAlgebra, Random, StatsBase, FiniteDifferences, LaTeXStrings , EasyFit, Printf, FFTW, Pkg, Noise, Clustering, Dierckx, BSplineKit, MultivariateStats, Flux, Combinatorics, Bigsimr, DataFrames, JLD, Base.Threads

#FUNCTIONS -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#Functions for optimization and compute expectation of the cost function
cd(@__DIR__)#to go back to the directory of the script
include("../MyFunctions/myFunctions_NeurIPS_2025.jl")

#DEFINITION OF THE PROBLEM -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#TASK PARAMETERS -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

println("\n------- NEW RUN ---------------------------------------------------- \n")

#PROBLEM PARAMETERS -- Reaching Task in (Todorov, 2005) and (Damiani et al, 2025)

dimension_of_state = 4
dimension_of_control = 1
dimension_of_observation = 1

Δt = 0.010 #seconds
T = 100 #time steps
m = 1 #Kg, mass of the hand, modelled as a point mass
target_position_mean = 0.15 #meters; mean of the gaussian distribution from which the target position is extracted. We use here the params used in the estimation task of 
#(Todorov, 2005), so that the target is in the periphery and multiplicative sensory noise plays a role.
target_position_std = 0.0 
tau_1 = 0.04 #seconds; tau_1 and tau_2 are the time constants of the second oreder low pass filter to map the control signal into the force acting on the hand
tau_2 = 0.04 #seconds
r = 0.00001 #as in the reaching atsk of (Todorov, 2005)
w_v = 0.2 #w_v and #w_f are computed by considering Todorov's choices in (Todorov, 2005) and rescaling them to our case in which target_displacement is 0.15 and N_steps = 100 
#(we thus just divided by 2 the values used in the paper)
w_f = 0.01

#NOISE terms -- Note that in the sensory-feedback dybamics we need to have non-zero noise, otherwise we have numerical problems 
#[it happens also in Todorov's approach]

σ_ξ = 0.1 
σ_ω_add_sensory_noise = 0.1 
σ_ρ_mult_sensory_noise = 0.5
σ_ϵ_mult_control_noise = 0.5 

σ_η_internal_noise = 0.1 #meters, as in the estimation task of (Todorov, 2005).
#we also consider here possible internal fluctuations on the other components of the state estimate 
σ_η_internal_noise_velocity = 0.1
σ_η_internal_noise_force = 0.1
σ_η_internal_noise_filtered_control = 0.1

#Initial conditions (mean state and state estimate, and their covariances) -- Note that initial state and state estimate are considered to be uncorrelated at t=1 (initial time step)

#Mean initial state and state estimates -- Note that we assume <x_1> = <z_1>
#note that x_state_0 is the initial mean state of the vector
x_state_mean_0 = zeros(dimension_of_state) #meters. 
x_state_mean_0[1] = target_position_mean 
x_state_mean_0[2] = 0.0
x_state_mean_0[3] = 0.0
x_state_mean_0[4] = 0.0

x_state_estimate_mean_0 = zeros(dimension_of_state)
x_state_estimate_mean_0[:] .= x_state_mean_0[:]

#State covariance 
target_position_variance = target_position_std^2
velocity_initial_variance = 0
force_initial_variance = 0
filtered_control_initial_variance = 0

Σ_1_x = Diagonal(zeros(dimension_of_state)) #intial covariance of the state

Σ_1_x[1,1] = target_position_variance
Σ_1_x[2,2] = velocity_initial_variance
Σ_1_x[3,3] = force_initial_variance
Σ_1_x[4,4] = filtered_control_initial_variance

#State estimate covariance 
target_position_estimate_initial_variance = 0
velocity_estimate_initial_variance = 0
force_estimate_initial_variance = 0
filtered_control_estimate_initial_variance = 0

Σ_1_z = Diagonal(zeros(dimension_of_state)) #intial covariance of the state estimate 

Σ_1_z[1,1] = target_position_estimate_initial_variance
Σ_1_z[2,2] = velocity_estimate_initial_variance
Σ_1_z[3,3] = force_estimate_initial_variance
Σ_1_z[4,4] = filtered_control_estimate_initial_variance

#auxialiary variables 
x_1_mean = deepcopy(x_state_mean_0)
z_1_mean = deepcopy(x_state_estimate_mean_0)   

#Matrices of the problem 
#Define the dynamical system
A = zeros(dimension_of_state, dimension_of_state)
B = zeros(dimension_of_state, dimension_of_control)
H = zeros(dimension_of_observation, dimension_of_state)
D = zeros(dimension_of_observation, dimension_of_state)

Ω_ξ = Diagonal([σ_ξ^2, 0, 0, 0])
Ω_ω = σ_ω_add_sensory_noise^2 #Additive sensory noise (1D sensory feedback)
Ω_η = Diagonal([σ_η_internal_noise, σ_η_internal_noise_velocity^2, σ_η_internal_noise_force^2, σ_η_internal_noise_filtered_control^2])

A[1,:] = [1, Δt, 0, 0]
A[2,:] = [0, 1, Δt/m, 0]
A[3,:] = [0, 0, 1-Δt/tau_2, Δt/tau_2]
A[4,:] = [0, 0, 0, 1-Δt/tau_1]

B[:] = [0, 0, 0, Δt/tau_1]

C = σ_ϵ_mult_control_noise .* B

H[1,:] = [1, 0, 0, 0] #we are selecting only the position as observable 

D[1,:] = σ_ρ_mult_sensory_noise .* H[1,:]

#Matrix for cost function (quadratic form)
p_vec = [1; 0; 0; 0]
v_vec = [0; w_v; 0; 0]
f_vec = [0; 0; w_f; 0]

R_matrix = zeros(dimension_of_control, dimension_of_control, T)
for i in 1:T-1
    R_matrix[:,:,i] .= r/(T-1)
end

Q_matrix = zeros(dimension_of_state, dimension_of_state, T)

Q_T = p_vec*transpose(p_vec) .+ v_vec*transpose(v_vec) .+ f_vec*transpose(f_vec) #Matrix for the cost at the last time step 
Q_matrix[:,:,T] .= Q_T[:,:]

#OPTIMIZATION PARAMETERS ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#Todorov's algorithm
N_iter_Todorov = 100

#Optimization with Lagrange multipliers method 
N_iterations_Lagrange = 100

#Numerical GD algorithm
algorithm_GD = GradientDescent()
iterations_GD = 5000
#iterations_GD = 50000
# Specify options for the optimization algorithm
options_GD = Optim.Options(
    # Step size for gradient descent
    iterations = iterations_GD,  # Number of iterations
    store_trace = false   # Show optimization trace
)

#TODOROV OPTIMIZATION -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

println("\n------- (1) Todorov's optimization --------------------------\n")

t_execution_TOD = @elapsed begin

    L_TOD, K_TOD = Todorov_optimization_multidimensional_case(A, B, H, C, D, T, dimension_of_state, dimension_of_control, dimension_of_observation, x_state_mean_0, Σ_1_x, Σ_1_z, Q_matrix, R_matrix, Ω_ξ, Ω_ω, Ω_η, N_iter_Todorov)

end

println("Execution time for TOD algorithm: ", t_execution_TOD, " seconds")

#Expected cost using moments propagation
expected_cost_mom_prop_TOD = expected_cost_raw_moments_propagation(T, dimension_of_state, dimension_of_control, dimension_of_observation, K_TOD, L_TOD, A, B, H, C, D, x_1_mean, z_1_mean, Σ_1_x, Σ_1_z, Q_matrix, R_matrix, Ω_ξ, Ω_ω, Ω_η)
#Expected cost using Todorov's algorithm
expected_cost_TOD_alg_TOD = expected_cost_using_Todorov(L_TOD, K_TOD, A, B, H, C, D, T, dimension_of_state, x_state_mean_0, Σ_1_x, Q_matrix, R_matrix, Ω_ξ, Ω_ω, Ω_η)
#Expected cost using trace formula [our formula proved by induction]
expected_cost_TOD_using_trace_formula = expected_cost_using_trace_formula_induction(L_TOD, K_TOD, A, B, H, C, D, T, dimension_of_state, x_state_mean_0, x_state_mean_0, Σ_1_x, Σ_1_z, Q_matrix, R_matrix, Ω_ξ, Ω_ω, Ω_η)

println("- Expected cost using moment propagation (Todorov): $expected_cost_mom_prop_TOD\n")
println("- Expected cost using trace formula (Todorov):$expected_cost_TOD_using_trace_formula\n")

#LAGRANGE MULTIPLIER OPTIMIZATION -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

println("\n------- (2) Lagrange multipliers optimization ---------------\n")

K_initial = zeros(dimension_of_state, dimension_of_observation, T - 1)
L_initial = zeros(dimension_of_control, dimension_of_state, T - 1)
#Initial conditions for the parameters to be optimized
K_initial[:,:,:] .= K_TOD[:,:,:]
L_initial[:,:,:] .= L_TOD[:,:,:]

t_execution_Lag_Mul = @elapsed begin

    #Lagrange multipliers optimization
    K_matrix_Lag_Mul_whole, L_matrix_Lag_Mul_whole, cost_mom_prop_Lag_Mul_whole = Optimal_Control_Estimation_with_Lagrange_Multipliers(N_iterations_Lagrange, dimension_of_state, dimension_of_control, dimension_of_observation, K_initial, L_initial, A, B, H, C, D, Ω_ξ, Ω_ω, Ω_η, Q_matrix, R_matrix, Σ_1_x, Σ_1_z, x_1_mean, z_1_mean)

end

println("Execution time for Lag Mul algorithm: ", t_execution_Lag_Mul, " seconds")

#Expected cost using moments propagation
expected_cost_mom_prop_Lag_Mul_whole = expected_cost_raw_moments_propagation(T, dimension_of_state, dimension_of_control, dimension_of_observation, K_matrix_Lag_Mul_whole, L_matrix_Lag_Mul_whole, A, B, H, C, D, x_1_mean, z_1_mean, Σ_1_x, Σ_1_z, Q_matrix, R_matrix, Ω_ξ, Ω_ω, Ω_η)
#Expected cost using Todorov's algorithm
expected_cost_TOD_alg_Lag_Mul_whole = expected_cost_using_Todorov(L_matrix_Lag_Mul_whole, K_matrix_Lag_Mul_whole, A, B, H, C, D, T, dimension_of_state, x_state_mean_0, Σ_1_x, Q_matrix, R_matrix, Ω_ξ, Ω_ω, Ω_η)
#Expected cost using trace formula [our formula proved by induction]
expected_cost_Lag_Mul_whole_using_trace_formula = expected_cost_using_trace_formula_induction(L_matrix_Lag_Mul_whole, K_matrix_Lag_Mul_whole, A, B, H, C, D, T, dimension_of_state, x_state_mean_0, x_state_mean_0, Σ_1_x, Σ_1_z, Q_matrix, R_matrix, Ω_ξ, Ω_ω, Ω_η)

println("- Expected cost using moment propagation (Lag_Mul_whole): $expected_cost_mom_prop_Lag_Mul_whole\n")
println("- Expected cost using trace formula (Lag_Mul_whole):$expected_cost_Lag_Mul_whole_using_trace_formula\n")

#GD OPTIMIZATION -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

println("\n------- (3) Numerical GD optimization --------------------------\n")

t_execution_GD = @elapsed begin

    #Objective function
    cost_function_optimization_GD(x) = expected_cost_using_mom_prop_for_GD_WHOLE_optimization_raw_moments(x, A, B, H, C, D, T, dimension_of_state, dimension_of_control, dimension_of_observation, x_1_mean, z_1_mean, Σ_1_x, Σ_1_z, Q_matrix, R_matrix, Ω_ξ, Ω_ω, Ω_η)

    x_0 = zeros(dimension_of_control*dimension_of_state*(T - 1)+dimension_of_state*dimension_of_observation*(T - 1))

    x_0[1:dimension_of_control*dimension_of_state*(T - 1)] = L_matrix_Lag_Mul_whole[:] #initial condition for the filters
    x_0[dimension_of_control*dimension_of_state*(T - 1)+1:dimension_of_control*dimension_of_state*(T - 1)+dimension_of_state*dimension_of_observation*(T - 1)] = K_matrix_Lag_Mul_whole[:] #initial condition for the control gains

    result_GD = optimize(cost_function_optimization_GD, x_0, algorithm_GD, options_GD)

    x_opt_GD = result_GD.minimizer
    expected_cost_GD_optim = result_GD.minimum

    L_GD = reshape(x_opt_GD[1:dimension_of_control*dimension_of_state*(T - 1)], dimension_of_control, dimension_of_state, T - 1)
    K_GD = reshape(x_opt_GD[dimension_of_control*dimension_of_state*(T - 1)+1:dimension_of_control*dimension_of_state*(T - 1)+dimension_of_state*dimension_of_observation*(T - 1)], dimension_of_state, dimension_of_observation, T - 1)

end

println("Execution time for GD algorithm: ", t_execution_GD, " seconds")

#Expected cost using moments propagation
expected_cost_mom_prop_GD = expected_cost_raw_moments_propagation(T, dimension_of_state, dimension_of_control, dimension_of_observation, K_GD, L_GD, A, B, H, C, D, x_1_mean, z_1_mean, Σ_1_x, Σ_1_z, Q_matrix, R_matrix, Ω_ξ, Ω_ω, Ω_η)
#Expected cost using GDorov's algorithm
expected_cost_GD_alg_GD = expected_cost_using_Todorov(L_GD, K_GD, A, B, H, C, D, T, dimension_of_state, x_state_mean_0, Σ_1_x, Q_matrix, R_matrix, Ω_ξ, Ω_ω, Ω_η)
#Expected cost using trace formula [our formula proved by induction]
expected_cost_GD_using_trace_formula = expected_cost_using_trace_formula_induction(L_GD, K_GD, A, B, H, C, D, T, dimension_of_state, x_state_mean_0, x_state_mean_0, Σ_1_x, Σ_1_z, Q_matrix, R_matrix, Ω_ξ, Ω_ω, Ω_η)

println("- Expected cost using moment propagation (GD): $expected_cost_mom_prop_GD\n")
println("- Expected cost using trace formula (GD):$expected_cost_GD_using_trace_formula\n")


###SAVE THE DATA ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

cd(@__DIR__)#to go back to the directory of the script
folder_location = "../New_Data/EC_1D_Reaching_Task"

# Create a dictionary to store the matrices
Data_dict = Dict(
    "t_execution_TOD" => t_execution_TOD,
    "t_execution_Lag_Mul" => t_execution_Lag_Mul,
    "t_execution_GD" => t_execution_GD,
    "dimension_of_state" => dimension_of_state,
    "dimension_of_control" => dimension_of_control,
    "dimension_of_observation" => dimension_of_observation,
    "N_iterations_Lagrange" => N_iterations_Lagrange,
    "N_iter_Todorov" => N_iter_Todorov,
    "iterations_GD" => iterations_GD,
    "A" => A,
    "B" => B,
    "H" => H,
    "D" => D,
    "C" => C,
    "Ω_ξ" => Ω_ξ,
    "Ω_ω" => Ω_ω,
    "Ω_η" => Ω_η,
    "Σ_1_x" => Σ_1_x,
    "Σ_1_z" => Σ_1_z,
    "x_state_mean_0" => x_state_mean_0,
    "x_state_estimate_mean_0" => x_state_estimate_mean_0,
    "x_1_mean" => x_1_mean,
    "z_1_mean" => z_1_mean,
    "Q_matrix" => Q_matrix,
    "R_matrix" => R_matrix,
    "L_TOD" => L_TOD,
    "K_TOD" => K_TOD,
    "L_matrix_Lag_Mul_whole" => L_matrix_Lag_Mul_whole,
    "K_matrix_Lag_Mul_whole" => K_matrix_Lag_Mul_whole,
    "L_GD" => L_GD,
    "K_GD" => K_GD,
    "cost_mom_prop_Lag_Mul_whole" => cost_mom_prop_Lag_Mul_whole,
    "expected_cost_mom_prop_TOD" => expected_cost_mom_prop_TOD,
    "expected_cost_mom_prop_Lag_Mul_whole" => expected_cost_mom_prop_Lag_Mul_whole,
    "expected_cost_mom_prop_GD" => expected_cost_mom_prop_GD,
    "expected_cost_TOD_alg_TOD" => expected_cost_TOD_alg_TOD,
    "expected_cost_TOD_using_trace_formula" => expected_cost_TOD_using_trace_formula,
    "expected_cost_TOD_alg_Lag_Mul_whole" => expected_cost_TOD_alg_Lag_Mul_whole,
    "expected_cost_Lag_Mul_whole_using_trace_formula" => expected_cost_Lag_Mul_whole_using_trace_formula,
    "expected_cost_GD_alg_GD" => expected_cost_GD_alg_GD,
    "expected_cost_GD_using_trace_formula" => expected_cost_GD_using_trace_formula
)

# Specify the file path
file_name_save_Data_dict = "EC_1D_Reaching_Task_new.jld2"
file_path_save_Data_dict = folder_location * "/" *file_name_save_Data_dict

# Save the dictionary to the file
@save file_path_save_Data_dict Data_dict
