# -*- coding: utf-8 -*-
import numpy as np
import torch

# From https://github.com/Bjarten/early-stopping-pytorch
class EarlyStopping:
    """Early stops the training if validation loss doesn't improve after a given patience."""
    def __init__(self, patience=50, verbose=False, delta=0, checkpoint_pth='chechpoint.pt'):
        """
        Args:
            patience (int): How long to wait after last time validation loss improved.
                            Default: 7
            verbose (bool): If True, prints a message for each validation loss improvement.
                            Default: False
            delta (float): Minimum change in the monitored quantity to qualify as an improvement.
                            Default: 0
        """
        self.patience = patience
        self.verbose = verbose
        self.counter = 0
        self.best_score = None
        self.early_stop = False
        self.val_loss_min = np.Inf
        self.delta = delta
        self.checkpoint_pth = checkpoint_pth

    def __call__(self, val_loss, model, classifier):
        score = val_loss
        if self.best_score is None:
            self.best_score = score
            self.save_checkpoint(val_loss, model, classifier, self.checkpoint_pth)
        elif score <= self.best_score + self.delta:
            self.counter += 1
            if self.counter >= self.patience:
                self.early_stop = True
        else:
            self.best_score = score
            self.save_checkpoint(val_loss, model, classifier, self.checkpoint_pth)
            self.counter = 0

    def save_checkpoint(self, val_loss, model, classifier, checkpoint_pth):
        if model is not None:
            if self.verbose:
                print(f'Validation accuracy increased ({self.val_loss_min:.6f} --> {val_loss:.6f}).  Saving model ...')
            torch.save({'model': model.state_dict(), 
                        'classifier': classifier.state_dict()}, checkpoint_pth)
            self.val_loss_min = val_loss