"""
Train a diffusion model on images.
"""

import argparse

import numpy as np
import torch
import wandb

from guided_diffusion import dist_util, logger
from guided_diffusion.ckpts_utils import ADDITIONAL_SAVESTEPS_CIFAR, ADDITIONAL_SAVESTEPS_IMAGENET
from guided_diffusion.image_datasets import load_data
from guided_diffusion.resample import create_named_schedule_sampler
from guided_diffusion.script_util import (
    add_dict_to_argparser,
    args_to_dict,
    create_model_and_diffusion,
    model_and_diffusion_defaults,
)
from guided_diffusion.train_util import TrainLoop


def main():
    args = create_argparser().parse_args()

    run_train(args=args)


def run_train(args):
    dist_util.setup_dist()
    setup_seed(seed=args.seed)
    logger.configure(wandb_project=args.log_wandb)

    logger.log("creating model and diffusion...")

    model, diffusion = create_model_and_diffusion(**args_to_dict(args, model_and_diffusion_defaults().keys()))

    if args.log_wandb:
        wandb.watch(model, log_freq=100)

    model.to(dist_util.dev())
    schedule_sampler = create_named_schedule_sampler(args.schedule_sampler, diffusion)

    logger.log("creating data loader...")
    data = load_data(
        data_dir=args.data_dir,
        batch_size=args.batch_size,
        image_size=args.image_size,
        class_cond=args.class_cond,
        dl_seed=args.seed,
    )

    logger.log("training...")
    TrainLoop(
        model=model,
        diffusion=diffusion,
        data=data,
        batch_size=args.batch_size,
        microbatch=args.microbatch,
        lr=args.lr,
        ema_rate=args.ema_rate,
        log_interval=args.log_interval,
        save_interval=args.save_interval,
        resume_checkpoint=args.resume_checkpoint,
        use_fp16=args.use_fp16,
        fp16_scale_growth=args.fp16_scale_growth,
        schedule_sampler=schedule_sampler,
        weight_decay=args.weight_decay,
        lr_anneal_steps=args.lr_anneal_steps,
        add_save_steps=(
            (ADDITIONAL_SAVESTEPS_CIFAR if "cifar" in args.data_dir else ADDITIONAL_SAVESTEPS_IMAGENET)
            if args.additional_save_steps
            else []
        ),
    ).run_loop()


def setup_seed(seed: int):
    print(f"Setting seed = {seed}")
    torch.manual_seed(seed)
    np.random.seed(seed)
    torch.cuda.manual_seed(seed)
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False


def create_argparser():
    defaults = dict(
        data_dir="",
        schedule_sampler="uniform",
        lr=1e-4,
        weight_decay=0.0,
        lr_anneal_steps=0,
        batch_size=1,
        microbatch=-1,  # -1 disables microbatches
        ema_rate="0.9999",  # comma-separated list of EMA values
        log_interval=10,
        save_interval=10000,
        resume_checkpoint="",
        use_fp16=False,
        fp16_scale_growth=1e-3,
        seed=0,
        log_wandb=False,
        additional_save_steps=False,
    )
    defaults.update(model_and_diffusion_defaults())
    parser = argparse.ArgumentParser()
    add_dict_to_argparser(parser, defaults)
    return parser


if __name__ == "__main__":
    main()
