import numpy as np
import torch


def angle_axis(angle, axis):
    # type: (float, np.ndarray) -> float
    r"""Returns a 3x3 rotation matrix that performs a rotation around axis by angle

    Parameters
    ----------
    angle : float
        Angle to rotate by
    axis: np.ndarray
        Axis to rotate about

    Returns
    -------
    torch.Tensor
        3x3 rotation matrix
    """
    u = axis / np.linalg.norm(axis)
    cosval, sinval = np.cos(angle), np.sin(angle)

    # yapf: disable
    cross_prod_mat = np.array([[0.0, -u[2], u[1]],
                               [u[2], 0.0, -u[0]],
                               [-u[1], u[0], 0.0]])

    R = torch.from_numpy(
        cosval * np.eye(3)
        + sinval * cross_prod_mat
        + (1.0 - cosval) * np.outer(u, u)
    )
    # yapf: enable
    return R.float()


def random_sample(points, num_points=2048):
    sampled_idxes = np.random.choice(points.shape[0], num_points, replace=num_points > points.shape[0])
    return points[sampled_idxes]


def pc_normalize(pc):
    # unite cube normalization
    centroid = np.mean(pc, axis=0)
    pc = pc - centroid
    m = np.max(np.sqrt(np.sum(pc ** 2, axis=1)))
    pc = pc / m
    return pc


class TwoCropTransform:
    """Create two crops of the same image"""
    def __init__(self, transform):
        self.transform = transform

    def __call__(self, x):
        return [self.transform(x), self.transform(x)]


class RandomSample(object):
    def __init__(self, num_points=1024):
        self.num_points = num_points

    def __call__(self, points):
        sampled_idxes = np.random.choice(points.shape[0], self.num_points, replace=self.num_points > points.shape[0])
        return points[sampled_idxes]


class PointCloudToTensor(object):
    def __call__(self, points):
        return torch.from_numpy(points).float()


class Center(object):
    def __call__(self, points):
        points[:, :3] -= np.mean(points[:, :3], axis=0)
        return points


class Scale(object):
    def __call__(self, points):
        diag = np.max(np.absolute(points[:, :3]))  # normalize_scale from @antoalli
        points[:, :3] /= (diag + np.finfo(float).eps)
        return points


class AugmentScale(object):
    def __init__(self, low=0.8, high=1.25):
        self.low, self.high = low, high

    def __call__(self, points):
        scaler = np.random.uniform(self.low, self.high)
        points[:, 0:3] *= scaler
        return points


class AugmentRotate(object):
    def __init__(self, axis=np.array([0.0, 1.0, 0.0])):
        self.axis = axis

    def __call__(self, points):
        rotation_angle = np.random.uniform() * 2 * np.pi
        rotation_matrix = angle_axis(rotation_angle, self.axis)

        normals = points.size(1) > 3
        if not normals:
            return torch.matmul(points, rotation_matrix.t())
        else:
            pc_xyz = points[:, 0:3]
            pc_normals = points[:, 3:]
            points[:, 0:3] = torch.matmul(pc_xyz, rotation_matrix.t())
            points[:, 3:] = torch.matmul(pc_normals, rotation_matrix.t())

            return points


class AugmentRotatePerturbation(object):
    def __init__(self, angle_sigma=0.06, angle_clip=0.18):
        self.angle_sigma, self.angle_clip = angle_sigma, angle_clip

    def _get_angles(self):
        angles = np.clip(
            self.angle_sigma * np.random.randn(3), -self.angle_clip, self.angle_clip
        )

        return angles

    def __call__(self, points):
        angles = self._get_angles()
        Rx = angle_axis(angles[0], np.array([1.0, 0.0, 0.0]))
        Ry = angle_axis(angles[1], np.array([0.0, 1.0, 0.0]))
        Rz = angle_axis(angles[2], np.array([0.0, 0.0, 1.0]))

        rotation_matrix = torch.matmul(torch.matmul(Rz, Ry), Rx)

        normals = points.size(1) > 3
        if not normals:
            return torch.matmul(points, rotation_matrix.t())
        else:
            pc_xyz = points[:, 0:3]
            pc_normals = points[:, 3:]
            points[:, 0:3] = torch.matmul(pc_xyz, rotation_matrix.t())
            points[:, 3:] = torch.matmul(pc_normals, rotation_matrix.t())

            return points


class AugmentJitter(object):
    def __init__(self, std=0.01, clip=0.05):
        self.std, self.clip = std, clip

    def __call__(self, points):
        jittered_data = points.new(points.size(0), 3).normal_(mean=0.0, std=self.std).clamp_(-self.clip, self.clip)
        points[:, 0:3] += jittered_data
        return points


class AugmentTranslate(object):
    def __init__(self, translate_range=0.1):
        self.translate_range = translate_range

    def __call__(self, points):
        translation = np.random.uniform(-self.translate_range, self.translate_range)
        points[:, 0:3] += translation
        return points


class AugmentRandomInputDropout(object):
    def __init__(self, max_dropout_ratio=0.875):
        assert 0 <= max_dropout_ratio < 1
        self.max_dropout_ratio = max_dropout_ratio

    def __call__(self, points):
        pc = points.numpy()

        dropout_ratio = np.random.random() * self.max_dropout_ratio  # 0~0.875
        drop_idx = np.where(np.random.random((pc.shape[0])) <= dropout_ratio)[0]
        if len(drop_idx) > 0:
            pc[drop_idx] = pc[0]  # set to the first point

        return torch.from_numpy(pc).float()


if __name__ == '__main__':
    rot_angle = np.random.uniform() * 2 * np.pi
    axis = np.array([0.0, 1.0, 0.0])
    rot_matrix = angle_axis(rot_angle, axis)
    print(rot_matrix)