from .base import Doctor
from utils.myutils import doctorMessage
import json

class DoctorCheck(Doctor):
    """
    The DoctorCheck class determines if the information is sufficient before asking.
    """
    def __init__(self, config, task, choices):
        super().__init__(config, task, choices)
        self.history = []
        # self.information = []
        self.max_rounds = int(self.config.get('doctor', 'max_questions'))

    def ask(self, answer):
        if answer is not None:
            self.history.append({"role": "patient", "content": answer})
        
        if self.finished():
            return None
        elif len(self.history) >= self.max_rounds * 2:
            doctorMessage("Truncated: The conversation has reached the maximum number of rounds.")
            return "Exceed the maximum number of rounds"
        
        system_prompt = (
            "You are a medical professional in medical inquiry. "
            "You ask ONE question at a time to gather the most useful information for diagnosis. \n"
            "During the consultation, your questions **MUST** cover the following key aspects:\n" \
            "1. **Demographics**: Age, sex/gender, occupation, and other relevant factors.\n" \
            "2. **Primary Symptoms**: Onset, duration, severity, and progression.\n" \
            "3. **Medical History**: Past conditions, medications, surgeries, family history.\n" \
            "4. **Examination/Test Results**: Key findings to differentiate diagnosis options.\n" \
            "5. **Lifestyle**: Daily habits, diet, alcohol, smoking, stress, etc."
            "Manage the dialogue efficiently and empathetically. "
            "Each question should be concise, clear, and respectful. "
            "Only provide one single question and nothing else."
        )

        if len(self.history) == 0:
            user_prompt = (
                "Begin by asking a suitable starting question. "
            )
        else:
            user_prompt = (
                f"You have asked the following {len(self.history)//2} questions and received these answers:\n"
                f"{json.dumps([msd['content'] for msd in self.history], indent=2, ensure_ascii=False)}\n"
                f"Possible choices are: {', '.join(self.choices)}\n\n"
                "Based on the above, ask ONE new question to collect the most critical missing information "
                "from one of the aspects (demographics, symptoms, history, exams/tests). "
                "If multiple are missing, prioritize the most fundamental for differential diagnosis. "
                "Avoid repeating or rephrasing previous questions."
            )

        messages = [
            {"role": "system", "content": system_prompt},
            {"role": "user", "content": user_prompt}
        ]

        new_question = self.client.get_response(messages=messages)
        
        self.history.append({"role": "doctor", "content": new_question})
        return new_question

    
    def finished(self):
        # Never finish if no history
        if len(self.history) == 0:
            return False

        system_prompt = (
            "You are a clinical decision readiness evaluator. "
            "Your task is to decide whether the DOCTOR has asked enough detailed questions in each category "
            "to safely make a diagnostic decision, regardless of whether the patient has answered them. "
            "Do NOT answer 'yes' unless the doctor's questions have explicitly covered:\n"
            "1. Demographics (age, sex/gender) with enough detail to identify important factors for diagnosis\n"
            "2. Primary symptoms, their duration, and their nature (severity, progression, etc.)\n"
            "3. Relevant medical history (past illness, medication, family history) with sufficient depth to rule out key conditions\n"
            "4. Key examination or test results that would be critical to differentiate the provided diagnosis choices\n\n"
            "For each of the above categories, ask whether the doctor has asked enough detailed questions to distinguish between the possible choices. "
            "If any category is underexplored or lacks sufficient detail for diagnosis, respond 'no'.\n"
            "Respond strictly with 'yes' or 'no'. Do not explain."
        )

        # Include full conversation history
        conversation = json.dumps(
            [f"{msg['role'].capitalize()}: {msg['content']}" for msg in self.history],  # Mark role as 'Doctor' or 'Patient'
            indent=2,
            ensure_ascii=False
        )

        user_prompt = (
            f"Task: {self.task}\nChoices: {', '.join(self.choices)}\n\n"
            "Based on the following conversation, has the doctor asked enough detailed questions in all required categories "
            "to confidently distinguish between the given diagnosis choices?\n"
            "Consider the necessary demographics, primary symptoms, relevant medical history, and "
            "key examination or test results needed to differentiate between the options."
            f"{conversation}\n\n"
            "Respond strictly with 'yes' or 'no'. Do not explain."
        )

        messages = [{"role": "system", "content": system_prompt},
                    {"role": "user", "content": user_prompt}]

        # Get LLM's response
        check = self.client.get_response(messages=messages).strip().lower()
        doctorMessage(f'Finished Check: {check}')

        if 'yes' in check:
            doctorMessage("The LLM-judge has determined that the doctor has asked enough detailed questions. The consultation ends.")
            return True

        return False