import json
from tqdm import tqdm

class EfficiencyEvaluator:
    def __init__(self, config, save_folder):
        self.config = config

    def evaluate(self, log_path):
        """
        Evaluate the efficiency by calculating the average number of interactions and doctor tokens.
        :param log_path: Path to the log file.
        :return: A dictionary containing the average scores and per-entry details.
        """
        # Step 1: Load and parse the log data
        with open(log_path, 'r', encoding='utf-8') as file:
            log_data = [json.loads(line) for line in file if line.strip()]

        details = []
        total_interactions = 0
        total_doctor_tokens = 0
        num_entries = len(log_data)

        # Step 2: Collect interactions and doctor tokens for each entry
        for entry in tqdm(log_data, total=num_entries, desc="Efficiency Evaluation"):
            interactions = entry.get('interactions', [])
            
            question_num = len(interactions)
            
            doctor_token = entry.get('doctor_token', 0)
            if isinstance(doctor_token, list):
                doctor_token = doctor_token[1]
            elif isinstance(doctor_token, (int, float)):
                doctor_token = doctor_token
                
            details.append({
                "case_id": entry.get("case_id"),
                "question_num": question_num,
                "token_cost": doctor_token
            })

            total_interactions += question_num
            total_doctor_tokens += doctor_token

        # Step 3: Calculate averages
        avg_interactions = total_interactions / num_entries if num_entries > 0 else 0.0
        avg_doctor_tokens = total_doctor_tokens / num_entries if num_entries > 0 else 0.0

        # Step 4: Return results
        return {
            "avg_scores": {
                "question_num": round(avg_interactions, 3),
                "token_cost": round(avg_doctor_tokens, 3),
            },
            "details": details,
        }
