from evaluators import load_evaluator
from utils.myutils import systemMessage
from utils.logger import save_scores
import os, datetime, pytz, json

class Evaluator:
    def __init__(self, config, folder):
        
        systemMessage("Begin initializing Evaluator...")
        
        time_tag = datetime.datetime.now(pytz.timezone('Asia/Shanghai')).strftime("%m-%d_%H-%M-%S")
        self.eval_log_folder = os.path.join(folder, f"eval_{time_tag}")        
        os.makedirs(self.eval_log_folder, exist_ok=True)
        
        with open(os.path.join(self.eval_log_folder, "meta.json"), "w", encoding="utf-8") as f:
            json.dump(config, f, indent=2, ensure_ascii=False)

        self.evaluators = {}
        
        self.mode = config["mode"]
        if 'static' in self.mode: # static evaluation mode
            self.evaluators['task_success'] = load_evaluator('task_success', config, self.eval_log_folder)
            return

        dimensions = config["dimensions"].split(',')
        dimensions = [dimension.strip() for dimension in dimensions]
        
        for dimension in dimensions:
            if dimension != '':
                self.evaluators[dimension] = load_evaluator(dimension, config, self.eval_log_folder)
        
        # systemMessage("Evaluator initialized.")
    
    def evaluate_from_log(self, log_path):
        ts, iq, dm, ef, ux = None, None, None, None, None
        
        # Call each evaluator and collect results
        ts = self.evaluators['task_success'].evaluate(log_path) if 'task_success' in self.evaluators else None
        # systemMessage("Task Success evaluation done.")
        
        iq = self.evaluators['info_quality'].evaluate(log_path) if 'info_quality' in self.evaluators else None
        # systemMessage("Information Quality evaluation done.")
        
        dm = self.evaluators['dialog_manage'].evaluate(log_path) if 'dialog_manage' in self.evaluators else None
        # systemMessage("Dialogue Management evaluation done.")
        
        ef = self.evaluators['efficiency'].evaluate(log_path) if 'efficiency' in self.evaluators else None
        # systemMessage("Efficiency evaluation done.")
        
        ux = self.evaluators['experience'].evaluate(log_path) if 'experience' in self.evaluators else None
        # systemMessage("User Experience evaluation done.")
        
        # Save detailed results
        details = {
            'task_success': ts['details'] if ts is not None and 'details' in ts else None,
            'info_quality': iq['details'] if iq is not None and 'details' in iq else None,
            'diag_manage': dm['details'] if dm is not None and 'details' in dm else None,
            'efficiency': ef['details'] if ef is not None and 'details' in ef else None,
            'experience': ux['details'] if ux is not None and 'details' in ux else None,
        }
        
        # Return average scores
        avg_scores = {
            'task_success': ts['avg_scores'] if ts is not None else None,
            'info_quality': iq['avg_scores'] if iq is not None else None,
            'diag_manage': dm['avg_scores'] if dm is not None else None,
            'efficiency': ef['avg_scores'] if ef is not None else None,
            'experience': ux['avg_scores'] if ux is not None else None,
        }
        
        all_scores = {
            'avg_scores': avg_scores,
            'details': details
        }
            
        save_scores(self.eval_log_folder, all_scores)
        
        return all_scores