import importlib
import time, datetime, pytz
import os, json, re
import traceback
from concurrent.futures import ProcessPoolExecutor, as_completed
from utils.config_loader import ConfigLoader
from utils.myutils import errorMessage
from maque import MAQuE

def snake_to_camel(snake_str):
    if '_' not in snake_str:
        return snake_str[0].upper() + snake_str[1:]
    return ''.join(word.capitalize() for word in snake_str.split('_'))

def camel_to_snake(camel_str: str) -> str:
    s = re.sub(r'(?<!^)(?=[A-Z])', '_', camel_str).lower()
    return s

def run_eval(config, doctor_name, patient_name, folder):
    start_time = time.time()

    try:
        doctor_class = None
        patient_class = None

        if doctor_name:  # interactive or evaluation mode
            print(f"### Doctor: {doctor_name}, Patient: {patient_name}")
            doctor_module = importlib.import_module(f"doctor.{camel_to_snake(doctor_name)}")
            patient_module = importlib.import_module(f"patient.{camel_to_snake(patient_name)}")
            doctor_class = getattr(doctor_module, snake_to_camel(doctor_name))
            patient_class = getattr(patient_module, snake_to_camel(patient_name))
        else:
            print(f"[Static Mode] Running static evaluation...")

        maque = MAQuE(config, doctor_class, patient_class, folder)
        result = maque.run()

    except Exception as e:
        error_message = str(e)
        error_trace = traceback.format_exc()
        result = {"error": error_message, "traceback": error_trace}
        print("Error occurred:")
        errorMessage(error_message)

    elapsed = time.time() - start_time
    return {
        "doctor": doctor_name,
        "patient": patient_name,
        "result": result,
        "time": f"{elapsed:.2f}s"
    }


def main():
    config_path = "config.cfg"
    config = ConfigLoader(config_path)
    max_processes = int(config.get('evaluation', 'max_processes'))
        
    session_name = datetime.datetime.now(pytz.timezone('Asia/Shanghai')).strftime("%m-%d_%H-%M-%S")

    base_log_folder = config.get_section("evaluation").get("base_log_folder", "log")
    
    continue_log_folder = config.get_section("evaluation").get("continue_log_folder", None)
    if continue_log_folder:
        folder = continue_log_folder
    else:
        folder = os.path.join(base_log_folder, session_name)
    os.makedirs(folder, exist_ok=True)

    mode = config.get('evaluation', 'mode')

    # Determine args list
    if 'interact' in mode:  # interactive mode
        print(f"Doctor model: {config.get('doctor', 'model')}")
        print("Dataset:", config.get('patient', 'data_path').split("/")[-1][:-6])
        
        # Get doctor and patient lists from config
        doctor_list = config.get('doctor', 'doctor_list').split(',')
        patient_list = config.get('patient', 'patient_list').split(',')
        args_list = [
            (config, doctor_name.strip(), patient_name.strip(), os.path.join(folder, f"{doctor_name.strip()}_x_{patient_name.strip()}"))
            for doctor_name in doctor_list
            for patient_name in patient_list
        ]
    elif 'evaluate' in mode:  # evaluation mode based on existing logs
        # Get doctor and patient lists from existing folders
        existing_folders = [d for d in os.listdir(folder) if os.path.isdir(os.path.join(folder, d))]
        existing_folders = [d.split("_x_") for d in existing_folders if "_x_" in d]
        doctor_list = list(set([pair[0] for pair in existing_folders]))
        patient_list = list(set([pair[1] for pair in existing_folders]))
        
        # Prepare args list
        args_list = [
            (config, doctor_name.strip(), patient_name.strip(), os.path.join(folder, f"{doctor_name.strip()}_x_{patient_name.strip()}"))
            for doctor_name in doctor_list
            for patient_name in patient_list
        ]
    elif 'static' in mode:  # static evaluation mode
        print("Dataset:", config.get('patient', 'data_path').split("/")[-1][:-6])
        args_list = [
            (config, None, None, os.path.join(folder, "static"))  # static模式不需要doctor/patient
        ]

    # Run evaluations in parallel
    results = []
    with ProcessPoolExecutor(max_workers=max_processes) as executor:
        futures = [executor.submit(run_eval, *args) for args in args_list]
        for future in as_completed(futures):
            results.append(future.result())

    # Save results
    with open(os.path.join(folder, f'results_{session_name}.json'), 'w', encoding='utf-8') as f:
        json.dump(results, f, indent=2, ensure_ascii=False)

    print(f"✅ All experiments completed and results saved in {os.path.join(folder, f'results_{session_name}.json')}. ")

if __name__ == "__main__":
    main()
