import random
import json
from .patient_select import PatientSelect
from utils.myutils import otherMessage, errorMessage

class PatientEmotion(PatientSelect):
    def __init__(self, config, data):
        super().__init__(config, data)
        
        # Refined list of emotions with specific behavioral impacts
        self.emotions = [
            ('anxiety', 'unease or dread about the future or health outcomes'),
            ('skepticism', 'doubt or mistrust regarding the doctor'),
            ('denial', 'refusal to acknowledge the severity of the condition'),
            ('frustration', 'feeling of dissatisfaction or helplessness')
        ]
        self.emotion_levels = [
            'not affected by emotion at all', 
            'barely affected by emotion', 
            'somewhat affected by emotion', 
            'significantly affected by emotion', 
            'completely dominated by emotion'
        ]
        
        # Predefined lists for language patterns with descriptions
        self.language_patterns = [
            ('formal', 'formal and structured language'),
            ('informal', 'casual and conversational language'),
            ('slang', 'incorporates colloquial terms and slang expressions')
        ]
        
        # Predefined lists for cognitive states with descriptions
        self.cognitive_states = [
            ('confused', 'unclear understanding of the situation, some disorientation'),
            ('distracted', 'difficulty focusing or maintaining attention, may lead to sudden topic shifts or revisiting previous discussions'),
            ('forgetful', 'difficulty remembering details')
        ]
        
        random.seed(self.case_id)
        
        # Randomly select an emotion and assign an intensity level
        self.current_emotion, self.emotion_description = random.choice(self.emotions)
        self.emotion_score = random.randint(0, 4)  # 0-4, corresponding to 5 emotional descriptions
        
        # Randomly select a language pattern and cognitive state
        self.language_pattern, self.language_pattern_description = random.choice(self.language_patterns)
        self.cognitive_state, self.cognitive_state_description = random.choice(self.cognitive_states)
        
        random.seed()
        
        otherMessage(f"Initial emotion: {self.current_emotion}, level: {self.emotion_score + 1}")
        otherMessage(f"Language pattern: {self.language_pattern}")
        otherMessage(f"Cognitive state: {self.cognitive_state}")
        
        self.additional_info["patient emotion"] = {"type": self.current_emotion, "description": self.emotion_description, "level": [self.emotion_score]}
        self.additional_info["language pattern"] = {"type": self.language_pattern, "description": self.language_pattern_description}
        self.additional_info["cognitive state"] = {"type": self.cognitive_state, "description": self.cognitive_state_description}

    def get_new_emotion_score(self, question):
        """ Update emotion intensity based on the question and return the new emotion score. """
        prompt = (
            f"A patient is currently feeling {self.current_emotion} ({self.emotion_description}) with an intensity level of {self.emotion_score+1}. "
            "- **Confrontational or challenging questions** may increase negative emotions.\n"
            "- **Reassuring or empathetic questions** may comfort the patient and decrease negative emotions.\n"
            f"The doctor asks: \"{question}\"\n"
            "After hearing the doctor's question, what will be the patient's new emotion intensity level? "
            "Your response must be an integer from 1 to 5 without any additional text."
        )
        
        try:
            response = self.client.get_response(
                messages=[{"role": "system", "content": "You are a psychological model that determines emotion intensity."},
                          {"role": "user", "content": prompt}]
            ).strip()
            
            # otherMessage(f"New emotion score: {response}")
            
            new_score = int(response) - 1  # Convert model's output (1-5) to internal range (0-4)
            if new_score < 0 or new_score > 4:
                # Fallback to current score
                new_score = self.emotion_score
            self.additional_info["patient emotion"]["level"].append(new_score)
            
            # Return the new score
            return new_score
        
        except (ValueError, IndexError, TypeError) as e:
            # Fallback to current score
            errorMessage(f"Error updating emotion score: {e}, remaining at level {self.emotion_score + 1}")
            return self.emotion_score

    def generate_emotional_answer(self, info, question):
        """ Generate an emotionally influenced answer based on the emotion intensity. """
        previous_interactions = " ".join([f"Q: {q}\nA: {a}" for q, a in zip(self.questions, self.answers)])
        
        # Adjust the response based on emotion intensity
        system_prompt = (
            f"You are feeling {self.current_emotion} ({self.emotion_description}), "
            f"and you are {self.emotion_levels[self.emotion_score]} when you speak. "
            f"You speak in {self.language_pattern} style ({self.language_pattern_description}). "
            f"Meanwhile, you are generally {self.cognitive_state} ({self.cognitive_state_description}.\n\n"
            "You will answer the doctor's question within 1-2 sentences, "
            "with your emotion, language pattern, and cognitive state affecting how you speak. "
            "Always refer to the patient using first-person pronouns, "
            "unless the patient is under 10 years old, unconscious, or deceased. "
            "In those cases, refer to the patient as a family member instead. "
            "Always include the information from the context, "
            "and NEVER make up any new information. "
            "Avoid repeated phrases or structured responses. "
        )
        
        if info:
            user_prompt = (
                f"Previous interactions:\n{previous_interactions}\n"
                f"Question: {question}\n"
                f"Info you remember:\n{json.dumps(info, indent=2)}\n"
                "Respond with your answer without any prefixes or suffixes."
            )
        else:
            user_prompt = (
                f"Previous interactions:\n{previous_interactions}\n"
                f"Question: {question}\n"
                "You don't remember any specific information.\n"
                "Respond with your answer without any prefixes or suffixes."
            )
        
        otherMessage(f"Updated emotion: {self.current_emotion}, level: {self.emotion_score + 1}")
        otherMessage(f"language pattern: {self.language_pattern}")
        otherMessage(f"Cognitive state: {self.cognitive_state}")
        
        response = self.client.get_response(messages=[{"role": "system", "content": system_prompt}, 
                                                     {"role": "user", "content": user_prompt}])
        return response.strip()

    def answer(self, question):
        """ Answer the question, dynamically adjusting emotional intensity and emotionality in the response. """
        if not question:
            return "Sorry, I didn’t catch that... Can you say it again?"
    
        self.questions.append(question)
        
        # Update emotion intensity
        self.emotion_score = self.get_new_emotion_score(question)
        
        # Dynamically determine the number of pieces of information to share
        max_info_items = 3
        selected_info_num = max(1, min(max_info_items, max_info_items - self.emotion_score))  # More emotion means less information shared
        
        # Select relevant information
        relevant_info = self.select_info(question, self.atom_information, selected_info_num)
        
        # Generate the answer with emotional influence
        answer = self.generate_emotional_answer(relevant_info, question)
        self.answers.append(answer)
        return answer
