import random
import json
from .patient_select import PatientSelect
from utils.myutils import otherMessage

class PatientLanguage(PatientSelect):
    def __init__(self, config, data):
        super().__init__(config, data)
        
        # Predefined lists for language patterns with descriptions
        self.language_patterns = [
            ('formal', 'Formal, polite, and structured language'),
            ('informal', 'Casual, friendly, and conversational language'),
            ('slang', 'Incorporates colloquial terms and slang expressions')
        ]
        
        random.seed(self.case_id)
        self.language_pattern, self.language_pattern_description = random.choice(self.language_patterns)
        random.seed()
        
        otherMessage(f"Language pattern: {self.language_pattern}")
        
        self.additional_info["language pattern"] = {"type": self.language_pattern, "description": self.language_pattern_description}

    def generate_answer(self, info, question):
        """ Generate an emotionally influenced answer based on the emotion intensity. """
        previous_interactions = " ".join([f"Q: {q}\nA: {a}" for q, a in zip(self.questions, self.answers)])

        system_prompt = (
            f"You speak in {self.language_pattern} language style ({self.language_pattern_description}). "
            "You will answer the doctor's question within 1-2 short sentences, "
            "with your language pattern affecting how you speak. "
            "Always refer to the patient in the first person, "
            "unless the patient is an infant, unconscious, or deceased, "
            "in which case refer to the patient as your family member. "
            "Only reveal the information that is directly asked for. "
            "For example, if the question generally asks about your symptoms, you should only answer with your chief complaint, excluding any further information like duration, location, or severity. "
            "NEVER make up any new information. "
        )
        
        user_prompt = (
            f"Previous interactions:\n{previous_interactions}\n"
            f"Question: {question}\n"
            f"Info you remember:\n{json.dumps(info, indent=2)}\n"
        )
        
        otherMessage(f"language pattern: {self.language_pattern}")
        
        response = self.client.get_response(messages=[{"role": "system", "content": system_prompt}, 
                                                     {"role": "user", "content": user_prompt}])
        return response.strip()

    def answer(self, question):
        """ Answer the question, dynamically adjusting emotional intensity and emotionality in the response. """
        if not question:
            return "Sorry, I didn’t catch that... Can you say it again?"
    
        self.questions.append(question)
           
        # Select relevant information
        relevant_info = self.select_info(question, self.atom_information, 3)
        
        # Generate the answer
        answer = self.generate_answer(relevant_info, question)
        self.answers.append(answer)
        return answer
