import json
from .base import Patient
from utils.myutils import otherMessage

class PatientSelect(Patient):
    """
    The patient class that selects relevant information based on the question and provides an answer.
    """
    def __init__(self, config, data):
        super().__init__(config, data)
        self.additional_info["selected_info"] = []

    def select_info(self, question, info_list, item_num=3):
        """
        Select relevant information from the info_list based on the question.
        Returns a subset of the info_list that is directly related to the question.
        """
        
        # Prepare the system prompt for information selection task
        system_prompt = (
            "You are tasked with selecting the most relevant pieces of information from a provided list. "
            "Select the most relevant items that directly answer the question. "
            "Each selected item must match *exactly* one entry from the list above. "
            "Do not add, rephrase, or infer any information. "
            f"Return up to {item_num} items, separated by semicolons. "
            "If nothing is relevant, return an empty string."
        )

        user_prompt = (
            f"Question: \"{question}\"\n\n"
            "List of information items:\n"
            + '; '.join(info_list)
        )

        # Call the model to get the relevant information subset
        messages = [
            {"role": "system", "content": system_prompt},
            {"role": "user", "content": user_prompt}
        ]

        info_set = set(item.strip().strip('.') for item in info_list)

        while True:
            response = self.client.get_response(messages).split(';')
            response_clean = [item.strip().strip('.') for item in response]

            valid_items = [item for item in response_clean if item in info_set]

            if not valid_items:
                otherMessage("No relevant information found.")
                return []

            if len(valid_items) > item_num:
                valid_items = valid_items[:item_num]
            
            otherMessage(f"Selected relevant information: {json.dumps(valid_items, indent=2)}")
            self.additional_info["selected_info"].append({"round": len(self.questions), "selected": valid_items})
            return valid_items


    def answer(self, question):
        """
        Provide an answer to the doctor's question based on the patient's context data,
        after selecting relevant information from the atomized information.
        """
        if not question:
            return "Sorry, I didn't recognize the question. Could you please rephrase it?"

        self.questions.append(question)

        # Select the relevant information based on the question
        relevant_info = self.select_info(question, self.atom_information)

        # Proceed with the original answering process using the selected information
        system_prompt = (
            "You are a patient in a hospital. You are required to answer the doctor's question based on the context paragraph. "
            "Always refer to the patient in the first person, "
            "unless the patient is an infant, unconscious, or deceased, "
            "in which case refer to the patient as your family member. "
            "Only reveal the information that is directly asked for. "
            "For example, if the question generally asks about your symptoms, you should only answer with your chief complaint, excluding any further information like duration, location, or severity. "
            "If there's no relevant information in the context paragraph, "
            "express your uncertainty instead of making assumptions. "
        )

        user_prompt = (
            f"Inquiry History: \n{"\n".join([f"Q: {q}\nA: {a}" for q, a in zip(self.questions, self.answers)])}\n\n"
            f"New question: \n\"{question}\"\n\n"
            "Your context paragraph:\n"
            f"{json.dumps(relevant_info, indent=2)}"
        )

        messages = [
            {"role": "system", "content": system_prompt},
            {"role": "user", "content": user_prompt}
        ]
        
        new_answer = self.client.get_response(messages).strip()
        
        self.answers.append(new_answer)
        return new_answer