from __future__ import print_function
import os, time

import torch
import torch.nn as nn
import torch.nn.parallel
from torch_geometric.loader import DataLoader
import torch.optim as optim
from torch.optim import lr_scheduler

from datasets.ZincFluor import get_ZincFluor

from models.GCN import *

from train.train_GNN import train_base
from train.validate_GNN import valid_base

from utils.config import *
from utils.common import hms_string

from utils.logger import logger
import copy
from losses.SupCon import *
from losses.loss import *

args = parse_args()
reproducibility(args.seed)
args.logger = logger(args)

best_acc = 0  # best test accuracy
many_best, med_best, few_best = 0, 0, 0
best_model = None


def train_stage1(args, model, trainloader, testloader, N_SAMPLES_PER_CLASS):
    global best_acc, many_best, med_best, few_best, best_model
    classwise_best_acc = [0,0,0]
    train_criterion = get_loss_by_name('bcl' if args.bcl else'ce_drw',trainloader.dataset.img_num_list,args)# nn.CrossEntropyLoss(label_smoothing=args.label_smooth)
    train_criterion_supcon = SupConLoss(class_num_list=trainloader.dataset.img_num_list) if args.SupCon else None
    test_criterion = nn.CrossEntropyLoss()  # For test, validation
    optimizer = optim.SGD(model.parameters(), lr=args.lr, momentum=args.momentum, weight_decay=args.wd, nesterov=args.nesterov)
    scheduler = lr_scheduler.CosineAnnealingLR(optimizer, args.epochs, eta_min = 0)

    best_model = None
    test_accs = []
    start_time = time.time()
    for epoch in range(args.epochs):

        train_loss, train_acc = train_base(trainloader, model, optimizer, train_criterion,train_criterion_supcon,epoch=epoch,args=args)
        lr = scheduler.get_last_lr()[0]
        scheduler.step()

        test_loss, test_acc, test_cls,classwise_acc = valid_base(testloader, model, test_criterion, N_SAMPLES_PER_CLASS,
                                                 num_class=args.num_class, mode='test Valid')

        if sum(classwise_best_acc[-6:]) <= sum(classwise_acc[-6:]):
            classwise_best_acc = classwise_acc
            best_acc = test_acc
            many_best = test_cls[0]
            med_best = test_cls[1]
            few_best = test_cls[2]

            best_model = copy.deepcopy(model)
        test_accs.append(test_acc)

        args.logger(f'Epoch: [{epoch + 1} | {args.epochs}]', level=1)

        args.logger(f'[Train]\tLoss:\t{train_loss:.4f}\tAcc:\t{train_acc:}', level=2)
        args.logger(f'[Test ]\tLoss:\t{test_loss:.4f}\tAcc:\t{test_acc:.4f}', level=2)
        args.logger(f'[Stats]\tMany:\t{test_cls[0]:.4f}\tMedium:\t{test_cls[1]:.4f}\tFew:\t{test_cls[2]:.4f}', level=2)
        args.logger(
            f'[Best ]\tAcc:\t{np.max(test_accs):.4f}\tMany:\t{100 * many_best:.4f}\tMedium:\t{100 * med_best:.4f}\tFew:\t{100 * few_best:.4f}',
            level=2)
        args.logger(f'[Param]\tLR:\t{lr:.8f}', level=2)
        args.logger(f'[Classwise_acc]\:\t{classwise_acc}', level=2)
        args.logger(f'[Best][Classwise_acc]\:\t{classwise_best_acc}', level=2)


    end_time = time.time()

    file_name = os.path.join(args.out, 'best_model_stage1.pth')
    torch.save(best_model, file_name)

    # Print the final results
    args.logger(f'Finish Training Stage 1...', level=1)
    args.logger(f'Final performance...', level=1)
    args.logger(f'best bAcc (test):\t{np.max(test_accs)}', level=2)
    args.logger(f'best statistics:\tMany:\t{many_best}\tMed:\t{med_best}\tFew:\t{few_best}', level=2)
    args.logger(f'Training Time: {hms_string(end_time - start_time)}', level=1)


def main():
    print(f'==> Preparing ZINC_FLUOR')
    args.epochs = 100
    
    trainset, testset = get_ZincFluor()
    N_SAMPLES_PER_CLASS = trainset.img_num_list
    args.cls_num_list = trainset.img_num_list
    print(N_SAMPLES_PER_CLASS)

    
    trainloader = DataLoader(trainset, batch_size=args.batch_size, shuffle=True, num_workers=args.workers,
                                  drop_last=False, pin_memory=True, sampler=None)
    testloader = DataLoader(testset, batch_size=args.batch_size, shuffle=True, num_workers=args.workers,
                                 pin_memory=True)
    args.num_class = trainset.num_classes
    
    # Model
    print("==> creating GCN")
    model = model_gcn(args,num_features=trainset.num_features,num_classes = trainset.num_classes).cuda()

    
    train_stage1(args, model, trainloader, testloader, N_SAMPLES_PER_CLASS)

if __name__ == '__main__':
    main()

#  CUDA_VISIBLE_DEVICES=2 python main_stage1_gnn.py --cur_stage stage1 --SupCon True   --label_smooth 0.2