import os
import json
import shutil
import re
from datetime import datetime

def parse_file_list(file_path):
    """Parses a file list, ignoring headers and empty lines."""
    with open(file_path, 'r') as f:
        lines = f.readlines()
    paths = [line.strip() for line in lines if line.strip().startswith('/')]
    return paths

def get_source_case_id(path):
    """Extracts the case ID like 'M75' or 'A06' from the path."""
    match = re.search(r'case-study/([MA]\d+)_', path)
    if match:
        return match.group(1)
    return "UNKNOWN"

def get_new_filename(original_path):
    """Creates a new, more descriptive filename."""
    source_case_id = get_source_case_id(original_path)
    original_filename = os.path.basename(original_path)
    
    # Sanitize original filename by removing special chars and extra extensions
    base, _ = os.path.splitext(original_filename)
    sanitized_base = re.sub(r'[^a-zA-Z0-9_-]', '_', base)
    
    return f"{source_case_id}_{sanitized_base}.md"

def main():
    """Main function to package the assets."""
    base_dir = "case-study/A07_Academic_Paper_Preparation_and_Asset_Collection"
    output_dir = os.path.join(base_dir, "paper_asset_package")
    
    # List of source file lists
    source_files = [
        os.path.join(base_dir, "A07_record_01_asset_file_collection_list.md"),
        os.path.join(base_dir, "A07_record_02_supplemental_asset_list.md")
    ]

    # --- 1. Setup Directories ---
    if os.path.exists(output_dir):
        shutil.rmtree(output_dir)
    os.makedirs(output_dir)
    print(f"Created output directory: {output_dir}")

    # --- 2. Collect all file paths ---
    all_paths = []
    for file_list_path in source_files:
        if os.path.exists(file_list_path):
            all_paths.extend(parse_file_list(file_list_path))
        else:
            print(f"[WARNING] Source file not found: {file_list_path}")

    # Remove duplicates
    all_paths = sorted(list(set(all_paths)))
    
    # --- 3. Process and copy files in blocks ---
    manifest_data = {
        "package_info": {},
        "files": []
    }
    
    file_counter = 0
    block_counter = 1
    block_dir = os.path.join(output_dir, f"block_{block_counter}")
    os.makedirs(block_dir)

    for original_path in all_paths:
        if not os.path.exists(original_path):
            print(f"[ERROR] Original file not found, skipping: {original_path}")
            continue

        if file_counter >= 20:
            file_counter = 0
            block_counter += 1
            block_dir = os.path.join(output_dir, f"block_{block_counter}")
            os.makedirs(block_dir)

        new_filename = get_new_filename(original_path)
        new_path_relative = os.path.join(f"block_{block_counter}", new_filename)
        new_path_absolute = os.path.join(output_dir, new_path_relative)
        
        shutil.copy(original_path, new_path_absolute)
        
        file_info = {
            "new_path": new_path_relative,
            "original_path": original_path,
            "source_case_id": get_source_case_id(original_path),
            "type": "case_summary" # Default type, can be refined
        }
        manifest_data["files"].append(file_info)
        
        file_counter += 1

    print(f"Processed and copied {len(all_paths)} files into {block_counter} blocks.")

    # --- 4. Create Manifest ---
    manifest_data["package_info"] = {
        "creation_date": datetime.now().isoformat(),
        "total_files": len(all_paths),
        "total_blocks": block_counter
    }
    
    manifest_path = os.path.join(output_dir, "manifest.json")
    with open(manifest_path, 'w') as f:
        json.dump(manifest_data, f, indent=2)
    print(f"Created manifest file: {manifest_path}")

if __name__ == "__main__":
    main()
