import re
from pathlib import Path

# --- Configuration ---
# The exact path to the problematic log file
LOG_FILE_PATH = Path('.ai-cli-log/gemini-20250715-175313.txt')

# The regular expression currently in use in 01_extract_data.py
METADATA_BLOCK_RE = re.compile(r"""---+\s*\n\s*\*?\s*METADATA LOG\s*\*?\s*(.*?)\n\s*---+""", re.DOTALL | re.IGNORECASE)

# --- Main Debugging Logic ---
def main():
    """
    Reads a specific log file and systematically tests the regex against its content.
    """
    print(f"--- Starting Regex Debug on: {LOG_FILE_PATH.name} ---")

    if not LOG_FILE_PATH.is_file():
        print(f"[ERROR] Log file not found at: {LOG_FILE_PATH}")
        return

    try:
        content = LOG_FILE_PATH.read_text(encoding='utf-8')
    except Exception as e:
        print(f"[ERROR] Could not read log file: {e}")
        return

    matches = METADATA_BLOCK_RE.finditer(content)

    found_blocks = []
    for match in matches:
        # Extract the core content of the block
        block_content = match.group(1).strip()
        found_blocks.append(block_content)

    print(f"\n--- Regex Analysis Complete ---")
    print(f"Found {len(found_blocks)} METADATA LOG blocks with the current regex.")
    
    if not found_blocks:
        print("\n[!] No blocks were found. The regex is failing to match anything in this file.")
    else:
        print("\n--- Captured Blocks: ---")
        for i, block in enumerate(found_blocks, 1):
            print(f"--- Block #{i} ---")
            print(block)
            print("-" * (len(f"--- Block #{i} ---")))


if __name__ == "__main__":
    main()
