#!/bin/bash
#
# M76 Quantitative Analysis Pipeline - Master Execution Script
#
# This script runs the full, end-to-end data analysis pipeline for the M76
# case study. It should be run from the root of the chac-workbench repository.

# --- Configuration ---
set -e # Exit immediately if a command exits with a non-zero status.
BASE_DIR="case-study/M76_Quantitative_Study_of_the_CHAC_Workbench"
LOG_DIR="$BASE_DIR/output_verbose_log"
LOG_FILE="$LOG_DIR/M76_verbose_analysis_run.log"
VENV_PYTHON="venv/bin/python3"

# --- Command Definition ---
# A list of commands to be executed in sequence.
# This makes the script more readable and easier to debug.
COMMANDS=(
    "bash $BASE_DIR/00_batch_transform.sh"
    "bash $BASE_DIR/01_generate_robust_map.sh"
    "$VENV_PYTHON $BASE_DIR/02_create_final_map.py"
    "$VENV_PYTHON $BASE_DIR/03_extract_metrics.py"
    "$VENV_PYTHON $BASE_DIR/04_verify_metrics.py"
    "$VENV_PYTHON $BASE_DIR/05_generate_reports.py"
    "$VENV_PYTHON $BASE_DIR/06_static_token_analysis.py"
    "$VENV_PYTHON $BASE_DIR/07_growth_analysis.py"
    "$VENV_PYTHON $BASE_DIR/08_generate_visualizations.py"
)

# --- Execution ---
# 1. Ensure the log directory exists.
mkdir -p "$LOG_DIR"

# 2. Clear the previous log and write a header.
echo "---" > "$LOG_FILE"
echo "Starting M76 Analysis Pipeline at $(date)" >> "$LOG_FILE"
echo "---" >> "$LOG_FILE"

# 3. Loop through and execute each command.
for cmd in "${COMMANDS[@]}"; do
    echo "" | tee -a "$LOG_FILE"
    echo "--- EXECUTING: $cmd ---" | tee -a "$LOG_FILE"
    
    # Execute the command and tee output to log.
    # Errors will cause the script to exit due to 'set -e'.
    (eval "$cmd") 2>&1 | tee -a "$LOG_FILE"
done

# --- Completion ---
echo "" >> "$LOG_FILE"
echo "---" | tee -a "$LOG_FILE"
echo "Analysis pipeline finished successfully at $(date)." | tee -a "$LOG_FILE"
echo "Verbose log saved to: $LOG_FILE" | tee -a "$LOG_FILE"
echo "Final reports are in: $BASE_DIR/output/" | tee -a "$LOG_FILE"
