#!/bin/bash
set -e

# ==============================================================================
# CHAC System Prompt Builder v4.0 (Guardian Edition)
#
# PURPOSE:
# This script builds a standalone, portable system prompt for a CHAC AI.
# It intelligently assembles the prompt from various source files based on a
# user-provided configuration, ensuring that the final product is complete,
# up-to-date, and user-centric.
#
# USAGE:
# ./CHAC_System_Prompt_Builder.sh /path/to/your/build_config.json
#
# WORKFLOW:
# 1. COPY the template file `chac_toolkit/build_config.template.json` to your
#    own project or experiment directory (e.g., `experiments/my_exp/build.json`).
# 2. MODIFY your copied configuration file to define the desired structure
#    and output path for your prompt.
# 3. RUN this script, providing the path to your custom configuration file
#    as the only argument.
#
# DEPENDENCIES:
# - jq: This script requires the 'jq' command-line JSON processor to safely
#       parse configuration files.
#
# GUARDIAN CHECKS:
# This script is designed with "Guardian" principles and will actively abort
# the build process if critical conditions are not met, such as:
# - The root `config.json` is missing.
# - The user's `chac_profile.md` for the active experiment is missing or empty.
# - The user attempts to use the configuration template directly.
# ==============================================================================

# --- PRE-FLIGHT CHECKS ---

# 0. Check for the existence of the user's config.json first.
if [ ! -f "config.json" ]; then
    echo "❌ FATAL ERROR: The root 'config.json' file does not exist."
    echo "This file is required to determine the active experiment and user profile."
    echo "Please create it by copying 'config.template.json' and setting your 'active_experiment'."
    exit 1
fi

# 1. Check for jq dependency
if ! command -v jq &> /dev/null; then
    echo "❌ ERROR: 'jq' is not installed. This script requires jq to parse configuration files."
    echo "Please install jq to proceed."
    exit 1
fi

# 2. Check for config file argument
CONFIG_FILE="$1"
if [ -z "$CONFIG_FILE" ]; then
    echo "❌ ERROR: You must provide a path to your build configuration file."
    echo "Usage: $0 /path/to/your/build_config.json"
    exit 1
fi

if [ ! -f "$CONFIG_FILE" ]; then
    echo "❌ ERROR: Config file not found at '$CONFIG_FILE'"
    exit 1
fi

# 3. Prevent direct use of the template
if [[ "$CONFIG_FILE" == *"build_config.template.json" ]]; then
    echo "❌ ERROR: Do not use the template file directly."
    echo "Please COPY 'chac_toolkit/build_config.template.json' to a new location"
    echo "(e.g., your_experiment/my_build.json), modify it, and then run the script with your new file."
    exit 1
fi

# --- UTILITY FUNCTION ---
append_file() {
    local file_path="$1"
    local output_file="$2"
    if [ -f "$file_path" ]; then
        echo "  -> Appending $file_path"
        echo -e "\n--- START OF FILE: $file_path ---\n" >> "$output_file"
        cat "$file_path" >> "$output_file"
    else
        echo "  -> WARNING: File not found, skipping: $file_path"
    fi
}

# --- MAIN SCRIPT ---
echo "INFO: Reading user's build config from: $CONFIG_FILE"

# Read configuration values using jq
BUILD_TARGET=$(jq -r '.build_target' "$CONFIG_FILE")
OUTPUT_PATH=$(jq -r '.output_path' "$CONFIG_FILE")
MANIFEST_FILE=$(jq -r '.build_stages.manifest_file' "$CONFIG_FILE")

# --- BUILD PROCESS ---
mkdir -p "$(dirname "$OUTPUT_PATH")"
rm -f "$OUTPUT_PATH"
touch "$OUTPUT_PATH"
echo "INFO: Build started for target: '$BUILD_TARGET'. Output will be at: '$OUTPUT_PATH'"

# Stage 1: Prepend files
echo "INFO: Stage 1: Prepending files..."
jq -r '.build_stages.prepend_files[]' "$CONFIG_FILE" | while IFS= read -r file; do
    append_file "$file" "$OUTPUT_PATH"
done

# Stage 2: Process Manifest file
echo "INFO: Stage 2: Processing manifest file: $MANIFEST_FILE..."
if [ -n "$MANIFEST_FILE" ] && [ -f "$MANIFEST_FILE" ]; then
    # New, robust parsing logic:
    # 1. `grep '^- '`: Find only lines that start with "- " (a list item).
    # 2. `awk '{print $2}'`: Extract the second field (the file path).
    while IFS= read -r file_path; do
        append_file "$file_path" "$OUTPUT_PATH"
    done < <(grep '^- ' "$MANIFEST_FILE" | awk '{print $2}')
else
    echo "  -> WARNING: Manifest file not found or not specified, skipping."
fi

# Stage 3: Append files from config
echo "INFO: Stage 3: Appending files from config..."
jq -r '.build_stages.append_files[]' "$CONFIG_FILE" | while IFS= read -r file; do
    append_file "$file" "$OUTPUT_PATH"
done

# Stage 4: Append DYNAMIC user profile (Guardian Check)
echo "INFO: Stage 4: Appending dynamic user profile..."
ACTIVE_EXP=$(jq -r '.active_experiment' "config.json")
PROFILE_PATH="experiments/$ACTIVE_EXP/chac_profile.md"
echo "  -> Checking for profile at: $PROFILE_PATH"

if [ -f "$PROFILE_PATH" ] && [ -s "$PROFILE_PATH" ]; then
    append_file "$PROFILE_PATH" "$OUTPUT_PATH"
else
    echo ""
    echo "❌ FATAL ERROR: User profile not found or is empty at '$PROFILE_PATH'."
    echo "A valid user profile is non-negotiable for a CHAC system prompt."
    echo "To resolve this, please create the profile, for example by following the"
    echo "'Collaborative Profile Generation Protocol'."
    echo ""
    echo "Build process aborted."
    # Clean up the incomplete artifact before exiting
    rm -f "$OUTPUT_PATH"
    exit 1
fi

echo "✅ Build finished successfully. Prompt saved to '$OUTPUT_PATH'"