#!/bin/bash

# CHAC Portable & Self-Initializing ID Generator v3.1
# Atomically generates the next sequential ID for a GIVEN project directory.
# It self-initializes by scanning for the highest existing ID if no state file is found.

# Read the case study root directory from the global config file.
# This makes the script a zero-argument, configuration-driven tool.
CONFIG_FILE="config.json"
if [ ! -f "$CONFIG_FILE" ]; then
  echo "❌ Error: Configuration file '$CONFIG_FILE' not found." >&2
  exit 1
fi
TARGET_DIR=$(jq -r '.toolkit_paths.case_study_root' "$CONFIG_FILE")

# --- Safety Checks ---
if [ -z "$TARGET_DIR" ] || [ "$TARGET_DIR" == "null" ]; then
  echo "❌ Error: '.toolkit_paths.case_study_root' not found or is null in '$CONFIG_FILE'." >&2
  exit 1
fi

if [ ! -d "$TARGET_DIR" ]; then
  echo "❌ Error: Target directory '$TARGET_DIR' defined in config does not exist." >&2
  exit 1
fi

# --- State Management ---
DATE=$(date +%Y%m%d)
STATE_DIR="$TARGET_DIR/.chac_state"
mkdir -p "$STATE_DIR"
SEQ_FILE="$STATE_DIR/$DATE.seq"

# --- Atomic Operation ---
(
  flock 200

  # --- Initialization Logic (The Truly Final Core) ---
  # If the sequence file for today doesn't exist, we must initialize it
  # by finding the highest existing ID in the entire project.
  if [ ! -f "$SEQ_FILE" ]; then
    # Find the highest sequence number from ALL existing CHAC-SD files,
    # regardless of date, by looking at the final number in the filename.
    # The sed expression is robust enough to handle various naming conventions.
    HIGHEST_EXISTING_SEQ=$(find "$TARGET_DIR" -type f -name "*CHAC-SD-*.md" 2>/dev/null | \
                          sed -n 's/.*CHAC-SD-[0-9]\{8\}-\([0-9]\{2,\}\)\.md/\1/p' | \
                          sort -rn | \
                          head -n 1)

    # If no files were found, start at 0. Otherwise, start from the highest found.
    if [ -z "$HIGHEST_EXISTING_SEQ" ]; then
      echo 0 > "$SEQ_FILE"
    else
      echo "$HIGHEST_EXISTING_SEQ" > "$SEQ_FILE"
    fi
  fi

  # Read the current sequence number, increment it, and write it back.
  CURRENT_SEQ=$(cat "$SEQ_FILE" | tr -d '[:space:]')
  # If the file was just created and is empty, default to 0.
  : "${CURRENT_SEQ:=0}"
  NEXT_SEQ=$((CURRENT_SEQ + 1))
  echo $NEXT_SEQ > "$SEQ_FILE"

  # Output the new, unique ID in the standard format.
  printf "CHAC-SD-%s-%02d\n" "$DATE" "$NEXT_SEQ"

) 200>"$STATE_DIR/.id.lock"