#!/bin/bash

# chac_conclude_case.sh v3.0
# Concludes a CHAC case study by logging an end marker.
# This version reads from a local .chac_state file instead of parsing the global log.

# Find the script's own directory and source the template
SCRIPT_DIR=$(dirname "$0")
source "$SCRIPT_DIR/script_template.sh"

# Load config and check dependencies
load_config

# --- MAIN LOGIC ---

if [ "$#" -ne 1 ]; then
    log_error "Usage: $0 <case_study_directory>"
fi

CASE_DIR=$1
STATE_FILE="${CASE_DIR}/.chac_state"

if [ ! -f "$STATE_FILE" ]; then
    log_error "State file not found: '$STATE_FILE'. Cannot conclude case."
fi

# Extract info from the JSON state file.
# Requires 'jq' to be installed. We should add this to our dependency check.
CASE_NAME=$(jq -r '.case_name' "$STATE_FILE")
SOLIDIFICATION_ID=$(jq -r '.solidification_id' "$STATE_FILE")

if [ -z "$CASE_NAME" ] || [ -z "$SOLIDIFICATION_ID" ]; then
    log_error "Failed to parse case_name or solidification_id from '$STATE_FILE'."
fi

# Generate a NEW UUID for the end marker, as per protocol.
UUID=$(uuidgen)
if [ -z "$UUID" ]; then
    log_error "Failed to generate UUID. Is 'uuidgen' installed?"
fi

# Get ISO 8601 timestamp in a truly cross-platform way by testing the feature.
# First, attempt the GNU version and suppress stderr.
ISO_TIMESTAMP=$(date -u --iso-8601=seconds 2>/dev/null)
# Check the exit code. If it failed, use the BSD/macOS compatible version.
if [ $? -ne 0 ]; then
    ISO_TIMESTAMP=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
fi
FULL_LOG_PATH="${CASE_STUDY_ROOT}/${GLOBAL_LOG_FILE}"

# New log format for the end marker
END_MARKER="[${ISO_TIMESTAMP}] CASE_STUDY_END_MARKER: [${CASE_NAME}] [${SOLIDIFICATION_ID}] [${UUID}]"

echo "$END_MARKER" >> "$FULL_LOG_PATH"

# Also add the end marker UUID to the local state file
TEMP_STATE_FILE=$(mktemp)
jq --arg uuid "$UUID" '. + {end_marker_uuid: $uuid}' "$STATE_FILE" > "$TEMP_STATE_FILE" && mv "$TEMP_STATE_FILE" "$STATE_FILE"
log_info "Successfully added end_marker_uuid to local state file."

log_info "Generated Marker: ${END_MARKER}"
log_info "Successfully concluded case study ${CASE_NAME} (${SOLIDIFICATION_ID})."
log_info "Logged end marker to global log."
