#!/bin/bash

# chac_generate_report.sh v2.0
# Generates a report DRAFT based on a template and a case's state file.
# The draft contains inline instructions for the AI to complete.

# Find the script's own directory and source the template
SCRIPT_DIR=$(dirname "$0")
source "$SCRIPT_DIR/script_template.sh"

# Load config and check dependencies
load_config
check_dependency jq

# --- MAIN LOGIC ---

if [ "$#" -ne 1 ]; then
    log_error "Usage: $0 <case_study_directory>"
    exit 1
fi

CASE_DIR=$1
STATE_FILE="${CASE_DIR}/.chac_state"
TEMPLATE_FILE="${SCRIPT_DIR}/report_template.md"
REPORT_DIR="${CASE_DIR}/report"

# --- Validate inputs ---
if [ ! -d "$CASE_DIR" ]; then
    log_error "Case directory not found: '$CASE_DIR'"
    exit 1
fi
if [ ! -f "$STATE_FILE" ]; then
    log_error "State file not found: '$STATE_FILE'"
    exit 1
fi
if [ ! -f "$TEMPLATE_FILE" ]; then
    log_error "Report template not found: '$TEMPLATE_FILE'"
    exit 1
fi

# --- Read data from state file ---
SOLIDIFICATION_ID=$(jq -r '.solidification_id' "$STATE_FILE")
CASE_ID=$(jq -r '.case_id' "$STATE_FILE")
CASE_NAME=$(jq -r '.case_name' "$STATE_FILE")
START_MARKER=$(jq -r '.start_marker_uuid // "N/A"' "$STATE_FILE")
END_MARKER=$(jq -r '.end_marker_uuid // "N/A"' "$STATE_FILE")

# --- Create report directory if it doesn't exist ---
mkdir -p "$REPORT_DIR"

# --- Assemble the report DRAFT ---
OUTPUT_FILE="${REPORT_DIR}/${SOLIDIFICATION_ID}_report_draft.md"

# Use sed to replace placeholders in the YAML front matter.
# The body of the template, with its AI instructions, is left untouched.
TEMP_CONTENT=$(<"$TEMPLATE_FILE")
TEMP_CONTENT="${TEMP_CONTENT//\{\{SOLIDIFICATION_ID\}\}/$SOLIDIFICATION_ID}"
TEMP_CONTENT="${TEMP_CONTENT//\{\{CASE_ID\}\}/$CASE_ID}"
TEMP_CONTENT="${TEMP_CONTENT//\{\{CASE_NAME\}\}/$CASE_NAME}"
TEMP_CONTENT="${TEMP_CONTENT//\{\{START_MARKER\}\}/$START_MARKER}"
TEMP_CONTENT="${TEMP_CONTENT//\{\{END_MARKER\}\}/$END_MARKER}"

echo "$TEMP_CONTENT" > "$OUTPUT_FILE"

log_info "Successfully generated report draft: $OUTPUT_FILE"
log_info "Next step: AI must read this draft, follow the inline instructions, and overwrite it with the final content."