#!/bin/bash

# chac_init_case.sh v3.0
# Initializes a new CHAC case study with a proper Case Number.

# Find the script's own directory and source the template
SCRIPT_DIR=$(dirname "$0")
source "$SCRIPT_DIR/script_template.sh"

# Load config and check dependencies
load_config

# --- MAIN LOGIC ---

if [ "$#" -ne 2 ]; then
    log_error "Usage: $0 <CaseNumber> \"<Case Study Title>\""
    exit 1
fi

CASE_NUMBER=$1
CASE_TITLE=$2
TIMESTAMP=$(date +%Y%m%d)
# A simple way to get a short unique ID without external tools
SHORT_ID=$(date +%s | sha256sum | base64 | head -c 8)

SOLIDIFICATION_ID="CHAC-SD-${SHORT_ID}-${TIMESTAMP}"
# Replace spaces with underscores for dir name, handle potential slashes
SAFE_TITLE=$(echo "$CASE_TITLE" | tr ' ' '_' | tr -d '/')
DIR_NAME="${CASE_STUDY_ROOT}/${CASE_NUMBER}_${SAFE_TITLE}"

if [ -d "$DIR_NAME" ]; then
    log_error "Directory '$DIR_NAME' already exists."
    exit 1
fi

mkdir -p "$DIR_NAME" || log_error "Failed to create directory '$DIR_NAME'."
log_info "Successfully created directory: $DIR_NAME"

FULL_LOG_PATH="${CASE_STUDY_ROOT}/${GLOBAL_LOG_FILE}"
# Get ISO 8601 timestamp in a truly cross-platform way by testing the feature.
# First, attempt the GNU version and suppress stderr.
ISO_TIMESTAMP=$(date -u --iso-8601=seconds 2>/dev/null)
# Check the exit code. If it failed, use the BSD/macOS compatible version.
if [ $? -ne 0 ]; then
    ISO_TIMESTAMP=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
fi
UUID=$(uuidgen)
DIR_BASENAME=$(basename "$DIR_NAME")

# New log format: [timestamp] [MARKER_TYPE]: [case_name] [solidification_id] [uuid]
START_MARKER="[${ISO_TIMESTAMP}] CASE_STUDY_START_MARKER: [${DIR_BASENAME}] [${SOLIDIFICATION_ID}] [${UUID}]"

echo "$START_MARKER" >> "$FULL_LOG_PATH"
log_info "Logged start marker to global log: $FULL_LOG_PATH"

# --- Create local state file ---
STATE_FILE_PATH="${DIR_NAME}/.chac_state"
# Create initial JSON and then add the UUID with jq to ensure format is always valid
INITIAL_JSON=$(cat << EOL
{
  "case_id": "${CASE_NUMBER}",
  "case_name": "${DIR_BASENAME}",
  "solidification_id": "${SOLIDIFICATION_ID}",
  "latest_record_index": 0
}
EOL
)

echo "$INITIAL_JSON" | jq --arg uuid "$UUID" '. + {start_marker_uuid: $uuid}' > "$STATE_FILE_PATH"
log_info "Successfully created local state file: ${STATE_FILE_PATH}"
# -----------------------------

# --- NEW: Update global ID state ---
GLOBAL_STATE_FILE="$PROJECT_ROOT/$CASE_STUDY_ROOT/.chac_state/id_state.json"
if [ -f "$GLOBAL_STATE_FILE" ]; then
    PREFIX=$(echo "$CASE_NUMBER" | cut -c1)
    NUMBER=$(echo "$CASE_NUMBER" | cut -c2-)
    
    JSON_KEY=""
    if [ "$PREFIX" == "M" ]; then
        JSON_KEY="last_m_id"
    else
        JSON_KEY="last_a_id"
    fi

    TMP_STATE_FILE="${GLOBAL_STATE_FILE}.tmp"
    jq ".${JSON_KEY} = ${NUMBER}" "$GLOBAL_STATE_FILE" > "$TMP_STATE_FILE" &&     mv "$TMP_STATE_FILE" "$GLOBAL_STATE_FILE"
    
    log_info "Successfully updated global state file: $GLOBAL_STATE_FILE"
else
    log_warn "Global state file not found, skipping update. This may be expected during initial setup."
fi
# ---------------------------------

echo "---"
echo "Case Study Initialized."
echo "ID: ${SOLIDIFICATION_ID}"
echo "Directory: ${DIR_NAME}"
echo "---"
