#!/bin/bash

# chac_new_record.sh v4.0 (State-Aware)
# Creates a new, correctly prefixed atomic record within a CHAC case study.

# Find the script's own directory and source the template
SCRIPT_DIR=$(dirname "$0")
source "$SCRIPT_DIR/script_template.sh"

# Load config and check dependencies
load_config

# --- MAIN LOGIC ---

if [ "$#" -ne 2 ]; then
    log_error "Usage: $0 <case_study_directory> \"<Record Title>\""
    exit 1
fi

CASE_DIR=$1
RECORD_TITLE=$2

if [ ! -d "$CASE_DIR" ]; then
    log_error "Case study directory not found: '$CASE_DIR'"
    exit 1
fi

STATE_FILE="${CASE_DIR}/.chac_state"
if [ ! -f "$STATE_FILE" ]; then
    log_error "State file (.chac_state) not found in '$CASE_DIR'"
    exit 1
fi

# --- STATE-AWARE LOGIC ---
# Read variables directly from the JSON state file
# Using 'jq' is a robust way to parse JSON in bash
CASE_ID=$(jq -r '.case_id' "$STATE_FILE")
LATEST_INDEX=$(jq -r '.latest_record_index' "$STATE_FILE")

if [ -z "$CASE_ID" ] || [ -z "$LATEST_INDEX" ]; then
    log_error "Could not read case_id or latest_record_index from state file."
    exit 1
fi

NEXT_INDEX=$((LATEST_INDEX + 1))
# --- END OF STATE-AWARE LOGIC ---

FORMATTED_NUM=$(printf "%02d" "$NEXT_INDEX")
SNAKE_CASE_TITLE=$(echo "$RECORD_TITLE" | tr '[:upper:]' '[:lower:]' | tr ' ' '_')
FILE_NAME="${CASE_DIR}/${CASE_ID}_record_${FORMATTED_NUM}_${SNAKE_CASE_TITLE}.md"

TEMPLATE_FILE="${TEMPLATES_DIR}/atomic_record.template.md"
if [ ! -f "$TEMPLATE_FILE" ]; then
    log_error "Template file not found: '$TEMPLATE_FILE'"
fi

# Use the case_name from the state file for the parent case ID
PARENT_CASE_ID=$(jq -r '.case_name' "$STATE_FILE")

sed -e "s/{{RECORD_TITLE}}/$RECORD_TITLE/g" \
    -e "s/{{PARENT_CASE_ID}}/$PARENT_CASE_ID/g" \
    -e "s/{{TOPIC_PROVIDED_BY_ARCHITECT}}/$RECORD_TITLE/g" \
    -e "s/{{CONTENT}}//g" \
    "$TEMPLATE_FILE" > "$FILE_NAME" || log_error "Failed to create record file '$FILE_NAME'."

# --- UPDATE STATE ---
# Use jq to update the latest_record_index in place
jq ".latest_record_index = ${NEXT_INDEX}" "$STATE_FILE" > "${STATE_FILE}.tmp" && mv "${STATE_FILE}.tmp" "$STATE_FILE"
# --------------------

log_info "Successfully created new record: $FILE_NAME"