#!/bin/bash

# chac_next_case_id.sh v2.0
# Determines the next available case study ID from a state file.

# Find the script's own directory and source the template
SCRIPT_DIR=$(dirname "$0")
source "$SCRIPT_DIR/script_template.sh"

# Load config and check dependencies
load_config

# --- DEPENDENCY CHECK ---
if ! command -v jq &> /dev/null;
then
    log_error "'jq' is not installed. Please install it to continue."
    exit 1
fi

# --- MAIN LOGIC ---

if [ "$#" -ne 1 ]; then
    log_error "Usage: $0 <M|A>"
    exit 1
fi

CLASS_PREFIX=$1
STATE_DIR="$PROJECT_ROOT/$CASE_STUDY_ROOT/.chac_state"
STATE_FILE="$STATE_DIR/id_state.json"

if [[ "$CLASS_PREFIX" != "M" && "$CLASS_PREFIX" != "A" ]]; then
    log_error "Invalid class specified. Must be 'M' or 'A'."
    exit 1
fi

# Ensure the state directory exists
mkdir -p "$STATE_DIR"

# Initialize state file if it doesn't exist (handles empty project case)
if [ ! -f "$STATE_FILE" ]; then
    log_info "State file not found. Initializing a new one."
    echo '{"last_m_id": 0, "last_a_id": 0}' > "$STATE_FILE"
fi

# Validate JSON and get the key based on prefix
JSON_KEY=""
if [ "$CLASS_PREFIX" == "M" ]; then
    JSON_KEY="last_m_id"
else
    JSON_KEY="last_a_id"
fi

LAST_ID=$(jq -e ".${JSON_KEY}" "$STATE_FILE")

if [ $? -ne 0 ]; then
    log_error "State file '$STATE_FILE' is corrupt or missing key '.${JSON_KEY}'."
    exit 1
fi

# Calculate the next number
NEXT_ID=$((LAST_ID + 1))

# Output the final, formatted Case ID
printf "%s%s\n" "$CLASS_PREFIX" "$NEXT_ID"
