#!/bin/bash

# CHAC Script Template v2.0
# This file contains shared functions and variables for all CHAC toolkit scripts.

# --- SHARED VARIABLES ---
# Color codes for logging
COLOR_GREEN='\033[0;32m'
COLOR_RED='\033[0;31m'
COLOR_YELLOW='\033[0;33m'
COLOR_NC='\033[0m' # No Color

# --- SHARED FUNCTIONS ---

# log_info <message>
function log_info() {
    echo -e "${COLOR_GREEN}[INFO]${COLOR_NC} $1"
}

# log_warn <message>
function log_warn() {
    echo -e "${COLOR_YELLOW}[WARN]${COLOR_NC} $1" >&2
}

# log_error <message>
function log_error() {
    echo -e "${COLOR_RED}[ERROR]${COLOR_NC} $1" >&2
    exit 1
}

# check_dependency <command>
function check_dependency() {
    if ! command -v "$1" &> /dev/null; then
        log_error "Dependency not found: '$1'. Please install it and ensure it's in your PATH."
    fi
}

# load_config
# Loads all toolkit paths from the root config.json file.
function load_config() {
    # This assumes the script is run from the project root, or the script can find the root.
    # A robust way is to find the .git directory.
    PROJECT_ROOT=$(git rev-parse --show-toplevel)
    if [ -z "$PROJECT_ROOT" ]; then
        log_error "Could not determine project root. Are you in a git repository?"
    fi

    if [ -n "$CHAC_CONFIG_PATH" ]; then
        CONFIG_FILE="$CHAC_CONFIG_PATH"
    else
        CONFIG_FILE="$PROJECT_ROOT/config.json"
    fi
    if [ ! -f "$CONFIG_FILE" ]; then
        log_error "Config file not found: '$CONFIG_FILE'"
    fi

    # Check for jq dependency
    check_dependency jq

    # Read paths from config file
    CASE_STUDY_ROOT=$(jq -r '.toolkit_paths.case_study_root' "$CONFIG_FILE")
    GLOBAL_LOG_FILE=$(jq -r '.toolkit_paths.global_log_file' "$CONFIG_FILE")
    TEMPLATES_DIR=$(jq -r '.toolkit_paths.templates_dir' "$CONFIG_FILE")
    SCRIPTS_DIR=$(jq -r '.toolkit_paths.scripts_dir' "$CONFIG_FILE")

    # --- NEW: VALIDATION STEP ---
    if [ -z "$CASE_STUDY_ROOT" ] || [ "$CASE_STUDY_ROOT" == "null" ]; then
        log_error "Config error: '.toolkit_paths.case_study_root' is not defined or null in '$CONFIG_FILE'."
    fi

    # Note: We check the absolute path for existence.
    if [ ! -d "$PROJECT_ROOT/$CASE_STUDY_ROOT" ]; then
        log_error "Config error: The specified case study root does not exist: '$PROJECT_ROOT/$CASE_STUDY_ROOT'"
    fi

    # Export them to be available in the calling script
    export PROJECT_ROOT
    export CASE_STUDY_ROOT
    export GLOBAL_LOG_FILE
    export TEMPLATES_DIR
    export SCRIPTS_DIR
}