#!/bin/bash

# CHAC Toolkit Test Suite v1.0

# Source the template to get the logging functions
source ./chac_toolkit/scripts/script_template.sh

# --- TEST SETUP ---
log_info "Setting up test environment..."
TEST_DIR="temp_test_suite"
rm -rf "$TEST_DIR"
mkdir -p "$TEST_DIR"
# We need the config file to be available for the scripts
cp config.json "$TEST_DIR/"
# The scripts will be called from the project root, but will operate on the test dir
# We need to adjust the config to point to our test directory
# Using jq to update the config in the test directory
jq '.toolkit_paths.case_study_root = "'$TEST_DIR'/case-study"' "$TEST_DIR/config.json" > "$TEST_DIR/config.tmp" && mv "$TEST_DIR/config.tmp" "$TEST_DIR/config.json"
mkdir -p "$TEST_DIR/case-study"

# --- HELPER FUNCTIONS ---
assert_success() {
    if [ $? -ne 0 ]; then
        log_error "TEST FAILED: '$1'"
    else
        log_info "TEST PASSED: '$1'"
    fi
}

assert_fail() {
    if [ $? -eq 0 ]; then
        log_error "TEST FAILED: '$1' was expected to fail but succeeded."
    else
        log_info "TEST PASSED: '$1' correctly failed as expected."
    fi
}

assert_file_exists() {
    if [ ! -f "$1" ]; then
        log_error "ASSERTION FAILED: File '$1' does not exist."
    else
        log_info "ASSERTION PASSED: File '$1' exists."
    fi
}

assert_dir_exists() {
    if [ ! -d "$1" ]; then
        log_error "ASSERTION FAILED: Directory '$1' does not exist."
    else
        log_info "ASSERTION PASSED: Directory '$1' exists."
    fi
}

assert_file_contains() {
    if ! grep -Fq "$2" "$1"; then
        log_error "ASSERTION FAILED: File '$1' does not contain '$2'."
    else
        log_info "ASSERTION PASSED: File '$1' contains '$2'."
    fi
}


# --- TEST EXECUTION ---
log_info "Starting CHAC Toolkit Test Suite..."

# Test 1: Initialize a new case study
log_info "--- Testing chac_init_case.sh ---"
export CHAC_CONFIG_PATH="$TEST_DIR/config.json"
bash chac_toolkit/scripts/chac_init_case.sh "A04_TestSuite_Test"
assert_success "chac_init_case.sh execution"

# Construct the expected directory name
SAFE_TITLE=$(echo "A04_TestSuite_Test" | tr ' ' '_' | tr -d '/')
TEST_CASE_DIR="$TEST_DIR/case-study/$(date +%Y%m%d)_${SAFE_TITLE}"
assert_dir_exists "$TEST_CASE_DIR"
GLOBAL_LOG_PATH="$TEST_DIR/case-study/_case_study_log.md"
assert_file_exists "$GLOBAL_LOG_PATH"
assert_file_contains "$GLOBAL_LOG_PATH" "CASE_STUDY_START_MARKER:"
assert_file_contains "$GLOBAL_LOG_PATH" "A04_TestSuite_Test"

# Test 2: Create a new record
log_info "--- Testing chac_new_record.sh ---"
export CHAC_CONFIG_PATH="$TEST_DIR/config.json"
bash chac_toolkit/scripts/chac_new_record.sh "$TEST_CASE_DIR" "First Record"
assert_success "chac_new_record.sh execution"

# Dynamically construct the expected filename based on the robust script's logic
# The script will extract "A04" from the directory name and use it as a prefix.
DIR_BASENAME=$(basename "$TEST_CASE_DIR")
CASE_ID_PART=$(echo "$DIR_BASENAME" | cut -d'_' -f2)
EXPECTED_RECORD_FILE="${TEST_CASE_DIR}/${CASE_ID_PART}_record_01_first_record.md"

assert_file_exists "$EXPECTED_RECORD_FILE"
assert_file_contains "$EXPECTED_RECORD_FILE" "# First Record"
assert_file_contains "$EXPECTED_RECORD_FILE" "**Parent Case:** $(basename $TEST_CASE_DIR)"

# Test 2.1: Create the "N+1" (second) record to test context-aware logic
log_info "--- Testing chac_new_record.sh (N+1 case) ---"
bash chac_toolkit/scripts/chac_new_record.sh "$TEST_CASE_DIR" "Second Record"
assert_success "chac_new_record.sh execution for second record"

# Construct the expected filename for the second record
# The script should now correctly parse the first record's name and increment the number
EXPECTED_SECOND_RECORD_FILE="${TEST_CASE_DIR}/${CASE_ID_PART}_record_02_second_record.md"
assert_file_exists "$EXPECTED_SECOND_RECORD_FILE"
assert_file_contains "$EXPECTED_SECOND_RECORD_FILE" "# Second Record"

# Test 3: Conclude the case study
log_info "--- Testing chac_conclude_case.sh ---"
export CHAC_CONFIG_PATH="$TEST_DIR/config.json"
bash chac_toolkit/scripts/chac_conclude_case.sh "$TEST_CASE_DIR"
assert_success "chac_conclude_case.sh execution"
assert_file_contains "$GLOBAL_LOG_PATH" "CASE_STUDY_END_MARKER:"

# --- TEST TEARDOWN ---
log_info "Tearing down test environment..."
rm -rf "$TEST_DIR"
log_info "Test suite completed successfully."
