#!/bin/bash

# CHAC Project Initializer v1.1
# This script scaffolds a new project directory with the complete
# Compensatory Human-AI Collaboration (CHAC) framework.

# --- Safety Check (Guardian Protocol) ---
# Exit immediately if any command fails.
set -e

# This check ensures the script is run from the root of the template repository.
if [ ! -d "chac_toolkit" ] || [ ! -d "case-study" ]; then
  echo "❌ Error: This script must be run from the root of the 'chac-template' repository."
  echo "It requires 'chac_toolkit' and 'case-study' directories to exist for replication."
  exit 1
fi

# --- Interactive Setup ---
read -p "Enter new project directory name (e.g., 'my-new-research'): " new_project_dir

if [ -z "$new_project_dir" ]; then
  echo "❌ Error: Project name cannot be empty."
  exit 1
fi

if [ -d "$new_project_dir" ]; then
  echo "❌ Error: Directory '$new_project_dir' already exists."
  exit 1
fi

echo "🚀 Initializing new CHAC project in './$new_project_dir'..."

# --- Scaffolding ---
# Create the main project directory
mkdir -p "$new_project_dir"

# Create the standard case-study structure inside the new project
mkdir -p "$new_project_dir/case-study/A00_Example_Applied_Case/data"
mkdir -p "$new_project_dir/case-study/A00_Example_Applied_Case/analysis"
mkdir -p "$new_project_dir/case-study/A00_Example_Applied_Case/report"
mkdir -p "$new_project_dir/case-study/M00_Example_Meta_Case/data"
mkdir -p "$new_project_dir/case-study/M00_Example_Meta_Case/analysis"
mkdir -p "$new_project_dir/case-study/M00_Example_Meta_Case/report"

# --- Protocol Replication ---
# Copy the entire toolkit and original protocols to the new project
cp -r chac_toolkit "$new_project_dir/chac_toolkit"
cp -r protocols "$new_project_dir/protocols"
cp README.md "$new_project_dir/README.md"

# --- Entrypoint Generation for New Project ---
# This generates a clean, decoupled entrypoint and instruction set for the new project.

# 1. Create the human-readable README in the new toolkit directory.
cp chac_toolkit/README.md "$new_project_dir/chac_toolkit/README.md"

# 2. Create the AI-specific instruction file.
cp chac_toolkit/AI_INSTRUCTIONS.md "$new_project_dir/chac_toolkit/AI_INSTRUCTIONS.md"

# 3. Create the main entrypoint file that points to the instructional file.
cp entrypoint.txt "$new_project_dir/entrypoint.txt"

# --- Finalization ---
echo "✅ CHAC project '$new_project_dir' initialized successfully."
echo "Navigate to your new project and use 'entrypoint.txt' to start your first session."
