#!/bin/bash

# This script initializes the CHAC environment and starts the AI.
# It includes checks to guide new users.

# --- PRE-FLIGHT CHECKS ---

# 1. Check for user's config.json
if [ ! -f "config.json" ]; then
    echo "❌ ERROR: The root 'config.json' file does not exist."
    echo "This file is required to determine the active experiment."
    echo "Please create it by copying 'config.template.json' and setting your 'active_experiment'."
    exit 1
fi

# 2. Check for jq dependency
if ! command -v jq &> /dev/null; then
    echo "INFO: 'jq' is not installed. Skipping further user guidance checks."
    echo "Starting CHAC AI..."
    # (Add the actual command to start the AI CLI here later)
    exit 0
fi

# --- User Guidance Checks ---

ACTIVE_EXP=$(jq -r '.active_experiment' "config.json")

echo "INFO: Active experiment is '$ACTIVE_EXP'"

# Check 1: Ensure the user is not running a template value.
if [ "$ACTIVE_EXP" == "my_first_experiment" ] || [ "$ACTIVE_EXP" == "_template" ]; then
    echo "------------------------------------------------------------------"
    echo "⚠️ WARNING: You are currently using a template experiment name."
    echo ""
    echo "Please create your own experiment first. To do so:"
    echo "1. Duplicate the 'experiments/_template/' directory."
    echo "2. Rename it (e.g., 'experiments/my_research/')."
    echo "3. Update 'config.json' to point to your new experiment."
    echo "------------------------------------------------------------------"
fi

# Check 2: Check if the active profile is empty and guide the user.
PROFILE_PATH="experiments/$ACTIVE_EXP/chac_profile.md"

if [ ! -f "$PROFILE_PATH" ] || { [ -f "$PROFILE_PATH" ] && [ ! -s "$PROFILE_PATH" ]; }; then
    echo "------------------------------------------------------------------"
    echo "INFO: Your active user profile is missing or empty."
    echo "Path: $PROFILE_PATH"
    echo ""
    echo "A high-quality profile is the foundation of the CHAC framework."
    echo ""
    echo "RECOMMENDED ACTION:"
    echo "Start a session with the AI and use the following master prompt"
    echo "to begin the collaborative profile generation process:"
    echo ""
    echo "---"
    echo "AI, we will now follow the 'Collaborative Profile Generation Protocol' (CHAC_Profile_Co-Creation_Protocol.md) to create my chac_profile.md."
    echo "---"
    echo "------------------------------------------------------------------"
    echo ""
fi

# --- Start the AI ---
echo "Starting CHAC AI with the '$ACTIVE_EXP' experiment..."
# (Add the actual command to start the AI CLI here later)
# Example: python -m chac_cli
