import Mathlib.Tactic.Basic
import Mathlib.Tactic.ExtractGoal

open Lean Parser Parser.Tactic Elab Command Elab.Tactic Meta

/-
The `to_theorem` tactic cannot be used before `intro` because `intro` introduces new variables,
but `hyps1` doesn't have these new variables, while `goal2` contains them,
resulting in an error in `hyps1 ⊢ goal2`.
-/

/-- insert Hypotheses of goalsAfter into the goals of goalsBefore, return the new goals -/
def insertHypsIntoGoals (goalsAfter : List MVarId) (goalsBefore : MVarId) : TacticM (List MVarId) := do
  let mut allNewGoals : List MVarId := []
  for goalAfter in goalsAfter do
    let newGoals ← goalAfter.withContext do
      let lctxAfter ← getLCtx
      goalsBefore.withContext do
        let lctxBefore ← getLCtx
        let mut newGoalsList : List MVarId := []
        for declAfter in lctxAfter do
          if !declAfter.isImplementationDetail then
            let mut IsModifiedOldHyp := lctxBefore.decls.any fun declBefore => -- case 1: modified old hyp
              match declBefore with
              | none => false
              | some d =>
                lctxBefore.getRoundtrippingUserName? d.fvarId ==
                  lctxAfter.getRoundtrippingUserName? declAfter.fvarId&&
                (d.type != declAfter.type ||
                (if d.isLet then d.value? != declAfter.value? else false))
            let mut IsNotNewHyp := lctxBefore.decls.any fun declBefore => -- case 2: new hyp
              match declBefore with
              | none => false
              | some d =>
                lctxBefore.getRoundtrippingUserName? d.fvarId ==
                  lctxAfter.getRoundtrippingUserName? declAfter.fvarId
            if IsModifiedOldHyp || !IsNotNewHyp then
              let newGoal ← mkFreshExprSyntheticOpaqueMVar declAfter.type
              --logInfo m! "tactic state of the extracted theorem:{newGoal}"
              newGoalsList := newGoalsList.concat newGoal.mvarId!
        return newGoalsList
    allNewGoals := allNewGoals ++ newGoals
  return allNewGoals

/-- insert goalAfter into the hyps of goalsBefore, return the new goals -/
def insertGoalIntoHyps (goalsAfter : List MVarId) (goalsBefore : MVarId) : TacticM (List MVarId) := do
  let mut newGoalsList : List MVarId := []
  for goalAfter in goalsAfter do
    let t ← goalAfter.getType
    let p ← mkFreshExprMVar t MetavarKind.syntheticOpaque `h
    let (_, goal) ← MVarId.intro1P $ ← goalsBefore.assert `h t p
    newGoalsList := newGoalsList.concat goal
  return newGoalsList

syntax (name := extractTacs) "extract_tacs" tacticSeq (" using " ident)? : tactic
elab_rules : tactic
  | `(tactic| extract_tacs $tacs:tacticSeq $[using $name?]?) => do
    let name ← if let some name := name?
                then pure name.getId
                else mkAuxName ((← getCurrNamespace) ++ `extracted) 1
    -- Pretty-print the tactic sequence
    let tacStr ← Lean.PrettyPrinter.formatTactic tacs.raw
    logInfo s!"extract proof for {name}:\n{tacStr}"

syntax "extract_name" ident : tactic
elab_rules : tactic
  | `(tactic| extract_name $name) => do
    logInfo m! "extract name: {name}"

syntax "to_theorem" "(" tacticSeq ")" (" using " ident)? : tactic

elab_rules : tactic
  | `(tactic| to_theorem ( $tac:tacticSeq ) $[using $name?]?) => do
  let goalBefore ← getMainGoal
  let type ← goalBefore.getType
  let goalBefore := (← goalBefore.withContext <| mkFreshExprSyntheticOpaqueMVar type).mvarId!
  if(← getUnsolvedGoals).length > 1 then -- When there are multiple goals, don't extract
    evalTactic tac
    return
  else
    -- logInfo m! "Proof state before the tactic: {goalBefore}"
    -- logInfo m! "executing tactic: {tac}"
    evalTactic tac
    let goalsAfter ← getUnsolvedGoals
    -- logInfo m! "Proof states after the tactic: {goalsAfter}"
    try
      if goalsAfter.length == 0 then -- i.e., when there are no unsolved goals
        setGoals [goalBefore]
      else if goalsAfter.length == 1 then
        let goalAfter ← getMainGoal
        let goalsEqual : IO.Ref Bool ← IO.mkRef false  -- create a reference that can be shared across contexts
        goalAfter.withContext do
          let pp2 ← Lean.Meta.ppExpr (← goalAfter.getType)
          goalBefore.withContext do
            let pp1 ← Lean.Meta.ppExpr (← goalBefore.getType)
            goalsEqual.set (pp1.pretty == pp2.pretty)  -- update the shared state
        -- if the two goals are the same, then only the hyps are changed
        if (← goalsEqual.get) then
          let newGoals ← insertHypsIntoGoals goalsAfter goalBefore
          setGoals newGoals
        else -- suppose only the goal is changed. insert the goalAfter into the hyps of goalsBefore
          let newGoals ← insertGoalIntoHyps goalsAfter goalBefore
          setGoals newGoals
      else -- suppose the goal is changed. insert the goalAfter into the hyps of goalsBefore
        let newGoals ← insertGoalIntoHyps goalsAfter goalBefore
        setGoals newGoals
      let newGoals ← getUnsolvedGoals
      let name ← if let some name := name?
                  then pure name
                  else pure $ mkIdent <| Name.mkSimple s!"_extracted_1"
      let full_name ← pure $ mkIdent <| Name.mkSimple $ "full" ++ name.getId.toString
      for _ in List.range (newGoals.length) do
          let g ← getMainGoal
          let ty ← instantiateMVars (← g.getType)
          if !ty.hasExprMVar then
            -- logInfo m! "tactic state of the extracted theorem: {g}"
            evalTactic (← `(tactic| extract_name $name))
            evalTactic (← `(tactic| extract_goal * using $name))
            evalTactic (← `(tactic| set_option pp.all true in extract_goal using $full_name))
            evalTactic (← `(tactic| extract_tacs $tac using $name))
          let _ ← popMainGoal
    catch _ =>
      logInfo m! "tactic state failed"
    setGoals goalsAfter


/-
This is a modified version of `have` that allows for the extraction of the goal.
It is used to extract the goal from the proof state after the `have` statement.
-/

syntax "have_with_extraction " haveDecl : tactic
macro_rules
  | `(tactic| have_with_extraction%$haveTk $id:haveId $bs* : $type := by%$byTk $tacs*) => do
    let tacSeq ← `(tacticSeq| $tacs*)
    let tac ← Lean.withRef byTk `(tactic| with_annotate_state $byTk ($tacs*))
    let tac ← `(tacticSeq| $tac:tactic)
    let tac ← Lean.withRef byTk `(tactic| case body => $(.mk tac):tacticSeq)
    let idStx : TSyntax `ident := ⟨id.raw[0]⟩
    let name := mkIdent <| Name.mkSimple s!"_have_extracted_{idStx.getId}"
    let full_name ← pure $ mkIdent <| Name.mkSimple $ "full" ++ name.getId.toString
    Lean.withRef haveTk `(tactic| focus
      refine no_implicit_lambda% (have $id:haveId $bs* : $type := ?body; ?_)
      (extract_name $name)
      (extract_goal * using $name);
      (set_option pp.all true in extract_goal * using $full_name);
      (extract_tacs $tacSeq using $name);
      $tac)
  | `(tactic| have_with_extraction $d:haveDecl) => `(tactic| refine_lift have $d:haveDecl; ?_; extract_goal)

/-
This is a modified version of `tactics` that allows for the extraction of the goal.
It is used to extract the goal between the proof state before and after the `tactics` statement.
-/

syntax "eval_tacs" tacticSeq : tactic
elab_rules : tactic
  | `(tactic| eval_tacs $tacs:tacticSeq) => do
    evalTactic tacs

syntax "apply_with_extraction " (num)? term tacticSeq : tactic
macro_rules
  | `(tactic| apply_with_extraction $[$len:num]? $h $tacs:tacticSeq) => do
    let len := match len with
      | some n => n.getNat
      | none => 1
    let applyTac ← `(tactic| apply $h)
    let `(tacticSeq| $[$tacList:tactic]*) := tacs | unreachable!
    let extTacs := #[applyTac] ++ tacList.take (len - 1)
    let extTacSeq ← `(tacticSeq| $[$extTacs:tactic]*)
    let resTacs := tacList.drop (len - 1)
    let hid := (⟨h.raw⟩ : TSyntax `ident).getId
    let name := mkIdent <| Name.mkSimple s!"_apply_extracted_${hid}"
    if resTacs.isEmpty then
      `(tactic| (to_theorem ($extTacSeq) using $name))
    else
      let resTacSeq ← `(tacticSeq| $[$resTacs:tactic]*)
      `(tactic| (to_theorem ($extTacSeq) using $name); eval_tacs ($resTacSeq))


syntax "rw_with_extraction " (num)? (optConfig) rwRuleSeq (location)? tacticSeq : tactic
macro_rules
  | `(tactic| rw_with_extraction $[$len:num]? $c:optConfig $s:rwRuleSeq $[$l:location]? $tacs:tacticSeq) => do
    let len := match len with
      | some n => n.getNat
      | none => 1
    let rwTac ← match s with
      | `(rwRuleSeq| [$rs,*]) =>
        -- We show the `rfl` state on `]`
        `(tactic| rw $(c) [$rs,*] $(l)?)
      | _ => Macro.throwUnsupported
    let `(tacticSeq| $[$tacList:tactic]*) := tacs | unreachable!
    let extTacs := #[rwTac] ++ tacList.take (len - 1)
    let extTacSeq ← `(tacticSeq| $[$extTacs:tactic]*)
    let resTacs := tacList.drop (len - 1)
    let hid := (⟨s.raw⟩ : TSyntax `ident).getId
    let name := mkIdent <| Name.mkSimple s!"_rw_extracted_${hid}"
    if resTacs.isEmpty then
      `(tactic| (to_theorem ($extTacSeq) using $name))
    else
      let resTacSeq ← `(tacticSeq| $[$resTacs:tactic]*)
      `(tactic| (to_theorem ($extTacSeq) using $name); eval_tacs ($resTacSeq))
