import json
import time
import subprocess
import os
import resource
from typing import Dict, List
from func_timeout import FunctionTimedOut, func_timeout  # type: ignore

def read_from_process(stdin, timeout: int = 3000) -> dict:
    s = ""
    start_time = time.time()
    
    while True:
        remaining_time = timeout - (time.time() - start_time)
        if remaining_time <= 0:
            raise TimeoutError(f"read process output timeout, over {timeout} seconds")
        
        line = stdin.readline()
        if not line:
            continue # empty data, continue waiting
        
        s += line
        try:
            return json.loads(s)
        except json.JSONDecodeError:
            continue # data is not complete, continue reading

def write_to_process(stdout, cmd):
    """write to process with timeout"""
    data = json.dumps(cmd, ensure_ascii=False) + '\n\n'
    try:
        stdout.write(data)
        stdout.flush()
    except (BrokenPipeError, IOError) as e:
        raise IOError(f"Failed to write to process: {e}") from e
    
def send_command(process, cmd):
    """send command to process"""
    write_to_process(process.stdin, cmd)
    return read_from_process(process.stdout)

def send_command_with_timeout(process, cmd, timeout=60):
    """send command to process with timeout"""
    try:
        return func_timeout(timeout, send_command, args=(process, cmd))
    except (FunctionTimedOut, TimeoutError):
        raise TimeoutError(f"send command timeout, over {timeout} seconds")

def run_lake_build(directory, target_name):
    print(f'{"-"*20} build {target_name} {"-"*20}')
    result = subprocess.run(
        ['lake', 'build', target_name],
        cwd=directory,
        text=True,
        capture_output=True
    )
    print(result.stdout.strip())
    return 
    
def run_env_build(work_dir, log_file, memory_limit=8):
    def preexec_fn():
        soft_limit = hard_limit = memory_limit * 1024**3
        resource.setrlimit(resource.RLIMIT_AS, (soft_limit, hard_limit))
        os.setsid()
        
    command = ['stdbuf', '-i0', '-o0', '-e0', 'lake', 'env', './repl/.lake/build/bin/repl']
    stderr = open(log_file, 'w') if log_file else subprocess.DEVNULL
    process = subprocess.Popen(
        command, 
        stdout=subprocess.PIPE,
        stdin=subprocess.PIPE,
        stderr=stderr,
        text=True,
        bufsize=1,
        cwd=os.path.abspath(work_dir),
        preexec_fn=preexec_fn
    )
    return process

def has_error_response(
    feedback: dict, accept_sorry: bool = True, return_error_messages: bool = False
):
    """
    Checks if the Lean feedback contains an error.

    Args:
        feedback: The Lean feedback as a dictionary.
        accept_sorry: Whether to accept "sorry" statements as "not an error".
            By default, "sorry" statements are not considered errors.
        return_error_messages: Whether to return the feedback error messages.
    """
    if "error" in feedback:
        r = (True, [feedback["error"]]) if return_error_messages else True
        return r

    if "stderr" in feedback:
        r = (True, [feedback["stderr"]]) if return_error_messages else True
        return r

    has_error = False
    error_data_values = []
    sorry_data_values = []
    if "messages" in feedback:
        error_data_values = [
            message["data"]
            for message in feedback.get("messages", [])
            if message.get("severity") == "error"
        ]
        has_error = bool(error_data_values)

        if not accept_sorry:
            warning_data_values = [
                message["data"]
                for message in feedback.get("messages", [])
                if message.get("severity") == "warning"
            ]
            sorry_data_values = [
                warning_data
                for warning_data in warning_data_values
                if "declaration uses 'sorry'" in warning_data
            ]
            has_error = has_error or bool(sorry_data_values)

    if return_error_messages:
        return has_error, error_data_values + sorry_data_values
    else:
        return has_error

def parse_client_response(response: dict):
    """
    Parses the response from the Lean4Client.
    Reponse should be the output of client.Lean4Client.(async_)verify

    Args:
        - response (dict): The response from the Lean4Client.

    Returns:
        - dict: A dictionary containing the following keys:
            - has_error: Whether the response contains an error.
            - is_valid_no_sorry: Whether the response is valid without "sorry" statements.
                this is used for proof verification.
            - is_valid_with_sorry: Whether the response is valid with "sorry.
                this is used for statement verification.
    """
    error_message = response.get("error", None)
    json_response = response.get("response", {})

    error = bool(error_message) or has_error_response(json_response)
    is_valid_no_sorry = (not bool(error_message)) and (
        not has_error_response(json_response, accept_sorry=False)
    )
    is_valid_with_sorry = (not bool(error_message)) and (
        not has_error_response(json_response, accept_sorry=True)
    )

    return {
        "has_error": error,
        "is_valid_no_sorry": is_valid_no_sorry,
        "is_valid_with_sorry": is_valid_with_sorry,
        "time": json_response.get("time", None) if json_response else None,
    }

def parse_mutated_thms(thm_name:str, result: Dict) -> Dict:
    """
    Parse the results to get the mutated thm and hyps
    Note that this is used after executing the HypDrop tactic
    
    Args:
        result: The result from the LeanREPL {"response": {}, "error": None}
        
    Returns:
        List[Dict]: The mutated thm and hypothesis
    """
    response = result.get("response", None)
    if response:
        thm_info = {}
        for m in response['messages']:
            # check illegal
            if m['severity'] == 'error':
                if "unknown free variable" in m['data']:
                    return {}
                else:
                    return None # incorrect format
                
            if thm_name in m['data'] and "dropped_hypothesis" in m['data'] and "sorry" in m['data']:
                if "full" in m['data']:
                    thm_info["full_dropped_hypothesis"] = m['data'].replace("\n", "")
                else:
                    thm_info["dropped_hypothesis"] = m['data'].replace("\n", "")
            elif thm_name in m['data'] and "mutated_version" in m['data'] and "sorry" in m['data']:
                if "full" in m['data']:
                    thm_info["full_mutated_version"] = m['data'].replace("\n", "")
                else:
                    thm_info["mutated_version"] = m['data'].replace("\n", "")
        return thm_info
    return None

def parse_extracted_thms(thm_name:str, result:Dict) -> Dict:
    """
    Parse the results to get the extracted thm
    """
    response = result.get("response", None)
    if response:
        thm_info = {}
        for m in response['messages']:
            if thm_name in m['data'] and "sorry" in m['data']:
                thm_info["formal_theorem"] = m['data'].replace("\n", "")
        return thm_info
    return None

def recover_proof(proof: str) -> str:
    """
    Recover the proof from the string
    """
    replace_dict = {
        "have_with_extraction" : "have",
    }
    for k, v in replace_dict.items():
        proof = proof.replace(k, v)
    return proof
        
def parse_convert_thms(thm_name:str, result:Dict) -> List[Dict]:
    """
    Parse the results to get the thm
    Note that this is used after executing the proof style translation tactics
    """
    response = result.get("response", None)
    if response and "messages" in response:
        thm_list = {}
        for m in response["messages"]:
            # check illegal
            if m['severity'] == 'error':
                if "unknown identifier" in m['data']:
                    return None
            pos = (m['pos']['line'], m['pos']['column'])
            if thm_name in m['data'] and "extract name:" in m['data']:
                thm_list[pos] = m['data'].split("extract name:", 1)[1].strip()
        results = []
        for pos, name in thm_list.items():
            thm_info = {"name": name}
            for m in response['messages']:
                if pos != (m['pos']['line'], m['pos']['column']):
                    continue
                if name in m['data'] and "sorry" in m['data']:
                    if f"full{name}" in m['data']:
                        thm = m['data'].replace("\n", "")
                        thm_info["full_formal_theorem"] = thm
                    else:
                        thm_info["formal_theorem"] = m['data'].replace("\n", "")
                elif f"extract proof for {name}:" in m['data']:
                    proof = m['data'].split(f"extract proof for {name}:\n", 1)[1]
                    thm_info["formal_proof"] = recover_proof(proof)
                # elif f"extract name" in m['data']:
                    # continue
            if "formal_theorem" in thm_info and "formal_proof" in thm_info and "full_formal_theorem" in thm_info:
                results.append(thm_info)
        return results
    else:
        return []