// Copyright 2019 DeepMind Technologies Ltd. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "open_spiel/algorithms/local_best_response.h"
#include "open_spiel/algorithms/best_response.h"

#include "open_spiel/games/kuhn_poker.h"

namespace open_spiel {
namespace algorithms {
namespace {

void TestLocalBestResponseLeduc() {
  // Leduc holdem
  std::string poker = "leduc_holdem";
  std::shared_ptr<const Game> game = LoadGame("leduc_poker");
  TabularPolicy policy = GetUniformPolicy(*game);

  auto start = std::chrono::high_resolution_clock::now();

  auto player = Player{0};
  int num_games = 10000;
  double total_return = 0.;
  std::cout << "Player 1 started\n";
  for (int seed = 0; seed < num_games; seed++) {
    total_return = total_return + LocalBestResponseLeduc(game, policy, player, seed);
  }
  std::cout << "Player 1 return: " << total_return / num_games << "\n";
  total_return = 0.;
  std::cout << "Player 2\n";
  for (int seed = 0; seed < num_games; seed++) {
    total_return = total_return + LocalBestResponseLeduc(game, policy, 1 - player, seed);
  }
  std::cout << "Player 2 return: " << total_return / num_games << "\n";

  algorithms::TabularBestResponse best_response_one(*game, 0, &policy);
  algorithms::TabularBestResponse best_response_two(*game, 1, &policy);

  std::cout << "Player 1 BR: " << best_response_one.Value(*game->NewInitialState()) << "\n";
  std::cout << "Player 2 BR: " << best_response_two.Value(*game->NewInitialState()) << "\n";

  auto stop = std::chrono::high_resolution_clock::now();
  auto duration = std::chrono::duration_cast<std::chrono::milliseconds>(stop - start);
  std::cout << "Local best response time: " << duration.count() << " ms\n";
}

void TestLocalBestResponseUniversal() {
  // Texas holdem
  std::string poker = "texas_holdem";

  TabularPolicy policy;
  std::string name = "universal_poker(betting=limit,numPlayers=2,numRounds=4,blind=10 5,"
                     "firstPlayer=2 1,numSuits=4,numRanks=13,numHoleCards=2,numBoardCards=0 3 "
                     "1 1,raiseSize=10 10 20 20,maxRaises=3 4 4 4)";
  std::shared_ptr<const Game> game = LoadGame(name);

  auto start = std::chrono::high_resolution_clock::now();

  auto player = Player{0};
  LocalBestResponseUniversal(game, policy, player, 0);

  auto stop = std::chrono::high_resolution_clock::now();
  auto duration = std::chrono::duration_cast<std::chrono::milliseconds>(stop - start);
  std::cout << "Local best response time: " << duration.count() << " ms\n";
}
}  // namespace
}  // namespace algorithms
}  // namespace open_spiel

int main(int argc, char **argv) {
  open_spiel::algorithms::TestLocalBestResponseLeduc();
//  open_spiel::algorithms::TestLocalBestResponseUniversal();
}
