// Copyright 2019 DeepMind Technologies Ltd. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.


#include "open_spiel/algorithms/ortools/trunk_exploitability.h"

#include <memory>
#include <utility>
#include <vector>
#include <unordered_map>

#include "open_spiel/algorithms/expected_returns.h"
#include "open_spiel/algorithms/ortools/sequence_form_lp.h"

namespace open_spiel {
namespace algorithms {
namespace ortools {

namespace opres = operations_research;


void RecursivelyUpdateTerminalUtilityConstraints(
    const InfostateNode* opponent_node, double opponent_range,
    std::unordered_map<const InfostateNode*, NodeSpecification>& node_spec,
    const std::map<const InfostateNode*, const InfostateNode*>& opponent_terminal_map) {
  if (opponent_node->type() == kTerminalInfostateNode) {
    const InfostateNode* player_node = opponent_terminal_map.at(opponent_node);
    SPIEL_CHECK_EQ(player_node->type(), kTerminalInfostateNode);
    SPIEL_CHECK_EQ(opponent_node->type(), kTerminalInfostateNode);
    SPIEL_CHECK_EQ(opponent_node->TerminalHistory(),
                   player_node->TerminalHistory());
    SPIEL_CHECK_EQ(opponent_node->terminal_chance_reach_prob(),
                   player_node->terminal_chance_reach_prob());

    opres::MPConstraint* ct = node_spec[opponent_node].ct_child_cf_value;
    SPIEL_CHECK_TRUE(ct);
    SPIEL_CHECK_EQ(ct->ub(), 0.);
    SPIEL_CHECK_TRUE(node_spec[opponent_node].var_cf_value);
    SPIEL_CHECK_EQ(ct->GetCoefficient(node_spec[opponent_node].var_cf_value), -1);
    SPIEL_CHECK_TRUE(node_spec[player_node].var_reach_prob);

    // Take care when changing this code: it is shared with ComputeGradient!
    const double value_weighted_by_opp_range
        = opponent_node->terminal_utility()
        * opponent_node->terminal_chance_reach_prob()
        * opponent_range;
    ct->SetCoefficient(node_spec[player_node].var_reach_prob,
                       value_weighted_by_opp_range);
    return;
  }

  for (const InfostateNode* opponent_child : opponent_node->child_iterator()) {
    RecursivelyUpdateTerminalUtilityConstraints(
        opponent_child, opponent_range, node_spec, opponent_terminal_map);
  }
}

//void RefineSpecToValueSolvingSubgame(
//    const InfostateNode* opponent_root,
//    absl::Span<const double> opponent_range,
//    const std::map<const InfostateNode*, const InfostateNode*>& opponent_terminal_map,
//    std::unordered_map<const InfostateNode*, NodeSpecification>& player_spec) {
//  SPIEL_CHECK_EQ(opponent_root->num_children(), opponent_range.size());
//
//  // Set the reach probabilities of empty sequences that would correspond to
//  // the trunk's leaf nodes to be equal to the player's range at those nodes.
//  for (int i = 0; i < opponent_range.size(); ++i) {
//    RecursivelyUpdateTerminalUtilityConstraints(
//        opponent_root->child_at(i), opponent_range[i],
//        player_spec, opponent_terminal_map);
//  }
//}

//std::vector<double> ComputeGradient(
//    const std::vector<InfostateNode*>& player_terminals,
//    const std::map<const InfostateNode*, const InfostateNode*>& player_terminal_map,
//    // We use player spec to get utility_weighted_by_opp_range.
//    std::unordered_map<const InfostateNode*, NodeSpecification>& player_spec,
//    std::unordered_map<const InfostateNode*, NodeSpecification>& opponent_spec) {
//  std::vector<double> gradient(player_terminals.size(), 0.);
//  for (int i = 0; i < player_terminals.size(); ++i) {
//    const InfostateNode* player_node = player_terminals[i];
//    const InfostateNode* opponent_node = player_terminal_map.at(player_node);
//
//    operations_research::MPVariable* opponent_reach =
//        opponent_spec.at(opponent_node).var_reach_prob;
//    SPIEL_CHECK_TRUE(opponent_reach);
//
//    opres::MPConstraint* ct = player_spec.at(opponent_node).ct_child_cf_value;
//    SPIEL_CHECK_TRUE(ct);
//    SPIEL_CHECK_TRUE(player_spec.at(player_node).var_reach_prob);
//
//    // Avoid recalculating this again. This corresponds to:
//    // = opponent_node->terminal_utility()
//    // * opponent_node->terminal_chance_reach_prob()
//    // * opponent_range;
//    const double utility_weighted_by_opp_range =
//        ct->GetCoefficient(player_spec[player_node].var_reach_prob);
//
//    // As we compute best response for the player, we need to flip the sign,
//    // as the utility was originally for the opponent.
//    gradient[i] = - utility_weighted_by_opp_range
//                * opponent_reach->solution_value();
//  }
//  return gradient;
//}

//// TODO: gradient as move
//DecisionVector<std::vector<double>> RefineBestResponseToCfBestResponse(
//    const InfostateTree& player_tree, absl::Span<double> player_cf_gradient,
//    std::unordered_map<const InfostateNode*, NodeSpecification>& player_spec) {
//  std::mt19937 mt;
//  DecisionVector<std::vector<double>> strategy(&player_tree);
//  BottomUp(
//    player_tree, player_cf_gradient,
//    /*observe_rewards_fn=*/[&](DecisionId id, absl::Span<const double> rewards)
//    {
//      const InfostateNode* node = player_tree.decision_infostate(id);
//      SPIEL_CHECK_EQ(rewards.size(), node->num_children());
//      strategy[id] = std::vector(node->num_children(), 0.);
//      auto node_reach = player_spec[node].var_reach_prob;
//      SPIEL_CHECK_TRUE(node_reach);
//      if (node_reach->solution_value() > 0) {
//        size_t i = 0;
//        for (InfostateNode* child : node->child_iterator()) {
//          auto child_reach = player_spec[child].var_reach_prob;
//          SPIEL_CHECK_TRUE(child_reach);
//          strategy[id][i++] = child_reach->solution_value()
//              / node_reach->solution_value();
//        }
//      } else {
//        auto iter_max = std::max_element(rewards.begin(), rewards.end());
//        const double max_reward = *iter_max;
//        std::vector<size_t> max_indices;
//        for (int i = 0; i < rewards.size(); i++) {
//          if (fabs(max_reward - rewards[i]) < 1e-10) max_indices.push_back(i);
//        }
//        std::uniform_int_distribution<int> dist(0, max_indices.size() - 1);
//        const int resp_idx = dist(mt);
//        strategy[id][resp_idx] = 1.;
//      }
//      return strategy[id];
//    });
//  return strategy;
//}

void RecursivelyRefineSpecFixStrategyWithPolicy(
    const InfostateNode* player_node,
    const Policy& fixed_policy,
    SequenceFormLpSpecification* specification,
    double strategy_epsilon) {
  if (player_node->type() == kDecisionInfostateNode) {
    ActionsAndProbs local_policy =
        fixed_policy.GetStatePolicy(player_node->infostate_string());
    if (!local_policy.empty()) {  // Fix policy at this node!
      SPIEL_DCHECK_EQ(local_policy.size(), player_node->num_children());
      SPIEL_DCHECK_TRUE(IsValidProbDistribution(local_policy));
      std::unordered_map<const InfostateNode*, NodeSpecification>& node_spec =
          specification->node_spec();

      const int num_actions = player_node->num_children();
      for (int i = 0; i < num_actions; ++i) {
        const InfostateNode* player_child = player_node->child_at(i);
        SPIEL_DCHECK_EQ(player_node->legal_actions()[i], local_policy[i].first);
        SPIEL_DCHECK_TRUE(node_spec[player_node].var_reach_prob);
        SPIEL_DCHECK_TRUE(node_spec[player_child].var_reach_prob);

        const double prob = (1. - strategy_epsilon) * local_policy[i].second
                          + strategy_epsilon / num_actions;
        // Creates a constraint: prob * r(parent) = r(child)
        opres::MPConstraint* ct = specification->solver()->MakeRowConstraint(0., 0.);
        ct->SetCoefficient(node_spec[player_node].var_reach_prob, prob);
        ct->SetCoefficient(node_spec[player_child].var_reach_prob, -1.);
      }
    }
  }

  for (const InfostateNode* player_child : player_node->child_iterator()) {
    RecursivelyRefineSpecFixStrategyWithPolicy(
        player_child, fixed_policy, specification, strategy_epsilon);
  }
}

double ComputeRootValueWhileFixingStrategy(
    SequenceFormLpSpecification* specification, const Policy& fixed_policy,
    Player fixed_player, double strategy_epsilon) {
  specification->SpecifyLinearProgram(fixed_player);
  RecursivelyRefineSpecFixStrategyWithPolicy(
      specification->trees()[fixed_player]->mutable_root(),
      fixed_policy, specification, strategy_epsilon);
  return specification->Solve();
}

double TrunkExploitability(SequenceFormLpSpecification* spec,
                           const Policy& trunk_policy,
                           double strategy_epsilon) {
  return (- ComputeRootValueWhileFixingStrategy(spec, trunk_policy, 0, strategy_epsilon)
          - ComputeRootValueWhileFixingStrategy(spec, trunk_policy, 1, strategy_epsilon)) / 2.;
}

double TrunkPlayerExploitability(
    SequenceFormLpSpecification* spec, const Policy& trunk_policy,
    Player p, absl::optional<double> maybe_game_value,
    double strategy_epsilon) {
  double game_value;
  if (maybe_game_value.has_value()) {
    game_value = maybe_game_value.value();
  } else {
    spec->SpecifyLinearProgram(Player{0});
    game_value = spec->Solve();
  }
  // Switch sign appropriately - game value is defined for player 0!
  const double root_value = game_value * (1 - 2*p);
  const double fixed_value =
      ComputeRootValueWhileFixingStrategy(spec, trunk_policy, p, strategy_epsilon);
  return root_value - fixed_value;
}

}  // namespace ortools
}  // namespace algorithms
}  // namespace open_spiel
