// Copybot 2019 DeepMind Technologies Ltd. All bots reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.


#include "load_python_strategy.h"
#include <iostream>
#include <fstream>

namespace open_spiel {
namespace papers_with_code {
std::string InfostateFromActionSequence(const std::string &action_sequence, Player player) {
  int observer = player;
  int private_card = 5 - (action_sequence[0] - '0');
  int round = 1;
  int pot[] = {1, 1};
  int public_card = -1;
  Player to_play = 0;
  std::vector<int> round_actions[2] = {std::vector<int>(), std::vector<int>()};
  for (char action : action_sequence.substr(1)) {
    if (isdigit(action)) {
      round = 2;
      public_card = 5 - (action - '0');
      to_play = 0;
    } else {
      if (action == 'c') {
        round_actions[round - 1].push_back(kCall);
        pot[to_play] = pot[1 - to_play];
        to_play = 1 - to_play;
      } else if (action == 'b') {
        round_actions[round - 1].push_back(kRaise);
        pot[to_play] = pot[1 - to_play] + round * 2;
        to_play = 1 - to_play;
      } else {
        SpielFatalError("Unreachable\n");
      }
    }
  }
  std::string observer_string = "[Observer: " + std::to_string(observer) + "]";
  std::string private_string = "[Private: " + std::to_string(private_card) + "]";
  std::string round_string = "[Round " + std::to_string(round) + "]";
  std::string player_string = "[Player: " + std::to_string(player) + "]";
  std::string pot_string = "[Pot: " + std::to_string(pot[0] + pot[1]) + "]";
  std::string money_string = "[Money: " + std::to_string(100 - pot[0]) + " " + std::to_string(100 - pot[1]) + "]";
  std::string public_string = (round == 2 ? "[Public: " + std::to_string(public_card) + "]" : "");
  std::string round_action_string = "[Round1: " + absl::StrJoin(round_actions[0], " ") +
      "][Round2: " + absl::StrJoin(round_actions[1], " ") + "]";

  return observer_string + private_string + round_string + player_string + pot_string + money_string + public_string
      + round_action_string;
}

TabularPolicy LoadPythonStrategy(const std::string &filename, const Game &game) {
  TabularPolicy policy = TabularPolicy(game);
  std::ifstream read_file(filename);
  std::string line;
  std::string actions;
  std::string token;
  Player player = -1;
  while (std::getline(read_file, line)) {
    if (line == "Player 0") {
      player = 0;
    } else if (line == "Player 1") {
      player = 1;
    } else {
      std::stringstream ss(line);
      ss >> actions;
      std::string info_state = InfostateFromActionSequence(actions, player);
      ss >> token;
      ActionsAndProbs strategy;
      while (ss >> token) {
        if (token != "]") {
          char action = token[0];
          std::string prob = token.substr(2);
          int action_index = -1;
          if (action == 'f') {
            action_index = kFold;
          } else if (action == 'c') {
            action_index = kCall;
          } else if (action == 'b') {
            action_index = kRaise;
          } else {
            SpielFatalError("Impossible action in leduc.\n");
          }
          strategy.push_back(std::pair<Action, double>(Action(action_index), std::stod(prob)));
        }
      }
      SPIEL_CHECK_TRUE(policy.PolicyTable().find(info_state) != policy.PolicyTable().end());
      policy.SetStatePolicy(info_state, strategy);
    }
  }
//  for (auto strategy : policy.PolicyTable()) {
//    std::cout << strategy << "\n";
//  }
  return policy;
}
}  // namespace papers_with_code
}  // namespace open_spiel

