// Copyright 2019 DeepMind Technologies Ltd. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "open_spiel/spiel_bots.h"
#include "open_spiel/papers_with_code/1906.06412.value_functions/bot.h"
#include "open_spiel/games/nfg_game.h"
#include "open_spiel/games/kuhn_poker.h"
#include "open_spiel/algorithms/ortools/sequence_form_lp.h"
#include "tabularize_bot.h"
#include "open_spiel/algorithms/tabular_exploitability.h"
#include "open_spiel/algorithms/best_response.h"
#include "open_spiel/game_transforms/turn_based_simultaneous_game.h"
#include "open_spiel/game_transforms/restricted_nash_response.h"
#include "open_spiel/algorithms/best_response.h"
#include "open_spiel/algorithms/expected_returns.h"
#include "open_spiel/algorithms/cfr.h"
#include "open_spiel/games/universal_poker.h"

#include <cmath>
#include <iostream>
#include <fstream>

namespace open_spiel {
namespace papers_with_code {
namespace {

enum ActionType { kFold = 0, kCall = 1, kBet = 2, kAllIn = 3, kHalfPot = 4 };

double BotAgainstPolicy(std::unique_ptr<Bot> bot,
                        const TabularPolicy &fixed_policy,
                        const std::shared_ptr<const Game> &game,
                        Player fixed_player, int num_games) {
  std::mt19937 rng;
  Player player = 1 - fixed_player;
  std::shared_ptr<TabularPolicy> policy = std::make_shared<TabularPolicy>();
  double reward = 0;
  for (int i = 0; i < num_games; i++) {
    bot->Restart();
    std::unique_ptr<State> state = game->NewInitialState();
    while (!state->IsTerminal()) {
      std::pair<ActionsAndProbs, Action> step = bot->StepWithPolicy(*state);
      if (state->IsPlayerActing(player)) {
        if (policy->PolicyTable().find(state->InformationStateString(player)) == policy->PolicyTable().end()) {
          policy->SetStatePolicy(state->InformationStateString(player), step.first);
        }
      }
      if (state->IsSimultaneousNode()) {
        std::vector<Action> joint_action(2);
        joint_action[player] = open_spiel::SampleAction(policy->PolicyTable()[state->InformationStateString(player)],
                                                        std::uniform_real_distribution<double>(0.0, 1.0)(rng)).first;
        joint_action[fixed_player] =
            open_spiel::SampleAction(fixed_policy.PolicyTable().at(state->InformationStateString(fixed_player)),
                                     std::uniform_real_distribution<double>(0.0, 1.0)(rng)).first;
        state->ApplyActions(joint_action);
      } else if (state->CurrentPlayer() == player) {
        Action action = open_spiel::SampleAction(policy->PolicyTable()[state->InformationStateString(player)],
                                                 std::uniform_real_distribution<double>(0.0, 1.0)(rng)).first;
        state->ApplyAction(action);
      } else if (state->CurrentPlayer() == fixed_player) {
        Action action =
            open_spiel::SampleAction(fixed_policy.PolicyTable().at(state->InformationStateString(fixed_player)),
                                     std::uniform_real_distribution<double>(0.0, 1.0)(rng)).first;
        state->ApplyAction(action);
      } else if (state->IsChanceNode()) {
        std::vector<std::pair<Action, double>> outcomes = state->ChanceOutcomes();
        Action action =
            open_spiel::SampleAction(outcomes, std::uniform_real_distribution<double>(0.0, 1.0)(rng)).first;
        state->ApplyAction(action);
      }
    }
    std::cout << reward << "\n";
    reward = reward + state->Returns()[1 - fixed_player];
  }
  return reward / num_games;
}

std::unique_ptr<Bot> CreateBot(const BotParameters &params,
                               const std::shared_ptr<const Game> &game,
                               const TabularPolicy &fixed_policy,
                               Player fixed_player,
                               double p) {
  SherlockBotFactory bot_factory = SherlockBotFactory();

  std::shared_ptr<const Game> rnr_game = ConvertToRNR(*game, fixed_player, p);

  return bot_factory.Create(rnr_game, 1 - fixed_player, params, "srnr", fixed_policy);
}

std::shared_ptr<TabularPolicy>
CreateStepRestrictedNashResponsePolicyFromSetup(const BotParameters &params,
                                      const std::shared_ptr<const Game> &game,
                                      const TabularPolicy &fixed_policy,
                                      Player fixed_player,
                                      double p) {
  SherlockBotFactory bot_factory = SherlockBotFactory();

  std::shared_ptr<const Game> rnr_game = ConvertToRNR(*game, fixed_player, p);

  std::unique_ptr<SherlockBot> bot = bot_factory.Create(rnr_game, 1 - fixed_player, params, true, "srnr", fixed_policy);

  std::shared_ptr<TabularPolicy> bot_policy = tabularize_bot::FullBotPolicy(
      std::move(bot), 1 - fixed_player, rnr_game);

  return bot_policy;
}

void TestWithSetupOnline(const std::shared_ptr<const Game> &game,
                         const BotParameters &params,
                         std::ofstream myfile,
                         int rounds,
                         double p) {
  auto start = std::chrono::high_resolution_clock::now();
  SequenceFormLpSpecification specification(*game);
  specification.SpecifyLinearProgram(0);
  double game_value = specification.Solve();
  auto stop = std::chrono::high_resolution_clock::now();
  auto duration = std::chrono::duration_cast<std::chrono::milliseconds>(stop - start);
  std::cout << "Lp solve: " << duration.count() << " ms\n";
  for (int iterations : {1, 2, 3, 5, 8, 13, 21, 34, 55, 89, 144}) {
    start = std::chrono::high_resolution_clock::now();
    auto fixed_player = Player(1);
    std::shared_ptr<const Game> sequential_game;
    if (game->GetType().dynamics == GameType::Dynamics::kSimultaneous) {
      sequential_game = ConvertToTurnBased(*game);
    } else {
      sequential_game = game;
    }
    stop = std::chrono::high_resolution_clock::now();
    duration = std::chrono::duration_cast<std::chrono::milliseconds>(stop - start);
    std::cout << "Game conversion: " << duration.count() << " ms\n";
    start = std::chrono::high_resolution_clock::now();
    algorithms::CFRSolver solver(*sequential_game);
    for (int i = 0; i < iterations; i++) {
      solver.EvaluateAndUpdatePolicy();
    }
    const std::shared_ptr<Policy> average_policy = solver.AveragePolicy();
    TabularPolicy fixed_policy = TabularPolicy(*sequential_game, *average_policy);
    stop = std::chrono::high_resolution_clock::now();
    duration = std::chrono::duration_cast<std::chrono::milliseconds>(stop - start);
    std::cout << "Cfr strategy creation: " << duration.count() << " ms\n";

    start = std::chrono::high_resolution_clock::now();
    algorithms::TabularBestResponse best_response(*sequential_game, 1 - fixed_player, &fixed_policy);
    stop = std::chrono::high_resolution_clock::now();
    duration = std::chrono::duration_cast<std::chrono::milliseconds>(stop - start);
    std::cout << "Best response: " << duration.count() << " ms\n";

    start = std::chrono::high_resolution_clock::now();
    std::unique_ptr<Bot> bot = CreateBot(params, game, fixed_policy, fixed_player, p);
    double result = BotAgainstPolicy(std::move(bot), fixed_policy, game, fixed_player, rounds);
    stop = std::chrono::high_resolution_clock::now();
    duration = std::chrono::duration_cast<std::chrono::milliseconds>(stop - start);
    std::cout << "Bot part: " << duration.count() << " ms\n";

    double br_value = best_response.Value(*game->NewInitialState());
    double sbr_value = result;
    double br_gain = br_value - game_value;
    double sbr_gain = sbr_value - game_value;
    double br_sbr_diff = br_gain - sbr_gain;

    std::cout << "BR gain: " << br_gain << "\n";
    std::cout << "SBR gain: " << sbr_gain << "\n";
    std::cout << "Diff: " << br_sbr_diff << "\n";

    if (myfile.is_open()) {
      myfile << br_gain << " " << sbr_gain << "\n";
    }
  }
  myfile.close();

}

void TestWithSetupTabular(const std::shared_ptr<const Game> &game,
                          const BotParameters &params,
                          std::ofstream myfile,
                          double p) {
  SequenceFormLpSpecification specification(*game);
  specification.SpecifyLinearProgram(0);
  double game_value = specification.Solve();
  for (int iterations : {1, 2, 3, 5, 8, 13, 21, 34}) {
    auto fixed_player = Player(1);
    std::shared_ptr<const Game> sequential_game;
    if (game->GetType().dynamics == GameType::Dynamics::kSimultaneous) {
      sequential_game = ConvertToTurnBased(*game);
    } else {
      sequential_game = game;
    }
    algorithms::CFRSolver solver(*sequential_game);
    for (int i = 0; i < iterations; i++) {
      solver.EvaluateAndUpdatePolicy();
    }
    const std::shared_ptr<Policy> average_policy = solver.AveragePolicy();
    TabularPolicy fixed_policy = TabularPolicy(*sequential_game, *average_policy);

    std::shared_ptr<TabularPolicy>
        bot_policy =
        CreateStepRestrictedNashResponsePolicyFromSetup(params, game, fixed_policy, fixed_player, p);

    algorithms::TabularBestResponse best_response(*sequential_game, 1 - fixed_player, &fixed_policy);
    algorithms::TabularBestResponse best_response_to_srnr(*sequential_game, fixed_player, &*bot_policy);

    fixed_policy.ImportPolicy(*bot_policy);

    double br_value = best_response.Value(*game->NewInitialState());
    double sbr_value = algorithms::ExpectedReturns(*game->NewInitialState(), fixed_policy, 1000)[0];
    double opp_br_value = best_response_to_srnr.Value(*game->NewInitialState());
    double br_gain = br_value - game_value;
    double sbr_gain = sbr_value - game_value;
    double br_sbr_diff = br_gain - sbr_gain;
    double exploitability = opp_br_value + game_value;

    std::cout << "CFR Iterations: " << iterations << "\n";
    std::cout << "BR gain: " << br_gain << "\n";
    std::cout << "SRNR gain: " << sbr_gain << "\n";
    std::cout << "Diff: " << br_sbr_diff << "\n";
    std::cout << "Exploitability: " << exploitability << "\n\n";

    if (myfile.is_open()) {
      myfile << br_gain << " " << sbr_gain << "\n";
    }
  }
  myfile.close();
}

void TestKuhn() {
  std::shared_ptr<const Game> game = LoadGame("kuhn_poker");

  BotParameters params{
      {"seed", BotParameter(0)},
      {"cfr_iterations", BotParameter(1000)},
      {"max_move_ahead_limit", BotParameter(10)},
      {"max_particles", BotParameter(1000)},
      {"use_bandits_for_cfr", BotParameter("RegretMatchingPlus")},
      {"save_values_policy", BotParameter("average")},
      {"non_terminal_evaluator", BotParameter("cfr")},
      {"subgame_cfr_iterations", BotParameter(10)},
  };
  std::ofstream myfile("srnr_kuhn.txt");
  double p = 0.;
  TestWithSetupTabular(game, params, std::move(myfile), p);
}

void TestGoofspielOpt(double p) {
  std::shared_ptr<const Game> game = LoadGame("goofspiel("
                                              "players=2,"
                                              "num_cards=5,"
                                              "imp_info=True,"
                                              "points_order=descending"
                                              ")");

  BotParameters params{
      {"seed", BotParameter(0)},
      {"cfr_iterations", BotParameter(1)},
      {"max_move_ahead_limit", BotParameter(1)},
      {"max_particles", BotParameter(1000)},
      {"use_bandits_for_cfr", BotParameter("RegretMatchingPlus")},
      {"save_values_policy", BotParameter("average")},
      {"non_terminal_evaluator", BotParameter("cfr")},
      {"subgame_cfr_iterations", BotParameter(1)},
  };
  std::ofstream myfile("srnr_gs_opt.txt");
  TestWithSetupTabular(game, params, std::move(myfile), p);
}

void TestGoofspielNet() {
  std::shared_ptr<const Game> game = LoadGame("goofspiel("
                                              "players=2,"
                                              "num_cards=3,"
                                              "imp_info=True,"
                                              "points_order=descending"
                                              ")");

  std::string current_dir = __FILE__;
  current_dir.resize(current_dir.rfind("/"));

  BotParameters params{
      {"seed", BotParameter(0)},
      {"num_layers", BotParameter(5)},
      {"num_width", BotParameter(5)},
      {"num_inputs_regression", BotParameter(8)},
      {"cfr_iterations", BotParameter(100)},
      {"max_move_ahead_limit", BotParameter(50)},
      {"max_particles", BotParameter(1000)},
      {"device", BotParameter("cpu")},
      {"use_bandits_for_cfr", BotParameter("RegretMatchingPlus")},
      {"save_values_policy", BotParameter("average")},
      {"zero_sum_regression", BotParameter(false)},
      {"load_from",
       BotParameter(
           absl::StrCat(current_dir, "/snapshots/iigs3/random.model"))},
  };
  std::ofstream myfile("srnr_gs_net.txt");
  double p = 0.5;
  TestWithSetupTabular(game, params, std::move(myfile), p);
}

void TestGoofspiel5Net() {
  std::shared_ptr<const Game> game = LoadGame("goofspiel("
                                              "players=2,"
                                              "num_cards=5,"
                                              "imp_info=True,"
                                              "points_order=descending"
                                              ")");

  std::string current_dir = __FILE__;
  current_dir.resize(current_dir.rfind("/"));

  BotParameters params{
      {"seed", BotParameter(0)},
      {"num_layers", BotParameter(5)},
      {"num_width", BotParameter(5)},
      {"num_inputs_regression", BotParameter(172)},
      {"cfr_iterations", BotParameter(100)},
      {"max_move_ahead_limit", BotParameter(11)},
      {"max_particles", BotParameter(1000)},
      {"device", BotParameter("cpu")},
      {"use_bandits_for_cfr", BotParameter("RegretMatchingPlus")},
      {"save_values_policy", BotParameter("average")},
      {"zero_sum_regression", BotParameter(false)},
      {"load_from",
       BotParameter(
           absl::StrCat(current_dir, "/snapshots/iigs5/448.model"))},
  };
  std::ofstream myfile("srnr_gs5_net.txt");
  double p = 0.5;
  TestWithSetupTabular(game, params, std::move(myfile), p);
}

void TestLeducNet() {
  std::shared_ptr<const Game> game = LoadGame("leduc_poker");

  std::string current_dir = __FILE__;
  current_dir.resize(current_dir.rfind("/"));

  BotParameters params{
      {"seed", BotParameter(0)},
      {"num_layers", BotParameter(5)},
      {"num_width", BotParameter(5)},
      {"num_inputs_regression", BotParameter(14)},
      {"cfr_iterations", BotParameter(1000)},
      {"max_move_ahead_limit", BotParameter(1)},
      {"max_particles", BotParameter(1000)},
      {"device", BotParameter("cpu")},
      {"use_bandits_for_cfr", BotParameter("RegretMatchingPlus")},
      {"save_values_policy", BotParameter("average")},
      {"zero_sum_regression", BotParameter(true)},
      {"load_from",
       BotParameter(
           absl::StrCat(current_dir, "/snapshots/leduc/448.model"))},
  };
  std::ofstream myfile("srnr_leduc_net.txt");
  double p = 0.;
  TestWithSetupTabular(game, params, std::move(myfile), p);
}

void TestLeducOnlineNet() {
  auto start = std::chrono::high_resolution_clock::now();
  std::shared_ptr<const Game> game = LoadGame("leduc_poker");

  std::string current_dir = __FILE__;
  current_dir.resize(current_dir.rfind("/"));

  BotParameters params{
      {"seed", BotParameter(0)},
      {"num_layers", BotParameter(5)},
      {"num_width", BotParameter(5)},
      {"num_inputs_regression", BotParameter(14)},
      {"cfr_iterations", BotParameter(1000)},
      {"max_move_ahead_limit", BotParameter(1)},
      {"max_particles", BotParameter(1000)},
      {"device", BotParameter("cpu")},
      {"use_bandits_for_cfr", BotParameter("RegretMatchingPlus")},
      {"save_values_policy", BotParameter("average")},
      {"zero_sum_regression", BotParameter(true)},
      {"load_from",
       BotParameter(
           absl::StrCat(current_dir, "/snapshots/leduc/448.model"))},
  };
  std::ofstream myfile("srnr_leduc_net_online.txt");
  auto stop = std::chrono::high_resolution_clock::now();
  auto duration = std::chrono::duration_cast<std::chrono::milliseconds>(stop - start);
  std::cout << "Initialization: " << duration.count() << " ms\n";
  double p = 0.5;
  TestWithSetupOnline(game, params, std::move(myfile), 1, p);
}

void TestLeducOnline() {
  auto start = std::chrono::high_resolution_clock::now();
  std::shared_ptr<const Game> game = LoadGame("leduc_poker");

  std::string current_dir = __FILE__;
  current_dir.resize(current_dir.rfind("/"));

  BotParameters params{
      {"seed", BotParameter(0)},
      {"cfr_iterations", BotParameter(100)},
      {"max_move_ahead_limit", BotParameter(1)},
      {"max_particles", BotParameter(1000)},
      {"use_bandits_for_cfr", BotParameter("RegretMatchingPlus")},
      {"save_values_policy", BotParameter("average")},
      {"non_terminal_evaluator", BotParameter("cfr")},
      {"subgame_cfr_iterations", BotParameter(10)},
  };
  std::ofstream myfile("srnr_leduc_opt_online.txt");
  auto stop = std::chrono::high_resolution_clock::now();
  auto duration = std::chrono::duration_cast<std::chrono::milliseconds>(stop - start);
  std::cout << "Initialization: " << duration.count() << " ms\n";
  double p = 0.5;
  TestWithSetupOnline(game, params, std::move(myfile), 1, p);
}

void TestLeducOpt(double p) {
  std::shared_ptr<const Game> game = LoadGame("leduc_poker");

  std::string current_dir = __FILE__;
  current_dir.resize(current_dir.rfind("/"));

  BotParameters params{
      {"seed", BotParameter(0)},
      {"cfr_iterations", BotParameter(100)},
      {"max_move_ahead_limit", BotParameter(1)},
      {"max_particles", BotParameter(1000)},
      {"use_bandits_for_cfr", BotParameter("RegretMatchingPlus")},
      {"save_values_policy", BotParameter("average")},
      {"non_terminal_evaluator", BotParameter("cfr")},
      {"subgame_cfr_iterations", BotParameter(10)},
  };
  std::ofstream myfile("srnr_leduc_opt_" + std::to_string(p)+ ".txt");
  TestWithSetupTabular(game, params, std::move(myfile), p);
}

void TestBlotto() {
  std::shared_ptr<const Game> game = LoadGame("blotto");

  std::string current_dir = __FILE__;
  current_dir.resize(current_dir.rfind("/"));

  BotParameters params{
      {"seed", BotParameter(0)},
      {"cfr_iterations", BotParameter(100)},
      {"max_move_ahead_limit", BotParameter(1)},
      {"max_particles", BotParameter(1000)},
      {"use_bandits_for_cfr", BotParameter("RegretMatchingPlus")},
      {"save_values_policy", BotParameter("average")},
      {"non_terminal_evaluator", BotParameter("cfr")},
      {"subgame_cfr_iterations", BotParameter(10)},
  };
  std::ofstream myfile("srnr_blotto.txt");
  double p = 0.5;
  TestWithSetupTabular(game, params, std::move(myfile), p);
}

void TestPhantomTTT() {
  std::string name = "liars_dice";
  std::shared_ptr<const Game> game = LoadGame(name);

  std::string current_dir = __FILE__;
  current_dir.resize(current_dir.rfind("/"));

  BotParameters params{
      {"seed", BotParameter(0)},
      {"cfr_iterations", BotParameter(100)},
      {"max_move_ahead_limit", BotParameter(1)},
      {"max_particles", BotParameter(1000)},
      {"use_bandits_for_cfr", BotParameter("RegretMatchingPlus")},
      {"save_values_policy", BotParameter("average")},
      {"non_terminal_evaluator", BotParameter("cfr")},
      {"subgame_cfr_iterations", BotParameter(10)},
  };
  std::ofstream myfile(name + ".txt");
  double p = 0.5;
  TestWithSetupTabular(game, params, std::move(myfile), p);
}

double CountNodesStep(std::unique_ptr<State> state) {
  double nodes = 0;

  if(state->IsTerminal()) {
    return 1;
  }

  for(Action action : state->LegalActions()) {
    std::unique_ptr<State> child = state->Child(action);
    nodes = nodes + CountNodesStep(std::move(child));
  }

  return nodes;
}

void UniversalPokerTest() {
  std::string name = "universal_poker(betting=limit,numPlayers=2,numRounds=4,blind=10 5,"
                     "firstPlayer=2 1,numSuits=4,numRanks=13,numHoleCards=2,numBoardCards=0 3 "
                     "1 1,raiseSize=10 10 20 20,maxRaises=3 4 4 4)";
  std::shared_ptr<const Game> game = LoadGame(name);

  std::unique_ptr<State> state = game->NewInitialState();

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{0});

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{1});

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{2});

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{3});

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{kCall});

  state->ApplyAction(Action{kBet});

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{kBet});

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{kBet});

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{kCall});

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{4});

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{5});

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{6});

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{kCall});

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{kBet});

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{kBet});

  std::cout << state->InformationStateString(0) << "\n";
  std::cout << state->InformationStateString(1) << "\n";
  std::cout << state->ObservationString(0) << "\n";
  std::cout << state->ObservationString(1) << "\n";
  std::cout << state->MoveNumber() << "\n";

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{kBet});

  std::cout << state->InformationStateString(0) << "\n";
  std::cout << state->InformationStateString(1) << "\n";
  std::cout << state->ObservationString(0) << "\n";
  std::cout << state->ObservationString(1) << "\n";
  std::cout << state->MoveNumber() << "\n";

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  state->ApplyAction(Action{kCall});

  std::cout << state->InformationStateString(0) << "\n";
  std::cout << state->InformationStateString(1) << "\n";
  std::cout << state->ObservationString(0) << "\n";
  std::cout << state->ObservationString(1) << "\n";
  std::cout << state->MoveNumber() << "\n";

  std::cout << "Is chance node: " << state->IsChanceNode() << "\n";
  std::cout << "Legal actions size:" << state->LegalActions().size() << "\n";

  std::cout << state->InformationStateString(0) << "\n";
  std::cout << state->InformationStateString(1) << "\n";
  std::cout << state->ObservationString(0) << "\n";
  std::cout << state->ObservationString(1) << "\n";
  std::cout << state->MoveNumber() << "\n";

  double nodes = CountNodesStep(std::move(state));
  std::cout << std::fixed;
  std::cout << "Nodes: " << nodes << "\n";
  std::cout << game->MaxGameLength() << "\n";
}
}
}
}

int main(int argc, char **argv) {
  // Test automatic CFV extraction on simple matrix game
//  open_spiel::papers_with_code::TestKuhn();
//  open_spiel::papers_with_code::TestGoofspielNet();
//  open_spiel::papers_with_code::TestGoofspielOpt(0.1);
//  open_spiel::papers_with_code::TestLeducOnline();
//  open_spiel::papers_with_code::TestLeducOnlineNet();
  open_spiel::papers_with_code::TestLeducOpt(0.1);
  open_spiel::papers_with_code::TestLeducOpt(0.5);
  open_spiel::papers_with_code::TestLeducOpt(0.9);
//  open_spiel::papers_with_code::TestLeducNet();
//  open_spiel::papers_with_code::TestBlotto();
//  open_spiel::papers_with_code::TestPhantomTTT();
//  open_spiel::papers_with_code::TestGoofspiel5Net();
//  open_spiel::papers_with_code::TestLeducOnline();

//    open_spiel::papers_with_code::UniversalPokerTest();
}