import torch
import numpy as np
import json
import os
import base64
import io
import re
import logging
import yaml
from PIL import Image
from transformers import AutoProcessor, AutoModelForCausalLM, AutoTokenizer, PreTrainedTokenizerFast, AutoModel
from tokenizers import Tokenizer
import torch.nn.functional as F
import sys
import time
import math

# --- Imports for dynamic_preprocess ---
import torchvision.transforms as T
from torchvision.transforms.functional import InterpolationMode
# --- End Imports ---

# --- Constants and Functions for dynamic_preprocess ---
IMAGENET_MEAN = (0.485, 0.456, 0.406)
IMAGENET_STD = (0.229, 0.224, 0.225)

def build_transform(input_size):
    MEAN, STD = IMAGENET_MEAN, IMAGENET_STD
    transform = T.Compose([
        T.Lambda(lambda img: img.convert('RGB') if img.mode != 'RGB' else img),
        T.Resize((input_size, input_size), interpolation=InterpolationMode.BICUBIC),
        T.ToTensor(),
        T.Normalize(mean=MEAN, std=STD)
    ])
    return transform

def find_closest_aspect_ratio(aspect_ratio, target_ratios, width, height, image_size):
    best_ratio_diff = float('inf')
    best_ratio = (1, 1)
    area = width * height
    for ratio in target_ratios:
        target_aspect_ratio = ratio[0] / ratio[1]
        ratio_diff = abs(aspect_ratio - target_aspect_ratio)
        if ratio_diff < best_ratio_diff:
            best_ratio_diff = ratio_diff
            best_ratio = ratio
        elif ratio_diff == best_ratio_diff:
            current_area = image_size * image_size * ratio[0] * ratio[1]
            best_area = image_size * image_size * best_ratio[0] * best_ratio[1]
            if abs(area - current_area) < abs(area - best_area):
                 best_ratio = ratio
    return best_ratio

def dynamic_preprocess(image, min_num=1, max_num=12, image_size=448, use_thumbnail=False):
    orig_width, orig_height = image.size
    aspect_ratio = orig_width / orig_height

    target_ratios = set(
        (i, j) for n in range(min_num, max_num + 1) for i in range(1, n + 1) for j in range(1, n + 1) if
        i * j <= max_num and i * j >= min_num)
    target_ratios = sorted(target_ratios, key=lambda x: x[0] * x[1])

    target_aspect_ratio = find_closest_aspect_ratio(
        aspect_ratio, target_ratios, orig_width, orig_height, image_size)

    target_width = image_size * target_aspect_ratio[0]
    target_height = image_size * target_aspect_ratio[1]
    blocks = target_aspect_ratio[0] * target_aspect_ratio[1]

    resized_img = image.resize((int(round(target_width)), int(round(target_height))))
    processed_images = []
    slice_width = int(round(target_width / target_aspect_ratio[0]))
    slice_height = int(round(target_height / target_aspect_ratio[1]))

    for i in range(blocks):
        box = (
            (i % target_aspect_ratio[0]) * slice_width,
            (i // target_aspect_ratio[0]) * slice_height,
            ((i % target_aspect_ratio[0]) + 1) * slice_width,
            ((i // target_aspect_ratio[0]) + 1) * slice_height
        )
        split_img = resized_img.crop(box)
        processed_images.append(split_img)

    if use_thumbnail and blocks > 1:
        thumbnail_img = image.resize((image_size, image_size))
        processed_images.append(thumbnail_img)

    final_images = []
    for img in processed_images:
         if img.size != (image_size, image_size):
              img = img.resize((image_size, image_size), Image.Resampling.BICUBIC)
         final_images.append(img)

    return final_images

class UncertaintyQuantifier:
    """
    Uncertainty Quantification Function (Φ) implementing dual-level uncertainty assessment.
    Supports both Generation-Process Based Uncertainty (Φ_gen+) and Semantic-Marker Based Uncertainty (Φ_sem).
    """
    
    def __init__(self, config=None):
        self.config = config or {}
        self.logger = logging.getLogger(__name__)
        
        # Parameters for generation-based uncertainty
        self.alpha = self.config.get('alpha', 0.6)  # Weight for entropy
        self.beta = self.config.get('beta', 0.4)    # Weight for probability difference
        
        # Parameters for semantic-based uncertainty
        self.sigmoid_k = self.config.get('sigmoid_k', 60)
        self.sigmoid_offset = self.config.get('sigmoid_offset', 0.04)
        
        # Multi-level uncertainty markers lexicon
        self.uncertainty_markers = {
            'strong': {
                'markers': ['uncertain', 'unclear', 'not sure', 'cannot determine', 'difficult to say',
                          'no clear answer', 'speculate', 'unknown', 'difficult to judge', 'hard to tell',
                          'impossible to determine', 'cannot tell', 'cannot identify', 'ambiguous', 'inconclusive',
                          'insufficient information', 'too vague', 'not possible to determine', 'beyond my ability',
                          'cannot be certain', 'no way to know', 'undetermined', 'indeterminate'],
                'weight': 3
            },
            'medium': {
                'markers': ['likely', 'probably', 'might be', 'could be', 'suggests', 'appears',
                          'seems', 'estimate', 'generally', 'often', 'guess', 'hypothesize',
                          'presumably', 'potentially', 'supposedly', 'apparently', 'reasonably',
                          'tentatively', 'arguably', 'plausibly', 'conceivably', 'ostensibly',
                          'seemingly', 'would suggest', 'indicates that', 'tends to be'],
                'weight': 2
            },
            'weak': {
                'markers': ['maybe', 'perhaps', 'possibly', 'somewhat', 'partially', 'relatively',
                          'around', 'about', 'or', 'can say', 'slightly', 'marginally',
                          'to some extent', 'to a degree', 'in some ways', 'sort of',
                          'kind of', 'more or less', 'approximately', 'roughly',
                          'loosely speaking', 'in a sense', 'so to speak', 'as it were'],
                'weight': 1
            }
        }
    
    def phi_gen_plus(self, generated_token_ids, scores):
        """
        Generation-Process Based Uncertainty (Φ_gen+)
        Computes uncertainty from probability distributions during token generation.
        
        Args:
            generated_token_ids: Generated token IDs tensor
            scores: Generation scores (logits) from model
            
        Returns:
            float: Generation-based uncertainty score
        """
        if scores is None or generated_token_ids is None:
            return None
            
        try:
            token_uncertainties = []
            
            for i, token_score in enumerate(scores):
                if i >= generated_token_ids.shape[1]:
                    break
                    
                with torch.no_grad():
                    probs = F.softmax(token_score[0].float(), dim=-1)
                    
                    # Information entropy (hesitation)
                    entropy = -torch.sum(probs * torch.log(probs + 1e-12))
                    
                    # Top-2 probability difference (conviction)
                    top2_probs, _ = torch.topk(probs, 2)
                    prob_diff = top2_probs[0] - top2_probs[1]
                    conviction_uncertainty = max(0, 1 - prob_diff.item())
                    
                    # Combined uncertainty for this token
                    token_uncertainty = (self.alpha * entropy.item() + 
                                       self.beta * conviction_uncertainty)
                    token_uncertainties.append(token_uncertainty)
            
            if token_uncertainties:
                avg_uncertainty = sum(token_uncertainties) / len(token_uncertainties)
                normalized_uncertainty = min(1.0, max(0.0, avg_uncertainty))
                self.logger.info(f"Generation-based uncertainty: {normalized_uncertainty:.4f}")
                return normalized_uncertainty
            else:
                return None
                
        except Exception as e:
            self.logger.warning(f"Error computing generation-based uncertainty: {e}")
            return None
    
    def phi_sem(self, response_text):
        """
        Semantic-Marker Based Uncertainty (Φ_sem)
        Quantifies uncertainty from semantic markers in text.
        
        Args:
            response_text: Text response to analyze
            
        Returns:
            float: Semantic-based uncertainty score
        """
        if not response_text:
            return 1.0
            
        response_lower = response_text.lower()
        words = [w for w in re.split(r'[\s,.!?;:()\[\]{}"\'\\]', response_lower) if w]
        num_words = len(words)
        
        if num_words == 0:
            return 1.0
        
        # Calculate weighted uncertainty score
        raw_weighted_score = 0
        
        for level, config in self.uncertainty_markers.items():
            for marker in config['markers']:
                if ' ' in marker:
                    count = response_lower.count(marker)
                else:
                    count = words.count(marker)
                raw_weighted_score += count * config['weight']
        
        # Add penalty for mid-sentence questions
        if '?' in response_lower[:-1]:
            raw_weighted_score += 2
        
        # Calculate uncertainty density
        score_density = raw_weighted_score / num_words if num_words > 0 else 0
        
        # Apply sigmoid transformation
        try:
            exponent = -self.sigmoid_k * (score_density - self.sigmoid_offset)
            uncertainty = 1 / (1 + math.exp(exponent))
        except OverflowError:
            uncertainty = 1.0 if exponent < 0 else 0.0
        
        self.logger.info(f"Semantic-based uncertainty: {uncertainty:.4f}, density: {score_density:.6f}")
        return uncertainty
    
    def quantify_uncertainty(self, response_text, generated_token_ids=None, scores=None):
        """
        Main uncertainty quantification function implementing priority strategy.
        Uses generation-based uncertainty if available, otherwise falls back to semantic-based.
        
        Args:
            response_text: Text response
            generated_token_ids: Optional token IDs
            scores: Optional generation scores
            
        Returns:
            float: Uncertainty score in [0, 1]
        """
        # Priority 1: Generation-based uncertainty if available
        gen_uncertainty = self.phi_gen_plus(generated_token_ids, scores)
        if gen_uncertainty is not None:
            return gen_uncertainty
        
        # Priority 2: Semantic-based uncertainty as fallback
        return self.phi_sem(response_text)

class ClaimParser:
    """
    Claim Parsing module (P) that transforms unstructured responses into structured information.
    Converts R_i → {(c_j, σ_j, e_j, r_j)}_{j=1}^{m_i}
    """
    
    def __init__(self, config=None):
        self.config = config or {}
        self.logger = logging.getLogger(__name__)
    
    def extract_claims(self, response_text):
        """Extract claims from response text"""
        claims = []
        
        # Pattern matching for structured claims
        claim_pattern = r"CLAIM\s+\d+:\s*(.*?)(?:\n|$)"
        confidence_pattern = r"CONFIDENCE:\s*(\d+)%?"
        evidence_pattern = r"EVIDENCE:\s*(.*?)(?:\n|$)"
        region_pattern = r"REGION:\s*(.*?)(?:\n|$)"
        
        claim_matches = re.finditer(claim_pattern, response_text, re.IGNORECASE)
        
        for claim_match in claim_matches:
            claim_text = claim_match.group(1).strip()
            claim_end_pos = claim_match.end()
            
            # Find corresponding confidence
            confidence_match = re.search(confidence_pattern, 
                                       response_text[claim_end_pos:claim_end_pos+200], 
                                       re.IGNORECASE)
            confidence = int(confidence_match.group(1)) / 100.0 if confidence_match else 0.5
            
            # Find evidence
            evidence_match = re.search(evidence_pattern, 
                                     response_text[claim_end_pos:claim_end_pos+500], 
                                     re.IGNORECASE)
            evidence = evidence_match.group(1).strip() if evidence_match else ""
            
            # Find region
            region_match = re.search(region_pattern, 
                                   response_text[claim_end_pos:claim_end_pos+500], 
                                   re.IGNORECASE)
            region = region_match.group(1).strip() if region_match else ""
            
            claims.append({
                'claim': claim_text,
                'confidence': confidence,
                'evidence': evidence,
                'region': region
            })
        
        # Fallback: treat entire response as one claim if no structured format found
        if not claims:
            claims.append({
                'claim': response_text.strip(),
                'confidence': 0.5,
                'evidence': "",
                'region': ""
            })
        
        return claims
    
    def assess_confidence(self, claims):
        """Assess confidence for extracted claims"""
        # This could be enhanced with more sophisticated confidence assessment
        for claim in claims:
            if 'confidence' not in claim or claim['confidence'] is None:
                claim['confidence'] = 0.5  # Default confidence
        return claims
    
    def associate_evidence(self, claims, response_text):
        """Associate textual evidence with claims"""
        for claim in claims:
            if not claim.get('evidence'):
                # Simple heuristic: use surrounding context as evidence
                claim['evidence'] = response_text[:200] + "..." if len(response_text) > 200 else response_text
        return claims
    
    def parse_response(self, response_text):
        """
        Main parsing function: R_i → {(c_j, σ_j, e_j, r_j)}
        
        Args:
            response_text: Unstructured response text
            
        Returns:
            list: Structured claim tuples
        """
        # Step 1: Extract claims
        claims = self.extract_claims(response_text)
        
        # Step 2: Assess confidence
        claims = self.assess_confidence(claims)
        
        # Step 3: Associate evidence
        claims = self.associate_evidence(claims, response_text)
        
        # Step 4: Visual region mapping (placeholder - would need image context)
        for claim in claims:
            if not claim.get('region'):
                claim['region'] = "global"  # Default to global region
        
        return claims

class EvidenceMapper:
    """
    Evidence Mapping module (M) that links claims to visual regions.
    Implements M_i: C_i × X → (r, σ)
    """
    
    def __init__(self, config=None):
        self.config = config or {}
        self.logger = logging.getLogger(__name__)
    
    def map_claim_to_region(self, claim, image_context=None):
        """
        Map a claim to visual region with confidence.
        
        Args:
            claim: Claim text
            image_context: Optional image context information
            
        Returns:
            tuple: (region, confidence)
        """
        # Placeholder implementation - would need visual understanding capabilities
        # This could be enhanced with actual visual grounding models
        
        # Simple heuristic based on claim content
        claim_lower = claim.lower()
        
        if any(word in claim_lower for word in ['center', 'middle', 'main']):
            return ("center_region", 0.8)
        elif any(word in claim_lower for word in ['left', 'right', 'top', 'bottom']):
            for direction in ['left', 'right', 'top', 'bottom']:
                if direction in claim_lower:
                    return (f"{direction}_region", 0.7)
        elif any(word in claim_lower for word in ['background', 'scene', 'overall']):
            return ("global_region", 0.9)
        elif any(word in claim_lower for word in ['object', 'item', 'thing']):
            return ("object_region", 0.6)
        else:
            return ("global_region", 0.5)
    
    def enhance_claims_with_regions(self, claims, image_context=None):
        """
        Enhance parsed claims with visual region mappings.
        
        Args:
            claims: List of claim dictionaries
            image_context: Optional image context
            
        Returns:
            list: Enhanced claims with region mappings
        """
        enhanced_claims = []
        
        for claim in claims:
            region, region_confidence = self.map_claim_to_region(claim['claim'], image_context)
            
            enhanced_claim = claim.copy()
            enhanced_claim['visual_region'] = region
            enhanced_claim['region_confidence'] = region_confidence
            
            enhanced_claims.append(enhanced_claim)
        
        return enhanced_claims

class BaseAgent:
    """
    Base Agent implementing specialized visual reasoning expertise.
    Part of the expert set E = {1, 2, ..., N}.
    """
    
    def __init__(self, agent_config, base_model):
        self.config = agent_config
        self.base_model = base_model
        self.logger = logging.getLogger(__name__)
        
        # Agent properties
        self.agent_id = agent_config.get('id', 0)
        self.name = agent_config.get('name', 'base_agent')
        self.display_name = agent_config.get('display_name', 'Base Agent')
        self.weight = agent_config.get('weight', 1.0)
        self.enabled = agent_config.get('enabled', True)
        self.keywords = agent_config.get('keywords', [])
        self.prompt_template = agent_config.get('prompt_template', '{instruction}')
    
    def create_prompt(self, instruction, options=None):
        """Create specialized prompt for this agent"""
        template_instruction = self.prompt_template.replace('{instruction}', instruction)
        
        if '{options}' in template_instruction:
            if options:
                options_str = ', '.join(map(str, options)) if isinstance(options, list) else str(options)
                template_instruction = template_instruction.replace('{options}', options_str)
            else:
                template_instruction = template_instruction.replace('{options}', '[No options provided]')
        
        return template_instruction
    
    def analyze(self, image_inputs, instruction, options=None):
        """
        Analysis capability mapping function A_i: X × P_r → R_i
        
        Args:
            image_inputs: Visual input X
            instruction: Question P_r
            options: Optional choices
            
        Returns:
            dict: Agent response with analysis
        """
        if not self.enabled:
            return {
                'response': '[Agent disabled]',
                'generated_token_ids': None,
                'scores': None
            }
        
        prompt = self.create_prompt(instruction, options)
        
        # Add evidence region instruction for better claim-evidence alignment
        prompt += "\n\nImportant: Please structure your response with specific claims and supporting evidence. For key findings, mention which parts of the image support your conclusions."
        
        result = self.base_model.generate_response(image_inputs, prompt)
        return result

class CriticalAgent:
    """
    Critical Agent implementing specialized reasoning critique expertise.
    Functions as specialized reasoning critique expert.
    """
    
    def __init__(self, agent_config, base_model):
        self.config = agent_config
        self.base_model = base_model
        self.logger = logging.getLogger(__name__)
        
        # Critic properties
        self.agent_id = agent_config.get('id', 0)
        self.name = agent_config.get('name', 'critical_agent')
        self.display_name = agent_config.get('display_name', 'Critical Agent')
        self.enabled = agent_config.get('enabled', True)
        self.critique_template = agent_config.get('critique_template', 
                                                'Evaluate the following response: {response}')
    
    def critique(self, image_inputs, instruction, response_to_critique, context=None):
        """
        Generate critique for given response
        
        Args:
            image_inputs: Visual input
            instruction: Original instruction
            response_to_critique: Response to evaluate
            context: Additional context
            
        Returns:
            dict: Critique analysis
        """
        if not self.enabled:
            return {
                'critique': '[Critic disabled]',
                'generated_token_ids': None,
                'scores': None
            }
        
        critique_prompt = self.critique_template.replace('{response}', response_to_critique)
        critique_prompt = critique_prompt.replace('{instruction}', instruction)
        
        result = self.base_model.generate_response(image_inputs, critique_prompt)
        return result

class ConflictDetector:
    """
    Conflict detection and debate triggering mechanism.
    Implements dual-criterion strategy for debate initiation.
    """
    
    def __init__(self, config=None):
        self.config = config or {}
        self.uncertainty_threshold = self.config.get('uncertainty_threshold', 0.6)
        self.conflict_threshold = self.config.get('conflict_threshold', 0.5)
        self.logger = logging.getLogger(__name__)
    
    def compute_system_uncertainty(self, agent_uncertainties, agent_weights):
        """
        Compute weighted average uncertainty U_sys^(0)
        
        Args:
            agent_uncertainties: List of agent uncertainty scores
            agent_weights: List of agent weights
            
        Returns:
            float: System uncertainty score
        """
        if not agent_uncertainties or not agent_weights:
            return 1.0
        
        weighted_uncertainty = sum(w * u for w, u in zip(agent_weights, agent_uncertainties))
        total_weight = sum(agent_weights)
        
        return weighted_uncertainty / total_weight if total_weight > 0 else 1.0
    
    def compute_conflict_score(self, agent_responses, parsed_claims_list):
        """
        Compute inter-expert conflict score based on claim consistency
        
        Args:
            agent_responses: List of agent responses
            parsed_claims_list: List of parsed claims from each agent
            
        Returns:
            float: Conflict score
        """
        # Simplified conflict detection based on claim diversity
        if len(parsed_claims_list) < 2:
            return 0.0
        
        # Analyze claim consistency across agents
        all_claims = []
        for claims in parsed_claims_list:
            for claim in claims:
                all_claims.append(claim['claim'].lower())
        
        # Simple heuristic: conflict based on claim overlap
        unique_claims = len(set(all_claims))
        total_claims = len(all_claims)
        
        if total_claims == 0:
            return 0.0
        
        # Higher diversity suggests more conflict
        diversity_ratio = unique_claims / total_claims
        conflict_score = min(1.0, diversity_ratio * 1.5)  # Scale factor
        
        return conflict_score
    
    def should_trigger_debate(self, system_uncertainty, conflict_score):
        """
        Decide whether to trigger debate based on dual criteria
        
        Args:
            system_uncertainty: System uncertainty score
            conflict_score: Inter-expert conflict score
            
        Returns:
            bool: Whether to trigger debate
        """
        uncertainty_trigger = system_uncertainty > self.uncertainty_threshold
        conflict_trigger = conflict_score > self.conflict_threshold
        
        should_debate = uncertainty_trigger or conflict_trigger
        
        self.logger.info(f"Debate decision: U_sys={system_uncertainty:.3f} (>{self.uncertainty_threshold}), "
                        f"Conflict={conflict_score:.3f} (>{self.conflict_threshold}) → Debate: {should_debate}")
        
        return should_debate

class DebateController:
    """
    Iterative Debate Process controller implementing the debate dynamics D.
    Manages dispute focus, expert argumentation, and convergence.
    """
    
    def __init__(self, config=None):
        self.config = config or {}
        self.max_rounds = self.config.get('max_rounds', 3)
        self.convergence_threshold = self.config.get('convergence_threshold', 0.01)
        self.uncertainty_threshold = self.config.get('uncertainty_threshold', 0.3)
        self.logger = logging.getLogger(__name__)
    
    def identify_disputes(self, parsed_claims_list, previous_response, system_uncertainty):
        """
        Identify key points of contention for focused debate
        
        Args:
            parsed_claims_list: Claims from all agents
            previous_response: Previous integrated response
            system_uncertainty: Current system uncertainty
            
        Returns:
            list: Key dispute points
        """
        disputes = []
        
        # Identify low-confidence claims
        for agent_claims in parsed_claims_list:
            for claim in agent_claims:
                if claim.get('confidence', 0.5) < 0.6:  # Low confidence threshold
                    disputes.append({
                        'type': 'low_confidence',
                        'claim': claim['claim'],
                        'confidence': claim.get('confidence', 0.5),
                        'evidence': claim.get('evidence', ''),
                        'region': claim.get('region', '')
                    })
        
        # Sort by confidence (lowest first) and take top disputes
        disputes.sort(key=lambda x: x['confidence'])
        return disputes[:3]  # Focus on top 3 disputes
    
    def generate_argumentation(self, agent, image_inputs, instruction, previous_response, 
                             dispute_focus, agent_responses):
        """
        Generate targeted argumentation for specific disputes
        
        Args:
            agent: Base agent to generate argument
            image_inputs: Visual input
            instruction: Original instruction
            previous_response: Previous round response
            dispute_focus: Key dispute points
            agent_responses: All agent responses for context
            
        Returns:
            dict: Argumentation package
        """
        # Construct focused argumentation prompt
        prompt_parts = [
            f"Original Question: {instruction}",
            f"Previous Integrated Answer: {previous_response}",
            "\nKey Disputes to Address:"
        ]
        
        for i, dispute in enumerate(dispute_focus):
            prompt_parts.append(
                f"Dispute {i+1}: {dispute['claim']} "
                f"(Confidence: {dispute['confidence']:.2f})"
            )
        
        prompt_parts.extend([
            f"\nAs a {agent.display_name}, please:",
            "1. Provide your expert analysis focusing on the disputed points",
            "2. Give specific evidence from the image(s) that supports your position",
            "3. Address any inconsistencies you identify",
            "4. Suggest improvements to the overall analysis",
            "\nFocused Expert Argumentation:"
        ])
        
        argumentation_prompt = "\n".join(prompt_parts)
        
        return agent.analyze(image_inputs, argumentation_prompt)
    
    def update_agent_weights(self, agents, uncertainties, contributions=None, round_num=0):
        """
        Dynamic weight adjustment based on uncertainty and contributions
        
        Args:
            agents: List of agents
            uncertainties: Current uncertainty scores
            contributions: Optional contribution scores
            round_num: Current round number
            
        Returns:
            list: Updated weights
        """
        if not uncertainties:
            return [1.0 / len(agents)] * len(agents)
        
        # Weight update based on uncertainty (lower uncertainty = higher weight)
        beta = self.config.get('weight_sensitivity', 1.0)
        raw_weights = [math.exp(-beta * u) for u in uncertainties]
        
        # Normalize weights
        total_weight = sum(raw_weights)
        normalized_weights = [w / total_weight for w in raw_weights] if total_weight > 0 else [1.0 / len(agents)] * len(agents)
        
        return normalized_weights
    
    def integrate_round_responses(self, image_inputs, instruction, previous_response, 
                                agent_responses, argumentation_results, agent_weights, 
                                dispute_focus):
        """
        Integrate expert views and argumentation for current round
        
        Args:
            image_inputs: Visual input
            instruction: Original instruction
            previous_response: Previous round response
            agent_responses: Initial agent responses
            argumentation_results: Current round argumentations
            agent_weights: Dynamic agent weights
            dispute_focus: Current dispute points
            
        Returns:
            dict: Integrated response
        """
        integration_parts = [
            f"Original Question: {instruction}",
            f"Previous Round Response: {previous_response}",
            "\nExpert Argumentations for Current Round:"
        ]
        
        for i, (agent_resp, arg_result, weight) in enumerate(zip(agent_responses, argumentation_results, agent_weights)):
            if arg_result and arg_result.get('response') != '[Agent disabled]':
                integration_parts.append(
                    f"--- Expert {i+1} (Weight: {weight:.3f}) ---\n"
                    f"Argumentation: {arg_result['response']}\n"
                )
        
        integration_parts.extend([
            "\nKey Disputes Being Addressed:",
            "\n".join([f"- {d['claim']}" for d in dispute_focus]),
            "\nInstructions for Integration:",
            "1. Focus on resolving the identified disputes while preserving consensus",
            "2. Weigh expert arguments according to their uncertainty and weights",
            "3. Provide a refined, coherent response that addresses the original question",
            "4. Maintain accuracy based on visual evidence",
            "\nIntegrated Response:"
        ])
        
        integration_prompt = "\n".join(integration_parts)
        
        # Use base model for integration (could be specialized integration model)
        from .SUPPVERSION import AgentModel  # Import to access base model
        # This is a simplified approach - in practice, would use the same base model
        # For now, return a placeholder that would be filled by actual integration
        return {
            'response': f"[Integrated response for round addressing: {', '.join([d['claim'][:50] for d in dispute_focus])}]",
            'generated_token_ids': None,
            'scores': None
        }
    
    def check_termination(self, round_num, system_uncertainty, prev_uncertainty):
        """
        Check debate termination conditions
        
        Args:
            round_num: Current round number
            system_uncertainty: Current system uncertainty
            prev_uncertainty: Previous round uncertainty
            
        Returns:
            tuple: (should_terminate, reason)
        """
        # Condition 1: Uncertainty threshold
        if system_uncertainty < self.uncertainty_threshold:
            return True, "uncertainty_threshold"
        
        # Condition 2: Maximum rounds
        if round_num >= self.max_rounds:
            return True, "max_rounds"
        
        # Condition 3: Convergence (minimal uncertainty change)
        if prev_uncertainty is not None:
            uncertainty_change = abs(system_uncertainty - prev_uncertainty)
            if uncertainty_change < self.convergence_threshold:
                return True, "convergence"
        
        return False, "continue"

class GAMAgent:
    """
    GAM-Agent implementing the six-tuple S = (E, A, Φ, M, P, D).
    
    E: Set of expert agents (Base + Critical)
    A: Analysis capability mapping functions
    Φ: Uncertainty assessment functions  
    M: Evidence localization mapping functions
    P: Claim and evidence parser
    D: Debate module for conflict resolution
    """
    
    def __init__(self, model_config, experts_config, api_config=None):
        self.model_config = model_config
        self.experts_config = experts_config
        self.device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
        self.logger = logging.getLogger(__name__)
        
        # Initialize base VLM
        self.base_model = self._initialize_base_model()
        
        # Initialize GAM-Agent components
        self.uncertainty_quantifier = UncertaintyQuantifier(model_config.get('uncertainty_config', {}))
        self.claim_parser = ClaimParser(model_config.get('parser_config', {}))
        self.evidence_mapper = EvidenceMapper(model_config.get('mapper_config', {}))
        self.conflict_detector = ConflictDetector(model_config.get('conflict_config', {}))
        self.debate_controller = DebateController(model_config.get('debate_config', {}))
        
        # Load expert configurations
        self.load_expert_configurations()
        
        # Initialize agent sets E
        self.base_agents = []  # Specialized visual reasoning experts
        self.critical_agents = []  # Specialized reasoning critique experts
        self._initialize_agents()
    
    def _initialize_base_model(self):
        """Initialize the underlying VLM"""
        from .SUPPVERSION import AgentModel  # Import the base model class
        return AgentModel(self.model_config)
    
    def load_expert_configurations(self):
        """Load expert configurations from file or use defaults"""
        config_file_path = self.experts_config.get('config_file')
        
        if config_file_path and os.path.exists(config_file_path):
            try:
                with open(config_file_path, 'r', encoding='utf-8') as f:
                    experts_data = yaml.safe_load(f)
                
                self.expert_configs = experts_data.get('experts', [])
                self.critic_configs = experts_data.get('critic_experts', [])
                self.debate_settings = experts_data.get('debate_settings', {})
                
                self.logger.info(f"Loaded {len(self.expert_configs)} base agents and "
                               f"{len(self.critic_configs)} critical agents from config file")
                
            except Exception as e:
                self.logger.error(f"Error loading expert config: {e}. Using defaults.")
                self._load_default_configurations()
        else:
            self.logger.info("Using default expert configurations")
            self._load_default_configurations()
    
    def _load_default_configurations(self):
        """Load default expert configurations suitable for visual reasoning"""
        self.expert_configs = [
            {
                'id': 0, 'name': "object_recognition", 'display_name': "Object Recognition Expert", 
                'weight': 1.0, 'enabled': True,
                'keywords': ["what is", "identify", "object", "item", "recognize"],
                'prompt_template': "As an object recognition expert, identify and analyze all significant objects in the image(s). Provide details about their appearance, location, and relationships. Original question: {instruction}"
            },
            {
                'id': 1, 'name': "scene_description", 'display_name': "Scene Description Expert", 
                'weight': 1.0, 'enabled': True,
                'keywords': ["describe", "scene", "environment", "background", "setting"],
                'prompt_template': "As a scene description expert, analyze the overall scene, environment, lighting, and spatial relationships in the image(s). Original question: {instruction}"
            },
            {
                'id': 2, 'name': "text_ocr", 'display_name': "Text/OCR Expert", 
                'weight': 1.0, 'enabled': True,
                'keywords': ["text", "read", "sign", "label", "document", "ocr"],
                'prompt_template': "As an OCR expert, identify and transcribe any text visible in the image(s). Pay attention to signs, labels, documents, or any written content. Original question: {instruction}"
            }
        ]
        
        self.critic_configs = [
            {
                'id': 0, 'name': "fact_checker", 'display_name': "Fact Checker", 'enabled': True,
                'critique_template': "As a fact-checking expert, evaluate the accuracy of this analysis based on visual evidence: {response}. Identify any inaccuracies and suggest corrections. Original question: {instruction}"
            },
            {
                'id': 1, 'name': "completeness_checker", 'display_name': "Completeness Checker", 'enabled': True,
                'critique_template': "As a completeness expert, assess if this analysis fully addresses the question: {response}. Identify missing information and suggest improvements. Original question: {instruction}"
            }
        ]
        
        self.debate_settings = {
            'use_debate': True,
            'max_rounds': 3,
            'uncertainty_threshold': 0.3,
            'conflict_threshold': 0.5
        }
    
    def _initialize_agents(self):
        """Initialize base and critical agent instances"""
        # Initialize base agents (E_base)
        for config in self.expert_configs:
            if config.get('enabled', True):
                agent = BaseAgent(config, self.base_model)
                self.base_agents.append(agent)
        
        # Initialize critical agents (E_critical)  
        for config in self.critic_configs:
            if config.get('enabled', True):
                agent = CriticalAgent(config, self.base_model)
                self.critical_agents.append(agent)
        
        self.logger.info(f"Initialized {len(self.base_agents)} base agents and "
                        f"{len(self.critical_agents)} critical agents")
    
    def compute_initial_weights(self, uncertainties):
        """
        Compute initial weights w_i^(0) based on uncertainty assessments
        
        Args:
            uncertainties: List of uncertainty scores
            
        Returns:
            list: Initial weight allocation
        """
        if not uncertainties:
            return []
        
        beta = self.model_config.get('weight_sensitivity', 1.0)
        
        # w_i^(0) = exp(-β * U_i) / Σ exp(-β * U_j)
        exp_weights = [math.exp(-beta * u) for u in uncertainties]
        total_weight = sum(exp_weights)
        
        normalized_weights = [w / total_weight for w in exp_weights] if total_weight > 0 else [1.0 / len(uncertainties)] * len(uncertainties)
        
        return normalized_weights
    
    def integrate_initial_responses(self, image_inputs, instruction, agent_responses, 
                                  agent_weights, options=None):
        """
        Initial integration using IntegrateJudge function
        
        Args:
            image_inputs: Visual input
            instruction: Original instruction  
            agent_responses: Initial agent responses
            agent_weights: Agent weights
            options: Optional answer choices
            
        Returns:
            dict: Integrated response
        """
        integration_parts = [
            "You are an expert judge integrating analyses from multiple visual reasoning specialists.",
            f"Original Question: {instruction}"
        ]
        
        if options:
            options_str = ', '.join(map(str, options)) if isinstance(options, list) else str(options)
            integration_parts.append(f"Available Options: {options_str}")
        
        integration_parts.append("\nExpert Analyses:")
        
        for i, (response, weight) in enumerate(zip(agent_responses, agent_weights)):
            if response.get('response') != '[Agent disabled]':
                agent_name = self.base_agents[i].display_name
                integration_parts.append(
                    f"--- {agent_name} (Weight: {weight:.3f}) ---\n"
                    f"{response['response']}\n"
                )
        
        integration_parts.extend([
            "\nInstructions for Integration:",
            "1. Synthesize insights from all experts with higher weight given to more certain analyses",
            "2. Resolve any contradictions by weighing evidence quality",
            "3. Provide a comprehensive answer addressing the original question",
            "4. If options provided, select the most appropriate option clearly",
            "\nIntegrated Analysis:"
        ])
        
        integration_prompt = "\n".join(integration_parts)
        
        return self.base_model.generate_response(image_inputs, integration_prompt)
    
    def process_visual_query(self, image_inputs, instruction, options=None, reference_output=None):
        """
        Main processing pipeline implementing the GAM-Agent methodology
        
        Args:
            image_inputs: List of PIL Images
            instruction: User question
            options: Optional answer choices
            reference_output: Optional reference for evaluation
            
        Returns:
            dict: Comprehensive processing results
        """
        if not image_inputs:
            self.logger.error("No image input provided")
            return {
                "error": "No image input provided",
                "final_response": "[Error: No image input]",
                "agent_uncertainties": [],
                "agent_responses": [],
                "initial_response": "[Error: No image input]"
            }
        
        start_time = time.time()
        self.logger.info(f"Processing visual query with {len(image_inputs)} images")
        
        # Phase 1: Initial Expert Analysis (A_i: X × P_r → R_i)
        self.logger.info("Phase 1: Initial expert analysis")
        agent_responses = []
        agent_uncertainties = []
        parsed_claims_list = []
        
        for i, agent in enumerate(self.base_agents):
            # Generate response using analysis capability A_i
            response_result = agent.analyze(image_inputs, instruction, options)
            agent_responses.append(response_result)
            
            # Quantify uncertainty using Φ_i
            uncertainty = self.uncertainty_quantifier.quantify_uncertainty(
                response_result.get('response', ''),
                response_result.get('generated_token_ids'),
                response_result.get('scores')
            )
            agent_uncertainties.append(uncertainty)
            
            # Parse claims using P
            parsed_claims = self.claim_parser.parse_response(response_result.get('response', ''))
            
            # Enhance with evidence mapping using M
            enhanced_claims = self.evidence_mapper.enhance_claims_with_regions(parsed_claims)
            parsed_claims_list.append(enhanced_claims)
            
            self.logger.info(f"Agent {agent.display_name}: uncertainty={uncertainty:.3f}, "
                           f"claims={len(parsed_claims)}")
        
        # Phase 2: Initial Weight Allocation and Integration
        self.logger.info("Phase 2: Initial integration and conflict detection")
        
        # Compute initial weights w_i^(0)
        initial_weights = self.compute_initial_weights(agent_uncertainties)
        
        # Generate initial integrated response R^(0)
        initial_integration = self.integrate_initial_responses(
            image_inputs, instruction, agent_responses, initial_weights, options
        )
        initial_response = initial_integration.get('response', '')
        
        # Compute system uncertainty U_sys^(0)
        system_uncertainty = self.conflict_detector.compute_system_uncertainty(
            agent_uncertainties, initial_weights
        )
        
        # Compute conflict score
        conflict_score = self.conflict_detector.compute_conflict_score(
            agent_responses, parsed_claims_list
        )
        
        # Phase 3: Conflict Detection and Debate Triggering
        should_debate = self.conflict_detector.should_trigger_debate(
            system_uncertainty, conflict_score
        )
        
        result = {
            "agent_uncertainties": agent_uncertainties,
            "agent_responses": agent_responses,
            "initial_response": initial_response,
            "system_uncertainty": system_uncertainty,
            "conflict_score": conflict_score,
            "parsed_claims": parsed_claims_list,
            "should_debate": should_debate
        }
        
        # Phase 4: Iterative Debate Process (if triggered)
        if should_debate and self.debate_settings.get('use_debate', True):
            self.logger.info("Phase 4: Iterative debate process")
            final_response = self._run_debate_process(
                image_inputs, instruction, options, initial_response, 
                agent_responses, parsed_claims_list, initial_weights, system_uncertainty
            )
            result["final_response"] = final_response
            result["debate_executed"] = True
        else:
            result["final_response"] = initial_response
            result["debate_executed"] = False
            self.logger.info("No debate triggered - using initial integration")
        
        # Compute task complexity
        result["task_complexity"] = system_uncertainty
        
        total_time = time.time() - start_time
        self.logger.info(f"Processing completed in {total_time:.2f}s")
        
        return result
    
    def _run_debate_process(self, image_inputs, instruction, options, initial_response,
                          agent_responses, parsed_claims_list, initial_weights, initial_uncertainty):
        """
        Execute iterative debate process implementing dynamics D
        
        Args:
            image_inputs: Visual input
            instruction: Original instruction
            options: Answer options
            initial_response: Initial integrated response
            agent_responses: Initial agent responses  
            parsed_claims_list: Parsed claims from agents
            initial_weights: Initial agent weights
            initial_uncertainty: Initial system uncertainty
            
        Returns:
            str: Final debated response
        """
        current_response = initial_response
        current_uncertainty = initial_uncertainty
        current_weights = initial_weights.copy()
        
        for round_num in range(1, self.debate_controller.max_rounds + 1):
            self.logger.info(f"--- Debate Round {round_num} ---")
            
            # Step 1: Identify disputes C_debate^(k)
            dispute_focus = self.debate_controller.identify_disputes(
                parsed_claims_list, current_response, current_uncertainty
            )
            
            if not dispute_focus:
                self.logger.info("No significant disputes identified - terminating debate")
                break
            
            # Step 2: Generate expert argumentations Arg_i^(k)
            argumentation_results = []
            new_uncertainties = []
            
            for agent in self.base_agents:
                arg_result = self.debate_controller.generate_argumentation(
                    agent, image_inputs, instruction, current_response, 
                    dispute_focus, agent_responses
                )
                argumentation_results.append(arg_result)
                
                # Update uncertainty for this round
                arg_uncertainty = self.uncertainty_quantifier.quantify_uncertainty(
                    arg_result.get('response', ''),
                    arg_result.get('generated_token_ids'),
                    arg_result.get('scores')
                )
                new_uncertainties.append(arg_uncertainty)
            
            # Step 3: Update weights w_i^(k)
            current_weights = self.debate_controller.update_agent_weights(
                self.base_agents, new_uncertainties, round_num=round_num
            )
            
            # Step 4: Generate integrated response R^(k)
            round_integration = self.debate_controller.integrate_round_responses(
                image_inputs, instruction, current_response, agent_responses,
                argumentation_results, current_weights, dispute_focus
            )
            
            # For now, use a simplified integration approach
            # In full implementation, this would call the actual integration model
            current_response = round_integration.get('response', current_response)
            
            # Step 5: Update system uncertainty U_sys^(k)
            prev_uncertainty = current_uncertainty
            current_uncertainty = self.conflict_detector.compute_system_uncertainty(
                new_uncertainties, current_weights
            )
            
            # Check termination conditions
            should_terminate, reason = self.debate_controller.check_termination(
                round_num, current_uncertainty, prev_uncertainty
            )
            
            self.logger.info(f"Round {round_num}: uncertainty={current_uncertainty:.3f}, "
                           f"disputes={len(dispute_focus)}, terminate={should_terminate} ({reason})")
            
            if should_terminate:
                break
        
        return current_response

# Maintain backward compatibility with the original AgentModel class
class AgentModel:
    """
    Legacy AgentModel class for backward compatibility.
    Local image-language agent model base class.
    """
    
    def __init__(self, config):
        """
        Initialize local agent model
        
        Args:
            config: Model configuration (includes model_path, generation_config, etc.)
        """
        self.config = config
        self.device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
        self.logger = logging.getLogger(__name__)

        # Load local model path from config
        self.model_path = config.get("model_path", "OpenGVLab/InternVL3-14B")

        # Load local model and processor
        self.logger.info(f"Loading local model and processor: {self.model_path} to device {self.device}")
        try:
            # Determine dtype based on device capability
            model_dtype = torch.bfloat16 if torch.cuda.is_available() and torch.cuda.is_bf16_supported() else torch.float16

            self.processor = AutoProcessor.from_pretrained(self.model_path, trust_remote_code=True)
            self.model = AutoModel.from_pretrained(
                self.model_path,
                torch_dtype=model_dtype,
                low_cpu_mem_usage=True,
                use_flash_attn=True,
                trust_remote_code=True
            ).eval().to(self.device)

            # Tokenizer initialization logic
            raw_tokenizer = None
            if hasattr(self.processor, 'tokenizer'):
                 raw_tokenizer = self.processor.tokenizer
                 self.logger.info(f"Retrieved raw tokenizer from processor.tokenizer (type: {type(raw_tokenizer)})")
            elif hasattr(self.processor, '_tokenizer'):
                 self.logger.warning("Accessing tokenizer via internal attribute _tokenizer")
                 raw_tokenizer = self.processor._tokenizer
                 self.logger.info(f"Retrieved raw tokenizer from processor._tokenizer (type: {type(raw_tokenizer)})")

            if raw_tokenizer:
                if isinstance(raw_tokenizer, Tokenizer) and not isinstance(raw_tokenizer, PreTrainedTokenizerFast):
                    self.logger.warning(f"Raw tokenizer (type: {type(raw_tokenizer)}) is not PreTrainedTokenizerFast, attempting to wrap.")
                    try:
                        kwargs_for_fast = {"tokenizer_object": raw_tokenizer}
                        for token_attr in ['bos_token', 'eos_token', 'unk_token', 'sep_token', 'pad_token', 'cls_token', 'mask_token']:
                             token_val = getattr(raw_tokenizer, token_attr, None)
                             if token_val:
                                 kwargs_for_fast[token_attr] = token_val
                                 self.logger.info(f"Passing {token_attr}='{token_val}' to PreTrainedTokenizerFast")

                        self.tokenizer = PreTrainedTokenizerFast(**kwargs_for_fast)
                        self.logger.info(f"Successfully wrapped raw tokenizer as PreTrainedTokenizerFast (type: {type(self.tokenizer)})")
                    except Exception as wrap_err:
                        self.logger.error(f"Failed to wrap raw tokenizer: {wrap_err}. Falling back to AutoTokenizer loading.")
                        raw_tokenizer = None

                elif isinstance(raw_tokenizer, (PreTrainedTokenizerFast, AutoTokenizer)):
                    self.logger.info(f"Processor-provided tokenizer (type: {type(raw_tokenizer)}) is directly usable.")
                    self.tokenizer = raw_tokenizer
                else:
                    if callable(raw_tokenizer):
                        self.logger.warning(f"Processor-provided tokenizer (type: {type(raw_tokenizer)}) is callable but not Fast version. Using directly.")
                        self.tokenizer = raw_tokenizer
                    else:
                        self.logger.error(f"Processor tokenizer is unexpected non-callable type: {type(raw_tokenizer)}. Falling back to AutoTokenizer loading.")
                        raw_tokenizer = None

            # Fallback if processor had no tokenizer or wrapping failed
            if not raw_tokenizer and not hasattr(self, 'tokenizer'):
                 self.logger.warning("Unable to get or wrap tokenizer from processor, attempting direct AutoTokenizer loading")
                 try:
                     self.tokenizer = AutoTokenizer.from_pretrained(self.model_path, trust_remote_code=True)
                     self.logger.info(f"Successfully loaded tokenizer via AutoTokenizer (type: {type(self.tokenizer)})")
                 except Exception as auto_tok_err:
                     self.logger.error(f"Loading tokenizer via AutoTokenizer also failed: {auto_tok_err}")
                     raise RuntimeError(f"Unable to initialize tokenizer for {self.model_path}") from auto_tok_err

            if not hasattr(self, 'tokenizer') or self.tokenizer is None:
                 raise RuntimeError(f"Tokenizer not successfully set at end of initialization for {self.model_path}")

            # Ensure model config is consistent with the final tokenizer's pad_token_id
            tokenizer_pad_id = getattr(self.tokenizer, 'pad_token_id', None)
            model_config_pad_id = getattr(self.model.config, 'pad_token_id', None)

            if tokenizer_pad_id is not None:
                if model_config_pad_id is None:
                     self.logger.warning(f"Model config lacks pad_token_id. Adding from Tokenizer ({tokenizer_pad_id}).")
                     self.model.config.pad_token_id = tokenizer_pad_id
                elif model_config_pad_id != tokenizer_pad_id:
                     self.logger.warning(f"Model config pad_token_id ({model_config_pad_id}) inconsistent with Tokenizer ({tokenizer_pad_id}). Updating model config.")
                     self.model.config.pad_token_id = tokenizer_pad_id
            elif model_config_pad_id is not None:
                 self.logger.warning(f"Tokenizer lacks pad_token_id but model config has one ({model_config_pad_id}). Keeping unchanged, but may cause issues.")

            self.logger.info(f"Model {self.model_path} loaded successfully, using data type {model_dtype}.")

        except Exception as e:
            self.logger.exception(f"Failed to load local model {self.model_path}: {e}")
            raise e

        # Define generation config (can be overridden by config)
        self.generation_config = config.get("generation_config", dict(
            do_sample=False,
            temperature=0.7,
            max_new_tokens=1024,
            num_beams=1
        ))
        self.logger.info(f"Using generation config: {self.generation_config}")

    def estimate_uncertainty(self, response, generated_token_ids=None, scores=None):
        """
        Quantify agent response uncertainty (prioritizing scores)
        
        Args:
            response: Agent generated text response
            generated_token_ids: Optional, model generated token ID list
            scores: Optional, scores from model generate method (logits)
            
        Returns:
            float: Uncertainty score (0-1, 1 indicates most uncertain)         
        """
        # Use the new uncertainty quantifier
        uncertainty_quantifier = UncertaintyQuantifier()
        return uncertainty_quantifier.quantify_uncertainty(response, generated_token_ids, scores)

    def generate_response(self, image_inputs: list[Image.Image], prompt: str):
        """
        Generate response using local model (handling image inputs)

        Args:
            image_inputs: List of PIL Image objects
            prompt: Prompt text

        Returns:
            dict: Dictionary containing generated text and probability information
                  {'text': str, 'generated_token_ids': torch.Tensor | None, 'scores': tuple[torch.Tensor] | None}
        """
        if not isinstance(image_inputs, list) or not all(isinstance(img, Image.Image) for img in image_inputs):
            error_msg = f"generate_response requires list of PIL Images, but received {type(image_inputs)}"
            self.logger.error(error_msg)
            return {"text": f"Processing error: {error_msg}", "generated_token_ids": None, "scores": None}

        # Format prompt for multi-image case
        if len(image_inputs) > 1:
             image_placeholders = ''.join([f"Image-{i+1}: <image>\n" for i in range(len(image_inputs))])
             formatted_prompt = f"{image_placeholders}{prompt}"
             self.logger.info(f"Using local model to generate response, multi-image ({len(image_inputs)} images), prompt: '{prompt[:100]}...'" )
        elif len(image_inputs) == 1:
             formatted_prompt = f"<image>\n{prompt}"
             self.logger.info(f"Using local model to generate response, single image, prompt: '{prompt[:100]}...'" )
        else:
             self.logger.info(f"Using local model to generate response, no image (pure text), prompt: '{prompt[:100]}...'" )

        # Modified Image Processing using dynamic_preprocess
        pixel_values_list = []
        num_patches_list = []
        
        vision_config = getattr(self.model.config, 'vision_config', None)
        if vision_config and hasattr(vision_config, 'image_size'):
            image_input_size = vision_config.image_size
            self.logger.info(f"Got image_size from model.config.vision_config: {image_input_size}")
        else:
            image_input_size = 448
            self.logger.warning(f"Unable to get image_size from model.config.vision_config, fallback to default: {image_input_size}")

        transform = build_transform(input_size=image_input_size)
        max_tiles = 6
        use_thumbnail = True

        for img in image_inputs:
            try:
                image_tiles = dynamic_preprocess(img, image_size=image_input_size, use_thumbnail=use_thumbnail, max_num=max_tiles)
                processed_tiles = [transform(tile) for tile in image_tiles]
                current_pixel_values = torch.stack(processed_tiles)

                pixel_values_list.append(current_pixel_values)
                num_patches_list.append(current_pixel_values.shape[0])
            except Exception as e:
                self.logger.error(f"Error processing individual image with dynamic_preprocess: {e}")
                return {"text": f"Image processing error: {e}", "generated_token_ids": None, "scores": None}

        if not pixel_values_list:
            self.logger.error("All image processing failed.")
            return {"text": "All image processing failed.", "generated_token_ids": None, "scores": None}

        pixel_values = torch.cat(pixel_values_list, dim=0)
        pixel_values = pixel_values.to(self.device).to(self.model.dtype)

        # Call chat interface based on number of images
        if len(image_inputs) > 1:
            response = self.model.chat(
                self.tokenizer,
                pixel_values,
                formatted_prompt,
                self.generation_config,
                num_patches_list=num_patches_list
            )
        else:
            response = self.model.chat(
                self.tokenizer,
                pixel_values,
                formatted_prompt,
                self.generation_config
            )

        return {"text": response.strip(), "generated_token_ids": None, "scores": None}

class ExpertAgentModel:
    """Expert agent model managing multiple specialized agent models (local image mode)"""
    
    def __init__(self, model_config, experts_config, api_config=None):
        """
        Initialize expert agent model

        Args:
            model_config: Local model configuration (includes model_path etc.)
            experts_config: Expert configuration
            api_config: API configuration (ignored)
        """
        # Initialize GAM-Agent with the same configuration
        self.gam_agent = GAMAgent(model_config, experts_config, api_config)
        self.logger = logging.getLogger(__name__)

    def process_with_experts(self, image_inputs: list[Image.Image], instruction: str, output=None, options=None, choice_answer=None):
        """
        Process with experts using GAM-Agent methodology
        
        Args:
            image_inputs: List of PIL Image objects
            instruction: Instruction
            output: Reference output (for training/evaluation)
            options: Options (for multiple choice)
            choice_answer: Reference answer for multiple choice (for training/evaluation)
            
        Returns:
            dict: Processing results
        """
        # Use GAM-Agent's main processing pipeline
        result = self.gam_agent.process_visual_query(
            image_inputs=image_inputs,
            instruction=instruction,
            options=options,
            reference_output=output
        )
        
        # Transform result to match expected format for backward compatibility
        return {
            "final_response": result.get("final_response", ""),
            "agent_uncertainties": result.get("agent_uncertainties", []),
            "agent_responses": result.get("agent_responses", []),
            "initial_response": result.get("initial_response", ""),
            "all_expert_raw_results": result.get("agent_responses", []),
            "task_complexity": result.get("task_complexity", 1.0),
            "system_uncertainty": result.get("system_uncertainty", 1.0),
            "conflict_score": result.get("conflict_score", 0.0),
            "debate_executed": result.get("debate_executed", False)
        }