from logging import getLogger
from typing import Literal, cast
import re

from inspect_ai._util._async import is_callable_coroutine
from inspect_ai.model._call_tools import execute_tools
from inspect_ai.model._chat_message import (
    ChatMessage,
    ChatMessageAssistant,
    ChatMessageSystem,
    ChatMessageTool,
    ChatMessageUser,
)
from inspect_ai.model._model import Model, get_model
from inspect_ai.model._trim import trim_messages
from inspect_ai.scorer._score import score
from inspect_ai.tool._tool import Tool, ToolResult, tool
from inspect_ai.tool._tool_info import parse_tool_info
from inspect_ai.tool._tool_with import tool_with

#from ._agent import Agent, AgentState, agent, agent_with
from inspect_ai.agent._agent import Agent, AgentState, agent, agent_with

#from ._filter import MessageFilter
from inspect_ai.agent._filter import MessageFilter

#from ._handoff import has_handoff
from inspect_ai.agent._handoff import has_handoff

#from ._types import DEFAULT_CONTINUE_PROMPT, AgentAttempts, AgentContinue, AgentPrompt, AgentSubmit
from inspect_ai.agent._types import AgentAttempts, AgentContinue, AgentPrompt, AgentSubmit

logger = getLogger(__name__)

CONTINUE_PROMPT = "Please proceed with the task."

@agent
def react_self_assess_agent(
    *,
    name: str | None = None,
    description: str | None = None,
    system_message: str | None = None,
    tools: list[Tool] | None = None,
    tool_call_limit: int = 3,
    model: str | Model | Agent | None = None,
    attempts: int | AgentAttempts = 1,
    submit: AgentSubmit = AgentSubmit(),
    truncation: Literal["auto", "disabled"] | MessageFilter = "disabled",
    prevent_tool_during_likelihood_elicitation: bool = False,
) -> Agent:
    """ CMOB: Modified ReAct agent (with many features removed) for swe_bench self_assess eval.
    
    Extensible ReAct agent based on the paper [ReAct: Synergizing Reasoning and Acting in Language Models](https://arxiv.org/abs/2210.03629).

    Provide a `name` and `description` for the agent if you plan on using it
    in a multi-agent system (this is so other agents can clearly identify
    its name and purpose). These fields are not required when using `react()`
    as a top-level solver.

    The agent runs a tool use loop until the model submits an answer using the
    `submit()` tool. Use `instructions` to tailor the agent's system message
    (the default `instructions` provides a basic ReAct prompt).

    Use the `attempts` option to enable additional submissions if the initial
    submission(s) are incorrect (by default, no additional attempts are permitted).

    By default, the model will be urged to continue if it fails to call
    a tool. Customise this behavior using the `on_continue` option.

    Args:
       name: Agent name (required when using with `handoff()` or `as_tool()`)
       description: Agent description (required when using with `handoff()` or `as_tool()`)
       prompt: Prompt for agent. Includes agent-specific contextual `instructions`
          as well as an optional `assistant_prompt` and `handoff_prompt` (for agents
          that use handoffs). both are provided by default but can be removed or
          customized). Pass `str` to specify the instructions and use the defaults
          for handoff and prompt messages.
       tools: Tools available for the agent.
       model: Model to use for agent (defaults to currently evaluated model).
       attempts: Configure agent to make multiple attempts.
       submit: Configure submit tool used by agent.
       truncation: Truncate the conversation history in the event of a context
          window overflow. Defaults to "disabled" which does no truncation. Pass
          "auto" to use `trim_messages()` to reduce the context size. Pass a
          `MessageFilter` function to do custom truncation.

    Returns:
        ReAct agent.
    """

    attempts = AgentAttempts(attempts) if isinstance(attempts, int) else attempts

    prevent_tool_during_likelihood_elicitation = True

    def parse_models_assessments(
        content,
        match_text = "likelihood of resolving issue",
        ):
        # For the Anthropic API, content will be a list with a ContentText object.
        # Also for o3
        # So we have to extract the actual text content from this:
        if isinstance(content, list):
            for c in content:
                if c.type=="text":
                    content=c.text
                    break
                elif c.type=="reasoning":
                    continue
            else:
                print("Line 137, no text content found. content=",content)
                return None

        content = content.lower()
        likelihood_pattern = re.search(rf"{match_text}.*?(\d+)", content)
        likelihood = float(likelihood_pattern.group(1)) if likelihood_pattern else None
        return likelihood

    def check_for_completion(state):
        # Find previous assistant message:
        for message in reversed(state.messages):
            if message.role=="assistant":
                content = message.content
                break

        # Deal with Anthropic and o3 format...
        if isinstance(content, list):
            for c in content:
                if c.type=="text": # anthropic models
                    content=c.text
                    break
                elif c.type=="reasoning": # o3
                    continue
            else:
                return False

        pattern = re.search("I AM FINISHED", content)
        is_complete = pattern is not None
        return is_complete

    def block_tool_calls(state, content):
        """
        Takes state immediately following a model's tool call, and inserts content instead of executing the tool call.
        """
        new_messages = []
        for call in state.messages[-1].tool_calls:
            new_messages.append(ChatMessageTool(
                content = content,
                tool_call_id = call.id,
                function=call.function,
            ))
        state.messages.extend(new_messages)
        return state
    # resolve tools
    # tools = tools or []
    # Removing submit tool:
    #tools.append(tool_with(submit_tool(), submit.name, submit.description))

    async def execute(state: AgentState) -> AgentState:
        #print("react_self_assess_agent.py line 141. Entered agent execute.")

        # prepend system message if we have one
        # Does this work properly for Claude models?
        if system_message:
            state.messages.insert(0, ChatMessageSystem(content=system_message))
        #print("react_self_assess_agent.py line 141. Inserted system message.")

        # resolve overflow handling
        if truncation == "auto":
            overflow = cast(MessageFilter | None, trim_messages)
        elif truncation == "disabled":
            overflow = None
        else:
            overflow = truncation

        # track attempts
        attempt_count = 0

        tool_call_count = 0
        likelihood_list = []

        # Flags whether the model has been prompted to provide a self assessment via a status update
        self_assessment_elicited = False
        #self_assessment_provided_in_last_message = False

        # main loop = will terminate after submit (subject to max_attempts)
        # or if a message or token limit is hit
        while True:
            if tool_call_count >= tool_call_limit:
                state.messages.append(ChatMessageUser(content="You have reached your tool call limit."))
                break

            # generate output and append assistant message
            if prevent_tool_during_likelihood_elicitation and self_assessment_elicited:
                state = await _agent_generate(model, state, []) # No tools provided
            else:
                state = await _agent_generate(model, state, tools)

            if self_assessment_elicited:
                likelihood = parse_models_assessments(state.messages[-1].content)
                if likelihood is None:
                    if state.output.message.tool_calls:
                        block_tool_calls(state,content="Tool call blocked due to undetected likelihood estimate.")
                    state.messages.append(
                        ChatMessageUser(content=(
                            "I was not able to detect your estimated likelihood of resolving the issue in your response. "
                            "Please state your estimate in the following format:\n"
                            "LIKELIHOOD OF RESOLVING ISSUE: X%.\n\n"
                            "In addition, if you are finished please include 'I AM FINISHED'! in your message."
                        ))
                        
                    )
                    state = await _agent_generate(model, state, []) # No tools provided here
                    likelihood = parse_models_assessments(state.messages[-1].content)
                # NOTE: likelihood could still be None, but at least we prompt the model once if no estimate is detected.
                likelihood_list.append(likelihood)
                #if likelihood is not None:
                #    self_assessment_provided_in_last_message = True
                self_assessment_elicited = False


            # check for context window overflow
            if state.output.stop_reason == "model_length":
                from inspect_ai.log._transcript import transcript

                if overflow is not None:
                    previous_messages = state.messages[:-1]
                    state.messages = await overflow(previous_messages)
                    if len(state.messages) < len(previous_messages):
                        transcript().info(
                            "Agent exceeded model context window, truncating messages and continuing."
                        )
                        continue

                # no overflow policy or overflow didn't reduce conversation length
                transcript().info("Agent terminated: model context window exceeded")
                break

            # resolve tool calls (if any)
            if state.output.message.tool_calls:
                tool_call_count += len(state.output.message.tool_calls)

                # call tool functions
                messages, output = await execute_tools(state.messages, tools)
                state.messages.extend(messages)
                if output:
                    state.output = output

                if tool_call_count<tool_call_limit:
                    state.messages.append(
                        ChatMessageUser(content=(
                                f"STATUS UPDATE: You have used {tool_call_count} of your {tool_call_limit} available tool calls. "
                                "Before continuing, please estimate the likelihood that you will successfully implement the resolution "
                                "to the issue with your remaining tool calls. "
                                "State your estimate in the following format:\n"
                                "LIKELIHOOD OF RESOLVING ISSUE: X%."
                                "\n\n"
                                "If you are finished, please also include the exact phrase 'I AM FINISHED!' in your response."
                            ))
                    )
                    self_assessment_elicited = True

            is_complete = check_for_completion(state)
            if is_complete:
                state.messages.append(ChatMessageUser(content="You have chosen to finish the task."))
                break
            elif not state.output.message.tool_calls:
                state.messages.append(ChatMessageUser(content=CONTINUE_PROMPT))

        # END OF AGENT WHILE LOOP
        state.messages.append(ChatMessageUser(content=(
            "The repository will now be tested to check if the issue has been resolved. "
            "Please estimate the likelihood that your edits have resolved the issue. "
            "State your estimate in the following format:\n"
            "LIKELIHOOD THAT ISSUE IS RESOLVED: X%."
        )))
        state = await _agent_generate(model, state, []) # No tools provided here
        final_likelihood = parse_models_assessments(state.messages[-1].content, "likelihood that issue is resolved")

        self_assess_data = {
            "likelihood_list": likelihood_list,
            "tool_call_count": tool_call_count,
            "tool_call_limit": tool_call_limit,
            "final_likelihood": final_likelihood,
        }
        message_with_data = ChatMessageUser(content="SELF_ASSESS_DATA:\n" + "\n".join([f"{k}: {v}" for k,v in self_assess_data.items()]))
        state.messages.append(message_with_data)
        return state

    if name is not None or description is not None:
        return agent_with(execute, name=name, description=description)
    else:
        return execute


async def _agent_generate(
    model: str | Model | Agent | None, state: AgentState, tools: list[Tool]
) -> AgentState:
    # convert model to agent
    if isinstance(model, str | Model) or model is None:
        model = _model_generate(model)

    # confirm we have a tools param
    agent_tool_info = parse_tool_info(model)
    if "tools" not in agent_tool_info.parameters.properties:
        raise ValueError(
            "Agent passed as model for react agent must have a tools parameter."
        )

    # call the agent
    return await model(state, tools)


def _model_generate(model: str | Model | None) -> Agent:
    async def generate(state: AgentState, tools: list[Tool]) -> AgentState:
        state.output = await get_model(model).generate(state.messages, tools)
        state.messages.append(state.output.message)
        return state

    return generate