﻿# Copyright (c) 2021, NVIDIA CORPORATION.  All rights reserved.
#
# NVIDIA CORPORATION and its licensors retain all intellectual property
# and proprietary rights in and to this software, related documentation
# and any modifications thereto.  Any use, reproduction, disclosure or
# distribution of this software and related documentation without an express
# license agreement from NVIDIA CORPORATION is strictly prohibited.

import numpy as np
import torch
from torch_utils import misc
from torch_utils import persistence
from torch_utils.ops import conv2d_resample
from torch_utils.ops import upfirdn2d
from torch_utils.ops import bias_act
from torch_utils.ops import fma
#from training.flow import DDSF

#----------------------------------------------------------------------------

@misc.profiled_function
def normalize_2nd_moment(x, dim=1, eps=1e-8):
    return x * (x.square().mean(dim=dim, keepdim=True) + eps).rsqrt()

#----------------------------------------------------------------------------

@misc.profiled_function
def modulated_conv2d(
    x,                          # Input tensor of shape [batch_size, in_channels, in_height, in_width].
    weight,                     # Weight tensor of shape [out_channels, in_channels, kernel_height, kernel_width].
    styles,                     # Modulation coefficients of shape [batch_size, in_channels].
    noise           = None,     # Optional noise tensor to add to the output activations.
    up              = 1,        # Integer upsampling factor.
    down            = 1,        # Integer downsampling factor.
    padding         = 0,        # Padding with respect to the upsampled image.
    resample_filter = None,     # Low-pass filter to apply when resampling activations. Must be prepared beforehand by calling upfirdn2d.setup_filter().
    demodulate      = True,     # Apply weight demodulation?
    flip_weight     = True,     # False = convolution, True = correlation (matches torch.nn.functional.conv2d).
    fused_modconv   = True,     # Perform modulation, convolution, and demodulation as a single fused operation?
):
    batch_size = x.shape[0]
    out_channels, in_channels, kh, kw = weight.shape
    misc.assert_shape(weight, [out_channels, in_channels, kh, kw]) # [OIkk]
    misc.assert_shape(x, [batch_size, in_channels, None, None]) # [NIHW]
    misc.assert_shape(styles, [batch_size, in_channels]) # [NI]

    # Pre-normalize inputs to avoid FP16 overflow.
    if x.dtype == torch.float16 and demodulate:
        weight = weight * (1 / np.sqrt(in_channels * kh * kw) / weight.norm(float('inf'), dim=[1,2,3], keepdim=True)) # max_Ikk
        styles = styles / styles.norm(float('inf'), dim=1, keepdim=True) # max_I

    # Calculate per-sample weights and demodulation coefficients.
    w = None
    dcoefs = None
    if demodulate or fused_modconv:
        w = weight.unsqueeze(0) # [NOIkk]
        w = w * styles.reshape(batch_size, 1, -1, 1, 1) # [NOIkk]
    if demodulate:
        dcoefs = (w.square().sum(dim=[2,3,4]) + 1e-8).rsqrt() # [NO]
    if demodulate and fused_modconv:
        w = w * dcoefs.reshape(batch_size, -1, 1, 1, 1) # [NOIkk]

    # Execute by scaling the activations before and after the convolution.
    if not fused_modconv:
        x = x * styles.to(x.dtype).reshape(batch_size, -1, 1, 1)
        x = conv2d_resample.conv2d_resample(x=x, w=weight.to(x.dtype), f=resample_filter, up=up, down=down, padding=padding, flip_weight=flip_weight)
        if demodulate and noise is not None:
            x = fma.fma(x, dcoefs.to(x.dtype).reshape(batch_size, -1, 1, 1), noise.to(x.dtype))
        elif demodulate:
            x = x * dcoefs.to(x.dtype).reshape(batch_size, -1, 1, 1)
        elif noise is not None:
            x = x.add_(noise.to(x.dtype))
        return x

    # Execute as one fused op using grouped convolution.
    with misc.suppress_tracer_warnings(): # this value will be treated as a constant
        batch_size = int(batch_size)
    misc.assert_shape(x, [batch_size, in_channels, None, None])
    x = x.reshape(1, -1, *x.shape[2:])
    w = w.reshape(-1, in_channels, kh, kw)
    x = conv2d_resample.conv2d_resample(x=x, w=w.to(x.dtype), f=resample_filter, up=up, down=down, padding=padding, groups=batch_size, flip_weight=flip_weight)
    x = x.reshape(batch_size, -1, *x.shape[2:])
    if noise is not None:
        x = x.add_(noise)
    return x

#----------------------------------------------------------------------------

@persistence.persistent_class
class FullyConnectedLayer(torch.nn.Module):
    def __init__(self,
        in_features,                # Number of input features.
        out_features,               # Number of output features.
        bias            = True,     # Apply additive bias before the activation function?
        activation      = 'linear', # Activation function: 'relu', 'lrelu', etc.
        lr_multiplier   = 1,        # Learning rate multiplier.
        bias_init       = 0,        # Initial value for the additive bias.
    ):
        super().__init__()
        self.activation = activation
        self.in_features = in_features
        self.out_features = out_features
        self.lr_multiplier = lr_multiplier
        self.weight = torch.nn.Parameter(torch.randn([out_features, in_features]) / lr_multiplier)
        self.bias = torch.nn.Parameter(torch.full([out_features], np.float32(bias_init))) if bias else None
        self.weight_gain = lr_multiplier / np.sqrt(in_features)
        self.bias_gain = lr_multiplier

    def forward(self, x):
        w = self.weight.to(x.dtype) * self.weight_gain
        b = self.bias
        if b is not None:
            b = b.to(x.dtype)
            if self.bias_gain != 1:
                b = b * self.bias_gain

        if self.activation == 'linear' and b is not None:
            x = torch.addmm(b.unsqueeze(0), x, w.t())
        else:
            x = x.matmul(w.t())
            x = bias_act.bias_act(x, b, act=self.activation)
        return x

    def __repr__(self):
        return self.__class__.__name__ + '(' + 'in:%s, out:%s, lr:%s, act:%s' % \
            (self.in_features, self.out_features, self.lr_multiplier, self.activation) + ')'


#----------------------------------------------------------------------------

@persistence.persistent_class
class Conv2dLayer(torch.nn.Module):
    def __init__(self,
        in_channels,                    # Number of input channels.
        out_channels,                   # Number of output channels.
        kernel_size,                    # Width and height of the convolution kernel.
        bias            = True,         # Apply additive bias before the activation function?
        activation      = 'linear',     # Activation function: 'relu', 'lrelu', etc.
        up              = 1,            # Integer upsampling factor.
        down            = 1,            # Integer downsampling factor.
        resample_filter = [1,3,3,1],    # Low-pass filter to apply when resampling activations.
        conv_clamp      = None,         # Clamp the output to +-X, None = disable clamping.
        channels_last   = False,        # Expect the input to have memory_format=channels_last?
        trainable       = True,         # Update the weights of this layer during training?
    ):
        super().__init__()
        self.activation = activation
        self.up = up
        self.down = down
        self.conv_clamp = conv_clamp
        self.register_buffer('resample_filter', upfirdn2d.setup_filter(resample_filter))
        self.padding = kernel_size // 2
        self.weight_gain = 1 / np.sqrt(in_channels * (kernel_size ** 2))
        self.act_gain = bias_act.activation_funcs[activation].def_gain

        memory_format = torch.channels_last if channels_last else torch.contiguous_format
        weight = torch.randn([out_channels, in_channels, kernel_size, kernel_size]).to(memory_format=memory_format)
        bias = torch.zeros([out_channels]) if bias else None
        if trainable:
            self.weight = torch.nn.Parameter(weight)
            self.bias = torch.nn.Parameter(bias) if bias is not None else None
        else:
            self.register_buffer('weight', weight)
            if bias is not None:
                self.register_buffer('bias', bias)
            else:
                self.bias = None

    def forward(self, x, gain=1):
        w = self.weight * self.weight_gain
        b = self.bias.to(x.dtype) if self.bias is not None else None
        flip_weight = (self.up == 1) # slightly faster
        x = conv2d_resample.conv2d_resample(x=x, w=w.to(x.dtype), f=self.resample_filter, up=self.up, down=self.down, padding=self.padding, flip_weight=flip_weight)

        act_gain = self.act_gain * gain
        act_clamp = self.conv_clamp * gain if self.conv_clamp is not None else None
        x = bias_act.bias_act(x, b, act=self.activation, gain=act_gain, clamp=act_clamp)
        return x

#----------------------------------------------------------------------------
@persistence.persistent_class
class MappingNetwork(torch.nn.Module):
    def __init__(self,
        z_dim,                      # Input latent (Z) dimensionality, 0 = no latent.
        c_dim,                      # Conditioning label (C) dimensionality, 0 = no label.
        w_dim,                      # Intermediate latent (W) dimensionality.
        num_ws,                     # Number of intermediate latents to output, None = do not broadcast.
        num_layers      = 8,        # Number of mapping layers.
        embed_features  = None,     # Label embedding dimensionality, None = same as w_dim.
        layer_features  = None,     # Number of intermediate features in the mapping layers, None = same as w_dim.
        activation      = 'lrelu',  # Activation function: 'relu', 'lrelu', etc.
        lr_multiplier   = 0.01,     # Learning rate multiplier for the mapping layers.
        w_avg_beta      = 0.995,    # Decay for tracking the moving average of W during training, None = do not track.
    ):
        super().__init__()
        self.z_dim = z_dim
        self.c_dim = c_dim
        self.w_dim = w_dim
        self.num_ws = num_ws
        self.num_layers = num_layers
        self.w_avg_beta = w_avg_beta

        if embed_features is None:
            embed_features = w_dim
        if c_dim == 0:
            embed_features = 0
        if layer_features is None:
            layer_features = w_dim
        features_list = [z_dim + embed_features] + [layer_features] * (num_layers - 1) + [w_dim]

        if c_dim > 0:
            self.embed = FullyConnectedLayer(c_dim, embed_features)
        for idx in range(num_layers):
            in_features = features_list[idx]
            out_features = features_list[idx + 1]
            layer = FullyConnectedLayer(in_features, out_features, activation=activation, lr_multiplier=lr_multiplier)
            setattr(self, f'fc{idx}', layer)

        if num_ws is not None and w_avg_beta is not None:
            self.register_buffer('w_avg', torch.zeros([w_dim]))

    def forward(self, z, c, truncation_psi=1, truncation_cutoff=None, skip_w_avg_update=False):
        # Embed, normalize, and concat inputs.
        x = None
        with torch.autograd.profiler.record_function('input'):
            if self.z_dim > 0:
                misc.assert_shape(z, [None, self.z_dim])
                x = normalize_2nd_moment(z.to(torch.float32))
            if self.c_dim > 0:
                misc.assert_shape(c, [None, self.c_dim])
                y = normalize_2nd_moment(self.embed(c.to(torch.float32)))
                x = torch.cat([x, y], dim=1) if x is not None else y

        # Main layers.
        for idx in range(self.num_layers):
            layer = getattr(self, f'fc{idx}')
            x = layer(x)

        # Update moving average of W.
        if self.w_avg_beta is not None and self.training and not skip_w_avg_update:
            with torch.autograd.profiler.record_function('update_w_avg'):
                self.w_avg.copy_(x.detach().mean(dim=0).lerp(self.w_avg, self.w_avg_beta))

        # Broadcast.
        if self.num_ws is not None:
            with torch.autograd.profiler.record_function('broadcast'):
                x = x.unsqueeze(1).repeat([1, self.num_ws, 1])

        # Apply truncation.
        if truncation_psi != 1:
            with torch.autograd.profiler.record_function('truncate'):
                assert self.w_avg_beta is not None
                if self.num_ws is None or truncation_cutoff is None:
                    x = self.w_avg.lerp(x, truncation_psi)
                else:
                    x[:, :truncation_cutoff] = self.w_avg.lerp(x[:, :truncation_cutoff], truncation_psi)
        return x

@persistence.persistent_class
class ParallelFullyConnectedLayer(torch.nn.Module):
    def __init__(self,
                 in_features,  # Number of input features.
                 out_features,  # Number of output features.
                 num_mlps,  # Number of parallel MLPs.
                 bias=True,  # Apply additive bias before the activation function?
                 shared=False,  # Use shared weights for all MLPs?
                 activation='linear',  # Activation function: 'relu', 'lrelu', etc.
                 lr_multiplier=1,  # Learning rate multiplier.
                 bias_init=0,  # Initial value for the additive bias.
                 ):
        super().__init__()
        self.in_features = in_features
        self.out_features = out_features
        self.num_mlps = num_mlps
        self.activation = activation
        self.lr_multiplier = lr_multiplier

        # Create weight for all MLPs at once
        if shared:
            self.weight = torch.nn.Parameter(torch.randn([1, in_features, out_features]) / lr_multiplier)
        else:
            self.weight = torch.nn.Parameter(torch.randn([num_mlps, in_features, out_features]) / lr_multiplier)

        if bias:
            # Create independent bias for each MLP
            if shared:
                self.bias = torch.nn.Parameter(torch.full([1, out_features], np.float32(bias_init)))
            else:
                self.bias = torch.nn.Parameter(torch.full([num_mlps, out_features], np.float32(bias_init)))

        else:
            self.bias = None

        self.weight_gain = lr_multiplier / np.sqrt(in_features)
        self.bias_gain = lr_multiplier
        self.shared = shared

    def forward(self, x):
        # x shape: [batch_size, num_mlps, in_features]
        assert x.shape[1] == self.num_mlps, f"Expected {self.num_mlps} vectors, but got {x.shape[1]}"

        w = self.weight.to(x.dtype) * self.weight_gain
        b = self.bias
        if b is not None:
            b = b.to(x.dtype)
            if self.bias_gain != 1:
                b = b * self.bias_gain
        if self.shared:
            w = w.repeat(self.num_mlps, 1, 1)
            b = b.repeat(self.num_mlps, 1)
            assert w.size() == (self.num_mlps, self.in_features, self.out_features)
            assert b.size() == (self.num_mlps, self.out_features)

        # Perform the matrix multiplication for all MLPs in parallel
        x = torch.einsum('bni,nio->bno', x, w)

        # Apply bias and activation
        if b is not None:
            x = x + b.unsqueeze(0)

        x = bias_act.bias_act(x, act=self.activation)

        return x

    def __repr__(self):
        return f'ParallelFullyConnectedLayer(in_features={self.weight.shape[1]}, out_features={self.weight.shape[2]}, num_mlps={self.weight.shape[0]}, activation={self.activation}, lr={self.lr_multiplier}, shared={self.shared})'
#----------------------------------------------------------------------------
import torch.nn as nn
import torch.nn.functional as F
@persistence.persistent_class
class FlowConceptMappingNetwork(torch.nn.Module):
    def __init__(self,
        z_dim,                      # Input latent (Z) dimensionality, 0 = no latent.
        c_dim,                      # Conditioning label (C) dimensionality, 0 = no label.
        w_dim,                      # Intermediate latent (W) dimensionality.
        num_ws,                     # Number of intermediate latents to output, None = do not broadcast.
        num_layers      = 8,        # Number of mapping layers.
        embed_features  = None,     # Label embedding dimensionality, None = same as w_dim.
        layer_features  = None,     # Number of intermediate features in the mapping layers, None = same as w_dim.
        activation      = 'lrelu',  # Activation function: 'relu', 'lrelu', etc.
        lr_multiplier   = 0.01,     # Learning rate multiplier for the mapping layers.
        w_avg_beta      = 0.995,    # Decay for tracking the moving average of W during training, None = do not track.
        cond_mode       = 'concat', # mode of coonditioning, stylegan3 uses concatenation
        i_dim           = 4,
        flow_blocks     = 2,
        flow_dim        = 10,
        flow_norm       = 0,
        use_label       = 0,
        min_dim         = 2,
    ):
        super().__init__()
        self.z_dim = z_dim
        self.c_dim = c_dim
        self.w_dim = w_dim
        self.i_dim = i_dim
        self.min_dim = min_dim
        self.num_ws = num_ws
        self.num_layers = num_layers
        self.w_avg_beta = w_avg_beta
        self.cond_mode = cond_mode
        self.flow_norm = flow_norm
        self.use_label = use_label
        if embed_features is None:
            embed_features = w_dim
        if c_dim == 0:
            embed_features = 0
        if layer_features is None:
            layer_features = w_dim
        features_list = [z_dim + embed_features] if self.cond_mode not in ['flow','mlp'] else [z_dim]
        features_list += [layer_features] * (num_layers - 1) + [w_dim]
        if c_dim > 0:
            if True:
                #self.embed = FullyConnectedLayer(2, embed_features)
                self.embed = ParallelFullyConnectedLayer(2, embed_features, num_mlps=c_dim)
                self.flow = DDSF(n_blocks=flow_blocks, hidden_dim=flow_dim)
                flow_features_list = [embed_features] + [layer_features] * (num_layers)
                #self.register_buffer('importance', torch.ones([1, self.i_dim*c_dim]).to(torch.float32))
                self.importance = torch.nn.Parameter(torch.ones([1, self.i_dim*c_dim]).to(torch.float32))
                for idx in range(num_layers):
                    in_features = flow_features_list[idx]
                    out_features = flow_features_list[idx + 1]
                    layer = ParallelFullyConnectedLayer(in_features, out_features, num_mlps=c_dim, activation=activation, lr_multiplier=lr_multiplier)
                    setattr(self, f'flow_fc{idx}', layer)
                layer = ParallelFullyConnectedLayer(layer_features, self.flow.num_params*i_dim, num_mlps=c_dim, lr_multiplier=lr_multiplier)
                setattr(self, f'flow_fc{num_layers}', layer)
                print(self)

        for idx in range(num_layers):
            in_features = features_list[idx]
            out_features = features_list[idx + 1]
            layer = FullyConnectedLayer(in_features, out_features, activation=activation, lr_multiplier=lr_multiplier)
            setattr(self, f'fc{idx}', layer)
            print(layer)

        if num_ws is not None and w_avg_beta is not None:
            self.register_buffer('w_avg', torch.zeros([w_dim]))

    def get_concepts(self, z):
        yact = F.one_hot(torch.zeros([len(z)]).long(), 2).float().to(z.device)
        ydeact = F.one_hot(torch.ones([len(z)]).long(), 2).float().to(z.device)
        yall = torch.cat([yact, ydeact], 0).unsqueeze(1).repeat(1, self.c_dim, 1)
        y = normalize_2nd_moment(self.embed(yall.to(torch.float32)), dim=-1) # B,C,W
        for idx in range(self.num_layers):
            layer = getattr(self, f'flow_fc{idx}')
            y = layer(y)
        layer = getattr(self, f'flow_fc{self.num_layers}')
        dsparams = layer(y) # B,C,flow.num_params*i_dim
        dsparams = dsparams.reshape(len(z)*2, self.c_dim, self.i_dim, self.flow.num_params)
        dsparams = dsparams.reshape(len(z)*2, self.c_dim*self.i_dim, self.flow.num_params)
        zz = torch.cat([z, z], 0)
        z_out = self.flow(zz, dsparams)
        z_act, z_deact = z_out.chunk(2)
        assert z_act.size() == z_deact.size()
        return z_act, z_deact, z

    def create_mask(self, c):
        mask_act = torch.zeros([len(c), self.z_dim]).to(c.device)
        mask_deact = torch.zeros([len(c), self.z_dim]).to(c.device)
        mask_rest = torch.ones([len(c), self.z_dim]).to(c.device)
        for i in range(self.c_dim):
            mask_act[:,i*self.i_dim:(i+1)*self.i_dim] = (c[:,i].unsqueeze(-1)==1).repeat(1, self.i_dim) * (self.importance[:,i*self.i_dim:(i+1)*self.i_dim]>0.1)
            mask_deact[:,i*self.i_dim:(i+1)*self.i_dim] = (c[:,i].unsqueeze(-1)==0).repeat(1, self.i_dim) * (self.importance[:,i*self.i_dim:(i+1)*self.i_dim]>0.1)
            mask_rest[:,i*self.i_dim:(i+1)*self.i_dim] = (self.importance[:,i*self.i_dim:(i+1)*self.i_dim]<=0.1)
        mask_act = (mask_act>0).bool()
        mask_deact = (mask_deact>0).bool()
        mask_rest = (mask_rest>0).bool()
        assert ((mask_act + mask_deact + mask_rest) == 1).all()
        return mask_act, mask_deact, mask_rest

    def forward(self, z, c, truncation_psi=1, truncation_cutoff=None, skip_w_avg_update=False,
                sparse_loss=False, label=None):
        # Embed, normalize, and concat inputs.
        x = None
        with torch.autograd.profiler.record_function('input'):
            if self.z_dim > 0:
                #misc.assert_shape(z, [None, self.z_dim])
                zsame = z[:, self.i_dim * self.c_dim:]
                zin = (z[:, :self.i_dim * self.c_dim])
                z_act, z_deact, z_null = self.get_concepts(zin)
                z_out = []
                imp = self.importance*(self.importance>0.1)
                for i in range(self.c_dim):
                    start = self.i_dim * i
                    end = self.i_dim * (i + 1)
                    cur_ci = c[:, i].unsqueeze(-1)
                    cur_out = (z_act[:, start:end]) * cur_ci +(z_deact[:, start:end]) * (1-cur_ci)
                    assert cur_out.size() == (len(z), self.i_dim)
                    z_out.append(cur_out)
                z_out = torch.cat(z_out, 1)
                if self.flow_norm == 0:
                    z_out = normalize_2nd_moment(z_out*imp+zin*(1-imp))
                    zsame = normalize_2nd_moment(zsame)
                    x = (torch.cat([(z_out), zsame], 1).to(torch.float32))
                elif self.flow_norm == 1:
                    z_out = normalize_2nd_moment(z_out*imp+zin)
                    zsame = normalize_2nd_moment(zsame)
                    x = (torch.cat([(z_out), zsame], 1).to(torch.float32))
                if self.flow_norm == 2:
                    z_out = z_out*imp+zin*(1-imp)
                    x = (torch.cat([(z_out), zsame], 1).to(torch.float32))
                    x = normalize_2nd_moment(x)
                elif self.flow_norm == 3:
                    z_out = z_out*imp+zin*(1-imp)
                    x = (torch.cat([(z_out), zsame], 1).to(torch.float32))
                    content_mask = self.create_mask(c)[-1]
                    style_mask = ~content_mask
                    style_mask = style_mask.expand(x.size(0), style_mask.size(1))
                    content_mask = content_mask.expand(x.size(0), content_mask.size(1))
                    assert x[style_mask].view(x.size(0), -1).size(-1) == torch.sum(self.importance > 0.1)
                    x[style_mask] = normalize_2nd_moment(x[style_mask].view(x.size(0), -1)).view(-1)
                    x[content_mask] = normalize_2nd_moment(x[content_mask].view(x.size(0), -1)).view(-1)
                elif self.flow_norm == 4:
                    z_out = z_out*imp+zin
                    x = (torch.cat([(z_out), zsame], 1).to(torch.float32))
                    content_mask = self.create_mask(c)[-1]
                    style_mask = ~content_mask
                    style_mask = style_mask.expand(x.size(0), style_mask.size(1))
                    content_mask = content_mask.expand(x.size(0), content_mask.size(1))
                    assert x[style_mask].view(x.size(0), -1).size(-1) == torch.sum(self.importance > 0.1)
                    x[style_mask] = normalize_2nd_moment(x[style_mask].view(x.size(0), -1)).view(-1)
                    x[content_mask] = normalize_2nd_moment(x[content_mask].view(x.size(0), -1)).view(-1)


        #x = self.layernorm(x)
        # Main layers.
        for idx in range(self.num_layers):
            layer = getattr(self, f'fc{idx}')
            x = layer(x)

        # Update moving average of W.
        if self.w_avg_beta is not None and self.training and not skip_w_avg_update:
            with torch.autograd.profiler.record_function('update_w_avg'):
                self.w_avg.copy_(x.detach().mean(dim=0).lerp(self.w_avg, self.w_avg_beta))

        # Broadcast.
        if self.num_ws is not None:
            with torch.autograd.profiler.record_function('broadcast'):
                x = x.unsqueeze(1).repeat([1, self.num_ws, 1])

        # Apply truncation.
        if truncation_psi != 1:
            with torch.autograd.profiler.record_function('truncate'):
                assert self.w_avg_beta is not None
                if self.num_ws is None or truncation_cutoff is None:
                    x = self.w_avg.lerp(x, truncation_psi)
                else:
                    x[:, :truncation_cutoff] = self.w_avg.lerp(x[:, :truncation_cutoff], truncation_psi)
        if sparse_loss:
            imp = self.importance*(self.importance>0.1)
            imp = imp.view(1, self.c_dim, self.i_dim)
            num_act = torch.sum(imp, dim=-1, keepdim=True).detach()
            should_sparse = (num_act>self.min_dim)
            should_sparse_imp = imp * should_sparse
            return x, torch.mean(torch.abs(should_sparse_imp)), torch.mean(z_out**2), torch.sum(imp>0.1).item(), torch.mean(torch.abs(z_out)).item(), torch.mean(torch.abs(zsame)).item()
        else:
            return x

@persistence.persistent_class
class ConceptMappingNetwork(torch.nn.Module):
    def __init__(self,
                 z_dim,                      # Input latent (Z) dimensionality, 0 = no latent.
                 c_dim,                      # Conditioning label (C) dimensionality, 0 = no label.
                 w_dim,                      # Intermediate latent (W) dimensionality.
                 num_ws,                     # Number of intermediate latents to output, None = do not broadcast.
                 num_layers      = 8,        # Number of mapping layers.
                 embed_features  = None,     # Label embedding dimensionality, None = same as w_dim.
                 layer_features  = None,     # Number of intermediate features in the mapping layers, None = same as w_dim.
                 activation      = 'lrelu',  # Activation function: 'relu', 'lrelu', etc.
                 lr_multiplier   = 0.01,     # Learning rate multiplier for the mapping layers.
                 w_avg_beta      = 0.995,    # Decay for tracking the moving average of W during training, None = do not track.
                 cond_mode       = 'concat', # mode of coonditioning, stylegan3 uses concatenation
                 i_dim           = 4,
                 flow_blocks     = 2,
                 flow_dim        = 10,
                 flow_norm       = 0,
                 use_label       = 0,
                 min_dim         = 2,
                 ):
        super().__init__()
        self.z_dim = z_dim
        self.c_dim = c_dim
        self.w_dim = w_dim
        self.i_dim = i_dim
        self.min_dim = min_dim
        self.num_ws = num_ws
        self.num_layers = num_layers
        self.w_avg_beta = w_avg_beta
        self.cond_mode = cond_mode
        self.flow_norm = flow_norm
        if c_dim > 0:
            num_hidden = min(512, max(i_dim, w_dim))
            self.activate_mlp_net = nn.Sequential(ParallelFullyConnectedLayer(i_dim, num_hidden, num_mlps=c_dim, activation=activation, lr_multiplier=lr_multiplier),
                                                    ParallelFullyConnectedLayer(num_hidden, i_dim, num_mlps=c_dim, activation=activation, lr_multiplier=lr_multiplier)
                                                    )
            self.deactivate_mlp_net = nn.Sequential(ParallelFullyConnectedLayer(i_dim, num_hidden, num_mlps=c_dim, activation=activation, lr_multiplier=lr_multiplier),
                                                    ParallelFullyConnectedLayer(num_hidden, i_dim, num_mlps=c_dim, activation=activation, lr_multiplier=lr_multiplier)
                                                    )
            self.null_mlp_net = nn.Sequential(ParallelFullyConnectedLayer(i_dim, num_hidden, num_mlps=c_dim, activation=activation, lr_multiplier=lr_multiplier, shared=True),
                                              ParallelFullyConnectedLayer(num_hidden, i_dim, num_mlps=c_dim, activation=activation, lr_multiplier=lr_multiplier, shared=True)
                                                    )
            self.main_mlp_net = nn.Sequential(FullyConnectedLayer(z_dim-i_dim*c_dim, num_hidden, activation=activation, lr_multiplier=lr_multiplier),
                                              FullyConnectedLayer(num_hidden, z_dim-i_dim*c_dim, activation=activation, lr_multiplier=lr_multiplier)
                                              )
            self.importance = torch.nn.Parameter(torch.ones([1, c_dim, self.i_dim]).to(torch.float32))
            print(self)

        if num_ws is not None and w_avg_beta is not None:
            self.register_buffer('w_avg', torch.zeros([w_dim]))

    def forward(self, z, c, truncation_psi=1, truncation_cutoff=None, skip_w_avg_update=False,
                sparse_loss=False, label=None):
        with (torch.autograd.profiler.record_function('input')):
            if self.z_dim > 0:
                zs = []
                for i in range(self.c_dim):
                    zs.append(normalize_2nd_moment(z[:, i*self.i_dim:(i+1)*self.i_dim]))
                zs = torch.stack(zs, 1)
                assert zs.size() == (len(z), self.c_dim, self.i_dim)
                activations = self.activate_mlp_net(zs)
                deactivations = self.deactivate_mlp_net(zs)
                nulls = self.null_mlp_net(zs)
                assert activations.size() == deactivations.size() == (len(z), self.c_dim, self.i_dim)
                z_out = []
                for i in range(self.c_dim):
                    cur_ci = c[:, i].unsqueeze(-1)
                    cur_act = activations[:, i]
                    cur_deact = deactivations[:, i]
                    cur_null = nulls[:,i]
                    cur_imp = self.importance[:,i] * (self.importance[:,i]>0.1)
                    assert cur_act.size() == (len(z), self.i_dim) == cur_deact.size() == cur_null.size()
                    assert cur_imp.size() == (1, self.i_dim)
                    cur_z_out = cur_act*cur_ci + cur_deact*(1-cur_ci)
                    cur_z_out = cur_z_out * cur_imp + cur_null*(1-cur_imp)
                    assert cur_z_out.size() == (len(z), self.i_dim)
                    z_out.append(cur_z_out)
                z_out = torch.cat(z_out, 1)
                z_main = self.main_mlp_net(normalize_2nd_moment(z[:, self.c_dim*self.i_dim:]))
        x = torch.cat([z_out, z_main], 1).to(torch.float32)

        # Update moving average of W.
        if self.w_avg_beta is not None and self.training and not skip_w_avg_update:
            with torch.autograd.profiler.record_function('update_w_avg'):
                self.w_avg.copy_(x.detach().mean(dim=0).lerp(self.w_avg, self.w_avg_beta))

        # Broadcast.
        if self.num_ws is not None:
            with torch.autograd.profiler.record_function('broadcast'):
                x = x.unsqueeze(1).repeat([1, self.num_ws, 1])

        # Apply truncation.
        if truncation_psi != 1:
            with torch.autograd.profiler.record_function('truncate'):
                assert self.w_avg_beta is not None
                if self.num_ws is None or truncation_cutoff is None:
                    x = self.w_avg.lerp(x, truncation_psi)
                else:
                    x[:, :truncation_cutoff] = self.w_avg.lerp(x[:, :truncation_cutoff], truncation_psi)

        if sparse_loss:
            importance = self.importance*(self.importance>0.1)
            num_act = torch.sum(importance>0.1, dim=-1, keepdim=True).detach()
            should_sparse = (num_act>self.min_dim)
            assert should_sparse.size() == (1, self.c_dim, 1)
            should_sparse_imp = importance * should_sparse
            return x, torch.mean(torch.abs(should_sparse_imp)), torch.mean(z_out**2), torch.sum(importance>0.1).item(), torch.mean(torch.abs(z_out)).item(), torch.mean(torch.abs(z_main)).item()
        else:
            return x




@persistence.persistent_class
class CausalConceptMappingNetwork(torch.nn.Module):
    def __init__(self,
                 z_dim,                      # Input latent (Z) dimensionality, 0 = no latent.
                 c_dim,                      # Conditioning label (C) dimensionality, 0 = no label.
                 w_dim,                      # Intermediate latent (W) dimensionality.
                 num_ws,                     # Number of intermediate latents to output, None = do not broadcast.
                 num_layers      = 8,        # Number of mapping layers.
                 embed_features  = None,     # Label embedding dimensionality, None = same as w_dim.
                 layer_features  = None,     # Number of intermediate features in the mapping layers, None = same as w_dim.
                 activation      = 'lrelu',  # Activation function: 'relu', 'lrelu', etc.
                 lr_multiplier   = 0.01,     # Learning rate multiplier for the mapping layers.
                 w_avg_beta      = 0.995,    # Decay for tracking the moving average of W during training, None = do not track.
                 cond_mode       = 'concat', # mode of coonditioning, stylegan3 uses concatenation
                 i_dim           = 4,
                 flow_blocks     = 2,
                 flow_dim        = 10,
                 flow_norm       = 0,
                 use_label       = 0,
                 min_dim         = 2,
                 ):
        super().__init__()
        self.z_dim = z_dim
        self.c_dim = c_dim
        self.w_dim = w_dim
        self.i_dim = i_dim
        self.min_dim = min_dim
        self.num_ws = num_ws
        self.num_layers = num_layers
        self.w_avg_beta = w_avg_beta
        self.cond_mode = cond_mode
        self.flow_norm = flow_norm
        self.ori_c_dim = 37

        if c_dim > 0:
            num_hidden = min(512, max(i_dim, w_dim))
            self.activate_mlp_net = nn.Sequential(ParallelFullyConnectedLayer(i_dim, num_hidden, num_mlps=c_dim, activation=activation, lr_multiplier=lr_multiplier),
                                                  ParallelFullyConnectedLayer(num_hidden, i_dim, num_mlps=c_dim, activation=activation, lr_multiplier=lr_multiplier)
                                                  )
            self.deactivate_mlp_net = nn.Sequential(ParallelFullyConnectedLayer(i_dim, num_hidden, num_mlps=c_dim, activation=activation, lr_multiplier=lr_multiplier),
                                                    ParallelFullyConnectedLayer(num_hidden, i_dim, num_mlps=c_dim, activation=activation, lr_multiplier=lr_multiplier)
                                                    )
            self.null_mlp_net = nn.Sequential(ParallelFullyConnectedLayer(i_dim, num_hidden, num_mlps=c_dim, activation=activation, lr_multiplier=lr_multiplier, shared=True),
                                                    ParallelFullyConnectedLayer(num_hidden, i_dim, num_mlps=c_dim, activation=activation, lr_multiplier=lr_multiplier, shared=True)
                                                    )
            self.main_mlp_net = nn.Sequential(FullyConnectedLayer(z_dim-i_dim*self.ori_c_dim, num_hidden, activation=activation, lr_multiplier=lr_multiplier),
                                              FullyConnectedLayer(num_hidden, z_dim-i_dim*self.ori_c_dim, activation=activation, lr_multiplier=lr_multiplier)
                                              )
            self.importance = torch.nn.Parameter(torch.ones([1, self.ori_c_dim, self.i_dim]).to(torch.float32))
            self.causal_nodes = [1, 1, 1, 1, 2, 2, 12, 5, 4, 2, 8, 2, 3, 2, 8, 3, 3, 8, 1, 3, 1, 11, 2, 4, 4, 2, 4, 2, 4, 4, 2, 2, 8, 4, 3, 2, 3]
            self.cum_causal_nodes = [0, 1, 2, 3, 4, 6, 8, 20, 25, 29, 31, 39, 41, 44, 46, 54, 57, 60, 68, 69, 72, 73, 84, 86, 90, 94, 96, 100, 102, 106, 110, 112, 114, 122, 126, 129, 131, 134]
            #self.causal_nodes = [1, 1, 1, 1, 2, 12, 5, 2, 2, 8, 2, 2, 2, 2, 2, 2, 9, 1, 2, 2, 4, 2, 2, 2, 4, 2, 2, 2, 2, 4, 4, 2, 4, 2, 3, 2, 3]
            #self.cum_causal_nodes = [0, 1, 2, 3, 4, 6, 18, 23, 25, 27, 35, 37, 39, 41, 43, 45, 47, 56, 57, 59, 61, 65, 67, 69, 71, 75, 77, 79, 81, 83, 87, 91, 93, 97, 99, 102, 104, 107]

            #self.register_buffer('causal_weights', torch.tensor([0.5, 0.5, 0.5, 0.5, 1, 1, 1, 1.5, 1.5, 1, 1, 1, 1, 1, 1.5, 1, 1, 1, 1, 1, 1, 1, 1.5, 1.5, 1, 1.5, 1.5, 1, 1.5, 1, 1.5, 1, 1.5, 1, 1, 1.5, 1.5]))
            #self.register_buffer('causal_dims', torch.tensor([5, 5, 5, 5, 4, 4, 4, 3, 3, 4, 4, 4, 4, 4, 3, 4, 4, 4, 4, 4, 4, 4, 3, 3, 4, 3, 3, 4, 3, 4, 3, 4, 3, 4, 4, 3, 3]))
            self.register_buffer('causal_weights',  torch.tensor([1., 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
            self.register_buffer('causal_dims', torch.tensor([8, 4, 6, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4]))

            print(self)

        if num_ws is not None and w_avg_beta is not None:
            self.register_buffer('w_avg', torch.zeros([w_dim]))

    def get_importance(self):
        cur_imp = self.importance*(self.importance>0.1)
        num_act = torch.sum(self.importance>0.1, dim=-1, keepdim=True).detach() # 1,C,1
        should_keep = (num_act>self.causal_dims.view(1,-1,1)).float()
        #notused = torch.ones_like(cur_imp)*(cur_imp>0.1).float()
        out = should_keep * cur_imp + (1-should_keep) * cur_imp.detach()
        return out+0*self.importance # add dummy to avoid error in DDP


    def forward(self, z, c, truncation_psi=1, truncation_cutoff=None, skip_w_avg_update=False,
                sparse_loss=False, label=None):
        with (torch.autograd.profiler.record_function('input')):
            if self.z_dim > 0:
                zs = []
                for i in range(self.ori_c_dim):
                    zs = zs + [normalize_2nd_moment(z[:, i*self.i_dim:(i+1)*self.i_dim])]*self.causal_nodes[i]
                zs = torch.stack(zs, 1)
                assert zs.size() == (len(z), self.c_dim, self.i_dim)
                all_activations = self.activate_mlp_net(zs)
                all_deactivations = self.deactivate_mlp_net(zs)
                all_nulls = self.null_mlp_net(zs)
                assert all_activations.size() == all_deactivations.size() == (len(z), self.c_dim, self.i_dim)
                z_out = []
                # merge activations
                importance = self.get_importance()
                for i in range(self.ori_c_dim):
                    start_dim = self.cum_causal_nodes[i]
                    end_dim = self.cum_causal_nodes[i+1]
                    cur_activations = all_activations[:, start_dim:end_dim] # B,T, I
                    cur_deactivations = all_deactivations[:, start_dim:end_dim] # B,T,I
                    cur_labels = c[:, start_dim:end_dim].unsqueeze(-1) # B,T,1
                    #assert torch.max(torch.sum(cur_labels, dim=1).view(-1)) <= 1
                    cur_act = torch.sum(cur_activations*cur_labels, dim=1) # B,I
                    cur_deact = torch.mean(cur_deactivations*torch.ones_like(cur_labels), dim=1) # B,I
                    cur_imp = importance[:,i] # 1,I
                    assert cur_imp.size() == (1, self.i_dim)
                    cur_ci = (torch.sum(cur_labels, dim=1)>0).float()
                    assert cur_ci.size() == (len(c), 1)
                    cur_z_out = cur_act*cur_ci + cur_deact*(1-cur_ci)
                    cur_z_out = cur_z_out * cur_imp + all_nulls[:,i]*(1-cur_imp)
                    assert cur_z_out.size() == (len(z), self.i_dim)
                    z_out.append(cur_z_out)

                z_out = torch.cat(z_out, 1)
                z_main = self.main_mlp_net(normalize_2nd_moment(z[:, self.ori_c_dim*self.i_dim:]))
        x = torch.cat([z_out, z_main], 1).to(torch.float32)

        # Update moving average of W.
        if self.w_avg_beta is not None and self.training and not skip_w_avg_update:
            with torch.autograd.profiler.record_function('update_w_avg'):
                self.w_avg.copy_(x.detach().mean(dim=0).lerp(self.w_avg, self.w_avg_beta))

        # Broadcast.
        if self.num_ws is not None:
            with torch.autograd.profiler.record_function('broadcast'):
                x = x.unsqueeze(1).repeat([1, self.num_ws, 1])

        # Apply truncation.
        if truncation_psi != 1:
            with torch.autograd.profiler.record_function('truncate'):
                assert self.w_avg_beta is not None
                if self.num_ws is None or truncation_cutoff is None:
                    x = self.w_avg.lerp(x, truncation_psi)
                else:
                    x[:, :truncation_cutoff] = self.w_avg.lerp(x[:, :truncation_cutoff], truncation_psi)

        if sparse_loss:
            assert importance.size() == (1, self.ori_c_dim, self.i_dim)
            reweight_importance = importance*(self.causal_weights.view(1,-1,1))
            return x, torch.mean(torch.abs(reweight_importance)), torch.mean(z_out**2), torch.sum(importance>0.1).item(), torch.mean(torch.abs(z_out)).item(), torch.mean(torch.abs(z_main)).item()
        else:
            return x




#----------------------------------------------------------------------------

@persistence.persistent_class
class SynthesisLayer(torch.nn.Module):
    def __init__(self,
        in_channels,                    # Number of input channels.
        out_channels,                   # Number of output channels.
        w_dim,                          # Intermediate latent (W) dimensionality.
        resolution,                     # Resolution of this layer.
        kernel_size     = 3,            # Convolution kernel size.
        up              = 1,            # Integer upsampling factor.
        use_noise       = True,         # Enable noise input?
        activation      = 'lrelu',      # Activation function: 'relu', 'lrelu', etc.
        resample_filter = [1,3,3,1],    # Low-pass filter to apply when resampling activations.
        conv_clamp      = None,         # Clamp the output of convolution layers to +-X, None = disable clamping.
        channels_last   = False,        # Use channels_last format for the weights?
    ):
        super().__init__()
        self.resolution = resolution
        self.up = up
        self.use_noise = use_noise
        self.activation = activation
        self.conv_clamp = conv_clamp
        self.register_buffer('resample_filter', upfirdn2d.setup_filter(resample_filter))
        self.padding = kernel_size // 2
        self.act_gain = bias_act.activation_funcs[activation].def_gain

        self.affine = FullyConnectedLayer(w_dim, in_channels, bias_init=1)
        memory_format = torch.channels_last if channels_last else torch.contiguous_format
        self.weight = torch.nn.Parameter(torch.randn([out_channels, in_channels, kernel_size, kernel_size]).to(memory_format=memory_format))
        if use_noise:
            self.register_buffer('noise_const', torch.randn([resolution, resolution]))
            self.noise_strength = torch.nn.Parameter(torch.zeros([]))
        self.bias = torch.nn.Parameter(torch.zeros([out_channels]))

    def forward(self, x, w, noise_mode='random', fused_modconv=True, gain=1):
        assert noise_mode in ['random', 'const', 'none']
        in_resolution = self.resolution // self.up
        misc.assert_shape(x, [None, self.weight.shape[1], in_resolution, in_resolution])
        styles = self.affine(w)

        noise = None
        if self.use_noise and noise_mode == 'random':
            noise = torch.randn([x.shape[0], 1, self.resolution, self.resolution], device=x.device) * self.noise_strength
        if self.use_noise and noise_mode == 'const':
            noise = self.noise_const * self.noise_strength

        flip_weight = (self.up == 1) # slightly faster
        x = modulated_conv2d(x=x, weight=self.weight, styles=styles, noise=noise, up=self.up,
            padding=self.padding, resample_filter=self.resample_filter, flip_weight=flip_weight, fused_modconv=fused_modconv)

        act_gain = self.act_gain * gain
        act_clamp = self.conv_clamp * gain if self.conv_clamp is not None else None
        x = bias_act.bias_act(x, self.bias.to(x.dtype), act=self.activation, gain=act_gain, clamp=act_clamp)
        return x

#----------------------------------------------------------------------------

@persistence.persistent_class
class ToRGBLayer(torch.nn.Module):
    def __init__(self, in_channels, out_channels, w_dim, kernel_size=1, conv_clamp=None, channels_last=False):
        super().__init__()
        self.conv_clamp = conv_clamp
        self.affine = FullyConnectedLayer(w_dim, in_channels, bias_init=1)
        memory_format = torch.channels_last if channels_last else torch.contiguous_format
        self.weight = torch.nn.Parameter(torch.randn([out_channels, in_channels, kernel_size, kernel_size]).to(memory_format=memory_format))
        self.bias = torch.nn.Parameter(torch.zeros([out_channels]))
        self.weight_gain = 1 / np.sqrt(in_channels * (kernel_size ** 2))

    def forward(self, x, w, fused_modconv=True):
        styles = self.affine(w) * self.weight_gain
        x = modulated_conv2d(x=x, weight=self.weight, styles=styles, demodulate=False, fused_modconv=fused_modconv)
        x = bias_act.bias_act(x, self.bias.to(x.dtype), clamp=self.conv_clamp)
        return x

#----------------------------------------------------------------------------

@persistence.persistent_class
class SynthesisBlock(torch.nn.Module):
    def __init__(self,
        in_channels,                        # Number of input channels, 0 = first block.
        out_channels,                       # Number of output channels.
        w_dim,                              # Intermediate latent (W) dimensionality.
        resolution,                         # Resolution of this block.
        img_channels,                       # Number of output color channels.
        is_last,                            # Is this the last block?
        architecture        = 'skip',       # Architecture: 'orig', 'skip', 'resnet'.
        resample_filter     = [1,3,3,1],    # Low-pass filter to apply when resampling activations.
        conv_clamp          = None,         # Clamp the output of convolution layers to +-X, None = disable clamping.
        use_fp16            = False,        # Use FP16 for this block?
        fp16_channels_last  = False,        # Use channels-last memory format with FP16?
        **layer_kwargs,                     # Arguments for SynthesisLayer.
    ):
        assert architecture in ['orig', 'skip', 'resnet']
        super().__init__()
        self.in_channels = in_channels
        self.w_dim = w_dim
        self.resolution = resolution
        self.img_channels = img_channels
        self.is_last = is_last
        self.architecture = architecture
        self.use_fp16 = use_fp16
        self.channels_last = (use_fp16 and fp16_channels_last)
        self.register_buffer('resample_filter', upfirdn2d.setup_filter(resample_filter))
        self.num_conv = 0
        self.num_torgb = 0

        if in_channels == 0:
            self.const = torch.nn.Parameter(torch.randn([out_channels, resolution, resolution]))

        if in_channels != 0:
            self.conv0 = SynthesisLayer(in_channels, out_channels, w_dim=w_dim, resolution=resolution, up=2,
                resample_filter=resample_filter, conv_clamp=conv_clamp, channels_last=self.channels_last, **layer_kwargs)
            self.num_conv += 1

        self.conv1 = SynthesisLayer(out_channels, out_channels, w_dim=w_dim, resolution=resolution,
            conv_clamp=conv_clamp, channels_last=self.channels_last, **layer_kwargs)
        self.num_conv += 1

        if is_last or architecture == 'skip':
            self.torgb = ToRGBLayer(out_channels, img_channels, w_dim=w_dim,
                conv_clamp=conv_clamp, channels_last=self.channels_last)
            self.num_torgb += 1

        if in_channels != 0 and architecture == 'resnet':
            self.skip = Conv2dLayer(in_channels, out_channels, kernel_size=1, bias=False, up=2,
                resample_filter=resample_filter, channels_last=self.channels_last)

    def forward(self, x, img, ws, force_fp32=False, fused_modconv=None, **layer_kwargs):
        misc.assert_shape(ws, [None, self.num_conv + self.num_torgb, self.w_dim])
        w_iter = iter(ws.unbind(dim=1))
        dtype = torch.float16 if self.use_fp16 and not force_fp32 else torch.float32
        memory_format = torch.channels_last if self.channels_last and not force_fp32 else torch.contiguous_format
        if fused_modconv is None:
            with misc.suppress_tracer_warnings(): # this value will be treated as a constant
                fused_modconv = (not self.training) and (dtype == torch.float32 or int(x.shape[0]) == 1)

        # Input.
        if self.in_channels == 0:
            x = self.const.to(dtype=dtype, memory_format=memory_format)
            x = x.unsqueeze(0).repeat([ws.shape[0], 1, 1, 1])
        else:
            misc.assert_shape(x, [None, self.in_channels, self.resolution // 2, self.resolution // 2])
            x = x.to(dtype=dtype, memory_format=memory_format)

        # Main layers.
        if self.in_channels == 0:
            x = self.conv1(x, next(w_iter), fused_modconv=fused_modconv, **layer_kwargs)
        elif self.architecture == 'resnet':
            y = self.skip(x, gain=np.sqrt(0.5))
            x = self.conv0(x, next(w_iter), fused_modconv=fused_modconv, **layer_kwargs)
            x = self.conv1(x, next(w_iter), fused_modconv=fused_modconv, gain=np.sqrt(0.5), **layer_kwargs)
            x = y.add_(x)
        else:
            x = self.conv0(x, next(w_iter), fused_modconv=fused_modconv, **layer_kwargs)
            x = self.conv1(x, next(w_iter), fused_modconv=fused_modconv, **layer_kwargs)

        # ToRGB.
        if img is not None:
            misc.assert_shape(img, [None, self.img_channels, self.resolution // 2, self.resolution // 2])
            img = upfirdn2d.upsample2d(img, self.resample_filter)
        if self.is_last or self.architecture == 'skip':
            y = self.torgb(x, next(w_iter), fused_modconv=fused_modconv)
            y = y.to(dtype=torch.float32, memory_format=torch.contiguous_format)
            img = img.add_(y) if img is not None else y

        assert x.dtype == dtype
        assert img is None or img.dtype == torch.float32
        return x, img

#----------------------------------------------------------------------------

@persistence.persistent_class
class SynthesisNetwork(torch.nn.Module):
    def __init__(self,
        w_dim,                      # Intermediate latent (W) dimensionality.
        img_resolution,             # Output image resolution.
        img_channels,               # Number of color channels.
        channel_base    = 32768,    # Overall multiplier for the number of channels.
        channel_max     = 512,      # Maximum number of channels in any layer.
        num_fp16_res    = 0,        # Use FP16 for the N highest resolutions.
        **block_kwargs,             # Arguments for SynthesisBlock.
    ):
        assert img_resolution >= 4 and img_resolution & (img_resolution - 1) == 0
        super().__init__()
        self.w_dim = w_dim
        self.img_resolution = img_resolution
        self.img_resolution_log2 = int(np.log2(img_resolution))
        self.img_channels = img_channels
        self.block_resolutions = [2 ** i for i in range(2, self.img_resolution_log2 + 1)]
        channels_dict = {res: min(channel_base // res, channel_max) for res in self.block_resolutions}
        fp16_resolution = max(2 ** (self.img_resolution_log2 + 1 - num_fp16_res), 8)

        self.num_ws = 0
        for res in self.block_resolutions:
            in_channels = channels_dict[res // 2] if res > 4 else 0
            out_channels = channels_dict[res]
            use_fp16 = (res >= fp16_resolution)
            is_last = (res == self.img_resolution)
            block = SynthesisBlock(in_channels, out_channels, w_dim=w_dim, resolution=res,
                img_channels=img_channels, is_last=is_last, use_fp16=use_fp16, **block_kwargs)
            self.num_ws += block.num_conv
            if is_last:
                self.num_ws += block.num_torgb
            setattr(self, f'b{res}', block)

    def forward(self, ws, **block_kwargs):
        block_ws = []
        with torch.autograd.profiler.record_function('split_ws'):
            misc.assert_shape(ws, [None, self.num_ws, self.w_dim])
            ws = ws.to(torch.float32)
            w_idx = 0
            for res in self.block_resolutions:
                block = getattr(self, f'b{res}')
                block_ws.append(ws.narrow(1, w_idx, block.num_conv + block.num_torgb))
                w_idx += block.num_conv

        x = img = None
        for res, cur_ws in zip(self.block_resolutions, block_ws):
            block = getattr(self, f'b{res}')
            x, img = block(x, img, cur_ws, **block_kwargs)
        return img

#----------------------------------------------------------------------------

@persistence.persistent_class
class Generator(torch.nn.Module):
    def __init__(self,
        z_dim,                      # Input latent (Z) dimensionality.
        c_dim,                      # Conditioning label (C) dimensionality.
        w_dim,                      # Intermediate latent (W) dimensionality.
        img_resolution,             # Output resolution.
        img_channels,               # Number of output color channels.
        is_causal = False,          # Use causal network?
        mapping_kwargs      = {},   # Arguments for MappingNetwork.
        synthesis_kwargs    = {},   # Arguments for SynthesisNetwork.
    ):
        super().__init__()
        self.z_dim = z_dim
        self.c_dim = c_dim
        self.w_dim = w_dim
        self.img_resolution = img_resolution
        self.img_channels = img_channels
        self.synthesis = SynthesisNetwork(w_dim=w_dim, img_resolution=img_resolution, img_channels=img_channels, **synthesis_kwargs)
        self.num_ws = self.synthesis.num_ws
        if is_causal:
            print('>>>>>>>>>>> Using CausalConceptMappingNetwork <<<<<<<<<<<<<<<<<<')
            self.mapping = CausalConceptMappingNetwork(z_dim=z_dim, c_dim=c_dim, w_dim=w_dim, num_ws=self.num_ws, **mapping_kwargs)
        else:
            self.mapping = ConceptMappingNetwork(z_dim=z_dim, c_dim=c_dim, w_dim=w_dim, num_ws=self.num_ws, **mapping_kwargs)

    def forward(self, z, c, truncation_psi=1, truncation_cutoff=None, label=None, **synthesis_kwargs):
        ws = self.mapping(z, c, truncation_psi=truncation_psi, truncation_cutoff=truncation_cutoff, label=label)
        img = self.synthesis(ws, **synthesis_kwargs)
        return img

#----------------------------------------------------------------------------

@persistence.persistent_class
class DiscriminatorBlock(torch.nn.Module):
    def __init__(self,
        in_channels,                        # Number of input channels, 0 = first block.
        tmp_channels,                       # Number of intermediate channels.
        out_channels,                       # Number of output channels.
        resolution,                         # Resolution of this block.
        img_channels,                       # Number of input color channels.
        first_layer_idx,                    # Index of the first layer.
        architecture        = 'resnet',     # Architecture: 'orig', 'skip', 'resnet'.
        activation          = 'lrelu',      # Activation function: 'relu', 'lrelu', etc.
        resample_filter     = [1,3,3,1],    # Low-pass filter to apply when resampling activations.
        conv_clamp          = None,         # Clamp the output of convolution layers to +-X, None = disable clamping.
        use_fp16            = False,        # Use FP16 for this block?
        fp16_channels_last  = False,        # Use channels-last memory format with FP16?
        freeze_layers       = 0,            # Freeze-D: Number of layers to freeze.
    ):
        assert in_channels in [0, tmp_channels]
        assert architecture in ['orig', 'skip', 'resnet']
        super().__init__()
        self.in_channels = in_channels
        self.resolution = resolution
        self.img_channels = img_channels
        self.first_layer_idx = first_layer_idx
        self.architecture = architecture
        self.use_fp16 = use_fp16
        self.channels_last = (use_fp16 and fp16_channels_last)
        self.register_buffer('resample_filter', upfirdn2d.setup_filter(resample_filter))

        self.num_layers = 0
        def trainable_gen():
            while True:
                layer_idx = self.first_layer_idx + self.num_layers
                trainable = (layer_idx >= freeze_layers)
                self.num_layers += 1
                yield trainable
        trainable_iter = trainable_gen()

        if in_channels == 0 or architecture == 'skip':
            self.fromrgb = Conv2dLayer(img_channels, tmp_channels, kernel_size=1, activation=activation,
                trainable=next(trainable_iter), conv_clamp=conv_clamp, channels_last=self.channels_last)

        self.conv0 = Conv2dLayer(tmp_channels, tmp_channels, kernel_size=3, activation=activation,
            trainable=next(trainable_iter), conv_clamp=conv_clamp, channels_last=self.channels_last)

        self.conv1 = Conv2dLayer(tmp_channels, out_channels, kernel_size=3, activation=activation, down=2,
            trainable=next(trainable_iter), resample_filter=resample_filter, conv_clamp=conv_clamp, channels_last=self.channels_last)

        if architecture == 'resnet':
            self.skip = Conv2dLayer(tmp_channels, out_channels, kernel_size=1, bias=False, down=2,
                trainable=next(trainable_iter), resample_filter=resample_filter, channels_last=self.channels_last)

    def forward(self, x, img, force_fp32=False):
        dtype = torch.float16 if self.use_fp16 and not force_fp32 else torch.float32
        memory_format = torch.channels_last if self.channels_last and not force_fp32 else torch.contiguous_format

        # Input.
        if x is not None:
            misc.assert_shape(x, [None, self.in_channels, self.resolution, self.resolution])
            x = x.to(dtype=dtype, memory_format=memory_format)

        # FromRGB.
        if self.in_channels == 0 or self.architecture == 'skip':
            misc.assert_shape(img, [None, self.img_channels, self.resolution, self.resolution])
            img = img.to(dtype=dtype, memory_format=memory_format)
            y = self.fromrgb(img)
            x = x + y if x is not None else y
            img = upfirdn2d.downsample2d(img, self.resample_filter) if self.architecture == 'skip' else None

        # Main layers.
        if self.architecture == 'resnet':
            y = self.skip(x, gain=np.sqrt(0.5))
            x = self.conv0(x)
            x = self.conv1(x, gain=np.sqrt(0.5))
            x = y.add_(x)
        else:
            x = self.conv0(x)
            x = self.conv1(x)

        assert x.dtype == dtype
        return x, img

#----------------------------------------------------------------------------

@persistence.persistent_class
class MinibatchStdLayer(torch.nn.Module):
    def __init__(self, group_size, num_channels=1):
        super().__init__()
        self.group_size = group_size
        self.num_channels = num_channels

    def forward(self, x):
        N, C, H, W = x.shape
        with misc.suppress_tracer_warnings(): # as_tensor results are registered as constants
            G = torch.min(torch.as_tensor(self.group_size), torch.as_tensor(N)) if self.group_size is not None else N
        F = self.num_channels
        c = C // F

        y = x.reshape(G, -1, F, c, H, W)    # [GnFcHW] Split minibatch N into n groups of size G, and channels C into F groups of size c.
        y = y - y.mean(dim=0)               # [GnFcHW] Subtract mean over group.
        y = y.square().mean(dim=0)          # [nFcHW]  Calc variance over group.
        y = (y + 1e-8).sqrt()               # [nFcHW]  Calc stddev over group.
        y = y.mean(dim=[2,3,4])             # [nF]     Take average over channels and pixels.
        y = y.reshape(-1, F, 1, 1)          # [nF11]   Add missing dimensions.
        y = y.repeat(G, 1, H, W)            # [NFHW]   Replicate over group and pixels.
        x = torch.cat([x, y], dim=1)        # [NCHW]   Append to input as new channels.
        return x

#----------------------------------------------------------------------------

@persistence.persistent_class
class DiscriminatorEpilogue(torch.nn.Module):
    def __init__(self,
        in_channels,                    # Number of input channels.
        cmap_dim,                       # Dimensionality of mapped conditioning label, 0 = no label.
        resolution,                     # Resolution of this block.
        img_channels,                   # Number of input color channels.
        architecture        = 'resnet', # Architecture: 'orig', 'skip', 'resnet'.
        mbstd_group_size    = 4,        # Group size for the minibatch standard deviation layer, None = entire minibatch.
        mbstd_num_channels  = 1,        # Number of features for the minibatch standard deviation layer, 0 = disable.
        activation          = 'lrelu',  # Activation function: 'relu', 'lrelu', etc.
        conv_clamp          = None,     # Clamp the output of convolution layers to +-X, None = disable clamping.
    ):
        assert architecture in ['orig', 'skip', 'resnet']
        super().__init__()
        self.in_channels = in_channels
        self.cmap_dim = cmap_dim
        self.resolution = resolution
        self.img_channels = img_channels
        self.architecture = architecture

        if architecture == 'skip':
            self.fromrgb = Conv2dLayer(img_channels, in_channels, kernel_size=1, activation=activation)
        self.mbstd = MinibatchStdLayer(group_size=mbstd_group_size, num_channels=mbstd_num_channels) if mbstd_num_channels > 0 else None
        self.conv = Conv2dLayer(in_channels + mbstd_num_channels, in_channels, kernel_size=3, activation=activation, conv_clamp=conv_clamp)
        self.fc = FullyConnectedLayer(in_channels * (resolution ** 2), in_channels, activation=activation)
        self.out = FullyConnectedLayer(in_channels, cmap_dim)

    def forward(self, x, img, cmap, force_fp32=False):
        misc.assert_shape(x, [None, self.in_channels, self.resolution, self.resolution]) # [NCHW]
        _ = force_fp32 # unused
        dtype = torch.float32
        memory_format = torch.contiguous_format

        # FromRGB.
        x = x.to(dtype=dtype, memory_format=memory_format)
        if self.architecture == 'skip':
            misc.assert_shape(img, [None, self.img_channels, self.resolution, self.resolution])
            img = img.to(dtype=dtype, memory_format=memory_format)
            x = x + self.fromrgb(img)

        # Main layers.
        if self.mbstd is not None:
            x = self.mbstd(x)
        x = self.conv(x)
        x = self.fc(x.flatten(1))
        x = self.out(x)

        # Conditioning.
        if self.cmap_dim > 0:
            misc.assert_shape(cmap, [None, self.cmap_dim])
            x = (x * cmap).sum(dim=1, keepdim=True) * (1 / np.sqrt(self.cmap_dim))
        assert x.dtype == dtype
        return x

#----------------------------------------------------------------------------

@persistence.persistent_class
class Discriminator(torch.nn.Module):
    def __init__(self,
        c_dim,                          # Conditioning label (C) dimensionality.
        img_resolution,                 # Input resolution.
        img_channels,                   # Number of input color channels.
        architecture        = 'resnet', # Architecture: 'orig', 'skip', 'resnet'.
        channel_base        = 32768,    # Overall multiplier for the number of channels.
        channel_max         = 512,      # Maximum number of channels in any layer.
        num_fp16_res        = 0,        # Use FP16 for the N highest resolutions.
        conv_clamp          = None,     # Clamp the output of convolution layers to +-X, None = disable clamping.
        cmap_dim            = None,     # Dimensionality of mapped conditioning label, None = default.
        block_kwargs        = {},       # Arguments for DiscriminatorBlock.
        mapping_kwargs      = {},       # Arguments for MappingNetwork.
        epilogue_kwargs     = {},       # Arguments for DiscriminatorEpilogue.
    ):
        super().__init__()
        self.c_dim = c_dim
        self.img_resolution = img_resolution
        self.img_resolution_log2 = int(np.log2(img_resolution))
        self.img_channels = img_channels
        self.block_resolutions = [2 ** i for i in range(self.img_resolution_log2, 2, -1)]
        channels_dict = {res: min(channel_base // res, channel_max) for res in self.block_resolutions + [4]}
        fp16_resolution = max(2 ** (self.img_resolution_log2 + 1 - num_fp16_res), 8)

        if cmap_dim is None:
            cmap_dim = channels_dict[4]
        if c_dim == 0:
            cmap_dim = 0
        common_kwargs = dict(img_channels=img_channels, architecture=architecture, conv_clamp=conv_clamp)
        cur_layer_idx = 0
        for res in self.block_resolutions:
            in_channels = channels_dict[res] if res < img_resolution else 0
            tmp_channels = channels_dict[res]
            out_channels = channels_dict[res // 2]
            use_fp16 = (res >= fp16_resolution)
            block = DiscriminatorBlock(in_channels, tmp_channels, out_channels, resolution=res,
                first_layer_idx=cur_layer_idx, use_fp16=use_fp16, **block_kwargs, **common_kwargs)
            setattr(self, f'b{res}', block)
            cur_layer_idx += block.num_layers

        self.causal_nodes = [1, 1, 1, 1, 2, 12, 5, 2, 2, 8, 2, 2, 2, 2, 2, 2, 9, 1, 2, 2, 4, 2, 2, 2, 4, 2, 2, 2, 2, 4,
                             4, 2, 4, 2, 3, 2, 3]
        self.cum_causal_nodes = [0, 1, 2, 3, 4, 6, 18, 23, 25, 27, 35, 37, 39, 41, 43, 45, 47, 56, 57, 59, 61, 65, 67,
                                 69, 71, 75, 77, 79, 81, 83, 87, 91, 93, 97, 99, 102, 104, 107]
        self.ori_c_dim = 37
        if c_dim > 0:
            self.embeder = ParallelFullyConnectedLayer(1, 16, num_mlps=c_dim)
            self.mapping = MappingNetwork(z_dim=0, c_dim=self.ori_c_dim*16, w_dim=cmap_dim, num_ws=None, w_avg_beta=None, **mapping_kwargs)

        self.b4 = DiscriminatorEpilogue(channels_dict[4], cmap_dim=cmap_dim, resolution=4, **epilogue_kwargs, **common_kwargs)

    def forward(self, img, c, **block_kwargs):
        x = None
        c = (c>0).float()
        for res in self.block_resolutions:
            block = getattr(self, f'b{res}')
            x, img = block(x, img, **block_kwargs)
        cmap = []
        if self.c_dim > 0:
            cin = F.one_hot(c.long(), num_classes=2).float()
            assert cin.size() == (len(c), self.c_dim, 2)
            cembeds = self.embeder(cin)
            for i in range(self.ori_c_dim):
                start_dim = self.cum_causal_nodes[i]
                end_dim = self.cum_causal_nodes[i+1]
                cur_label = c[:, start_dim:end_dim].unsqueeze(-1) # B,T,1
                cur_embed = cembeds[:,start_dim:end_dim] # B,T,16
                cur_cmap = torch.sum(cur_embed*cur_label, dim=1)
                assert cur_cmap.size() == (len(c), 16)
            cmap.append(normalize_2nd_moment(cur_cmap))
            cmap = torch.cat(cmap, 1)
            cmap = self.mapping(None, cmap)

        x = self.b4(x, img, cmap)
        return x

#----------------------------------------------------------------------------
