

import torch
from torch import nn

import torchvision.transforms as T
from torchvision.transforms.functional import InterpolationMode
from transformers import AutoTokenizer, AutoModel, Blip2ForConditionalGeneration, Blip2Processor
from .insta_flow import RectifiedFlowPipeline

IMAGENET_MEAN = (0.485, 0.456, 0.406)
IMAGENET_STD = (0.229, 0.224, 0.225)

def build_transform(input_size):
    MEAN, STD = IMAGENET_MEAN, IMAGENET_STD
    transform = T.Compose([
        T.Lambda(lambda img: img.convert('RGB') if img.mode != 'RGB' else img),
        T.Resize((input_size, input_size), interpolation=InterpolationMode.BICUBIC),
        T.ToTensor(),
        T.Normalize(mean=MEAN, std=STD)
    ])
    return transform

def find_closest_aspect_ratio(aspect_ratio, target_ratios, width, height, image_size):
    best_ratio_diff = float('inf')
    best_ratio = (1, 1)
    area = width * height
    for ratio in target_ratios:
        target_aspect_ratio = ratio[0] / ratio[1]
        ratio_diff = abs(aspect_ratio - target_aspect_ratio)
        if ratio_diff < best_ratio_diff:
            best_ratio_diff = ratio_diff
            best_ratio = ratio
        elif ratio_diff == best_ratio_diff:
            if area > 0.5 * image_size * image_size * ratio[0] * ratio[1]:
                best_ratio = ratio
    return best_ratio

def dynamic_preprocess(image, min_num=1, max_num=12, image_size=448, use_thumbnail=False):
    orig_width, orig_height = image.size
    aspect_ratio = orig_width / orig_height

    # calculate the existing image aspect ratio
    target_ratios = set(
        (i, j) for n in range(min_num, max_num + 1) for i in range(1, n + 1) for j in range(1, n + 1) if
        i * j <= max_num and i * j >= min_num)
    target_ratios = sorted(target_ratios, key=lambda x: x[0] * x[1])

    # find the closest aspect ratio to the target
    target_aspect_ratio = find_closest_aspect_ratio(
        aspect_ratio, target_ratios, orig_width, orig_height, image_size)

    # calculate the target width and height
    target_width = image_size * target_aspect_ratio[0]
    target_height = image_size * target_aspect_ratio[1]
    blocks = target_aspect_ratio[0] * target_aspect_ratio[1]

    # resize the image
    resized_img = image.resize((target_width, target_height))
    processed_images = []
    for i in range(blocks):
        box = (
            (i % (target_width // image_size)) * image_size,
            (i // (target_width // image_size)) * image_size,
            ((i % (target_width // image_size)) + 1) * image_size,
            ((i // (target_width // image_size)) + 1) * image_size
        )
        # split the image
        split_img = resized_img.crop(box)
        processed_images.append(split_img)
    assert len(processed_images) == blocks
    if use_thumbnail and len(processed_images) != 1:
        thumbnail_img = image.resize((image_size, image_size))
        processed_images.append(thumbnail_img)
    return processed_images

def load_image(image, input_size=448, max_num=12):
    # image = Image.open(image_file).convert('RGB')
    transform = build_transform(input_size=input_size)
    images = dynamic_preprocess(image, image_size=input_size, use_thumbnail=True, max_num=max_num)
    pixel_values = [transform(image) for image in images]
    pixel_values = torch.stack(pixel_values)
    return pixel_values

class MainObjectExtractor(nn.Module):
    def __init__(
            self,
            extra_model_name="OpenGVLab/InternVL3-2B",
            rf_model_name="XCLIU/instaflow_0_9B_from_sd_1_5",
            dtype=torch.bfloat16,
            ):
        super().__init__()
        if 'blip' in extra_model_name.lower():
            self.extra_processor = Blip2Processor.from_pretrained(extra_model_name)
            self.extra_model = Blip2ForConditionalGeneration.from_pretrained(
                extra_model_name,
                torch_dtype=dtype,
                low_cpu_mem_usage=True,
                trust_remote_code=True).eval()
        else:
            self.extra_tokenizer = AutoTokenizer.from_pretrained(extra_model_name, trust_remote_code=True, use_fast=False)
            self.extra_model = AutoModel.from_pretrained(
                extra_model_name,
                torch_dtype=dtype,
                low_cpu_mem_usage=True,
                use_flash_attn=True,
                trust_remote_code=True).eval()
        if rf_model_name is not None:
            self.rf_model = RectifiedFlowPipeline.from_pretrained(rf_model_name, torch_dtype=dtype, safety_checker=None)
            self.rf_model.set_progress_bar_config(leave=False)
        else:
            self.rf_model = None
        self.dtype = dtype
        self.device = self.extra_model.device

    def to(self, *args, **kwargs):
        if self.rf_model is not None:
            self.rf_model.to(*args, **kwargs)
        self.extra_model.to(*args, **kwargs)
        super(MainObjectExtractor, self).to(*args, **kwargs)

    @torch.no_grad()
    def forward(self, prompts, **kwargs):
        if 'images' in kwargs:
            images = kwargs['images']
        else:
            images = self.rf_model(
                prompts, 
                num_inference_steps=1, 
                guidance_scale=0.0,
            ).images
        
        device = self.extra_model.device
        dtype = self.extra_model.dtype

        if 'blip' in self.extra_model.config._name_or_path.lower():
            questions = ['Describe the main object in 1-3 words.'] * len(images)
            inputs = self.extra_processor(images=images, text=questions, return_tensors="pt").to(device, dtype=dtype)
            outs = self.extra_model.generate(
                **inputs,
            )
            responses = self.extra_processor.batch_decode(outs, skip_special_tokens=True)
            responses = [r.split('Describe the main object in 1-3 words.')[1].strip() for r in responses]
        else:
            pixel_values = [load_image(i).to(device, dtype=dtype) for i in images]
            num_patches_list = [p.size(0) for p in pixel_values]
            pixel_values = torch.cat(pixel_values, dim=0).to(device, dtype=dtype)

            questions = ['<image>\nDescribe the main object in 1-3 words.'] * len(num_patches_list)
            generation_config = dict(max_new_tokens=1024, do_sample=True)
    
        
            responses = self.extra_model.batch_chat(self.extra_tokenizer, pixel_values,
                                        num_patches_list=num_patches_list,
                                        questions=questions,
                                        generation_config=generation_config)
        return responses

