import gradio as gr
import os
import re
import time
from Split_User_Question_Agent import split_question
from Workflow_Solution_Agent import solution
from embedding_test import Embedding
from Select_Agent import Select_agent
from Commandline_Generate_Agent import commandline_agent
from Judger_Agent import judger
from Debuger_Agent import debuger
import subprocess
DATA_DIR = "./output"

state = {
    "analysis_goal": None,
    "input_files": None,
    "output_files": None,
    "current_used_tools": [],
    "current_available_output_files": [],
    "installation_command": [],
    "setup_command": [],
    "execution_command": [],
    "tool_name": None,
    "step_ready": False,
    "workflow_done": False,
    "selected_input_files": None,
    "expected_outputs_info": None,
    "try_count": 0,
    "chat": [] 
}

def extract_xml_help(xml_path):
    try:
        if not os.path.exists(xml_path):
            return ''
        with open(xml_path, 'r', encoding='utf-8') as f:
            xml_content = f.read()
        xml_content = re.sub(r'<!\[CDATA\[', '', xml_content)
        xml_content = re.sub(r'\]\]>', '', xml_content)
        match = re.search(r'<command(.*?)>(.*?)</command>', xml_content, re.DOTALL)
        if match:
            return match.group(2).strip()
        else:
            return ''
    except:
        return ''

def gain_command(file_os_path):
    description = []
    filedata_list = os.listdir(file_os_path)
    for filedata in filedata_list:
        if filedata.endswith('.xml'):
            data = extract_xml_help(os.path.join(file_os_path, filedata))
            description.append(data)
    return description

import os

def get_bottom_files(root_dir):
    bottom_files = []
    for dirpath, dirnames, filenames in os.walk(root_dir):
        if not dirnames:
            for f in filenames:
                bottom_files.append(os.path.join(dirpath, f))
    return bottom_files

def _append_and_yield(log_text):
    state.setdefault("chat", [])
    state["chat"].append((None, log_text))  
    yield state["chat"], state
    time.sleep(0.05)  


def _append_many_and_yield(lines):
    for line in lines:
        yield from _append_and_yield(line)

def process_question(user_question, files):
    state["chat"] = []
    filename_list = get_bottom_files(DATA_DIR)
    yield from _append_many_and_yield(["📝 Parsing your question..."])
    question_split = split_question.split(user_question, filename_list)
    if not question_split:
        yield from _append_many_and_yield(["❌ Failed to parse question"])
        return

    state["analysis_goal"] = question_split["analysis_goal"]
    state["input_files"] = question_split["input_files"]
    state["output_files"] = question_split["output_files"]
    state["current_available_output_files"] = filename_list
    state["current_used_tools"] = []
    state["workflow_done"] = False

    yield from _append_many_and_yield([
        f"🎯 Analysis goal: {state['analysis_goal']}",
        f"💾 Input requirement: {state['input_files']}",
        f"💾 Output requirement: {state['output_files']}",
        "✅ Question parsed. Click [Next Step] to start the workflow."
    ])


def next_step():
    if state["workflow_done"]:
        yield from _append_many_and_yield(["⚠️ Workflow is already completed!"])
        return

    yield from _append_many_and_yield(["🔍 Selecting the next tool..."])

    workflow_solution = solution.workflow_solution(
        state["analysis_goal"],
        state["input_files"],
        state["output_files"],
        state["current_used_tools"],
        state["current_available_output_files"]
    )

    if not workflow_solution:
        yield from _append_many_and_yield(["❌ Failed to get workflow solution"])
        return

    toolname = workflow_solution["toolname"]
    function = workflow_solution["function"]
    description = workflow_solution["description"]
    inputformat = workflow_solution.get("inputformat")
    outputformat = workflow_solution.get("outputformat")

    yield from _append_many_and_yield([
        f"📍 Candidate tool: {toolname}, function: {function}",
        "🔮 Searching similar tools in the knowledge base..."
    ])

    conference_tool_list = Embedding.similarity(
        f"tool name:{toolname},tool function:{function},tool description:{description}"
    )
    conference = [
        {"toolname": tool["toolname"], "description": tool["description"]}
        for tool in conference_tool_list
    ]

    top_similar = ", ".join([c["toolname"] for c in conference[:5]]) if conference else "(none)"
    yield from _append_many_and_yield([
        f"⭐Top similar tools: {top_similar}",
        "🔍 Selecting the best tool..."
    ])

    select_tool = Select_agent.select_tool(
        state["analysis_goal"],
        state["current_used_tools"],
        state["current_available_output_files"],
        f"tool name:{toolname},tool function:{function},tool description:{description}",
        conference
    )

    if not select_tool:
        yield from _append_many_and_yield(["❌ Tool selection failed"])
        return

    toolname = select_tool["toolname"]
    description = select_tool["description"]
    used_reference_tool = select_tool["used_reference_tool"]
    toolid = select_tool["toolid"]
    selected_input_files = select_tool["selected_input_files"]
    expected_outputs_info = select_tool["expected_outputs_info"]

    yield from _append_many_and_yield([
        f"✅ Selected tool: {toolname}, description: {description}",
        "⚙️ Generating command line..."
    ])

    if used_reference_tool:
        commandline_reference = gain_command(conference_tool_list[toolid]["toolid"])  
    else:
        commandline_reference = ""

    commandline_of_tool = commandline_agent.generate_commandline(
        toolname, description, commandline_reference,
        state["analysis_goal"], state["current_used_tools"],
        selected_input_files, expected_outputs_info,[],state["current_available_output_files"]
    )

    if not commandline_of_tool:
        yield from _append_many_and_yield(["❌ Command line generation failed"])
        return

    state["tool_name"] = commandline_of_tool["tool_name"]
    state["installation_command"] = commandline_of_tool["installation_command"]
    state["setup_command"] = commandline_of_tool["setup_command"]
    state["execution_command"] = commandline_of_tool["execution_command"]
    state["input_files"] = commandline_of_tool["input_files"]
    state["output_files"] = commandline_of_tool["output_files"]
    state["selected_input_files"] = selected_input_files
    state["expected_outputs_info"] = expected_outputs_info
    state["try_count"] = 0
    state["step_ready"] = True

    yield from _append_many_and_yield([
        "⚙️ Commands generated. Click [Run Command] to execute.",
        f"📜 Installation commands: {state['installation_command']}",
        f"📜 Execution commands: {state['execution_command']}"
    ])

def run_step():
    if not state["step_ready"]:
        yield from _append_many_and_yield(["⚠️ No commands ready to execute"])
        return

    try:
        commands = [state["installation_command"], state["setup_command"], state["execution_command"]]
        yield from _append_and_yield(f"▶️ Running the command: {commands}")
        joined_cmd = " && ".join(commands)
        full_command = f"source ~/miniconda3/etc/profile.d/conda.sh && {joined_cmd}"

        result = subprocess.run(
            ["bash", "-c", full_command],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
            check=True
        )

        yield from _append_many_and_yield(["✅ Step executed successfully"])

        state["current_used_tools"].append(state["tool_name"]) if state.get("tool_name") else None
        state["current_available_output_files"]=get_bottom_files(DATA_DIR)

        judge_result = judger.judge(
            state["analysis_goal"],
            state["input_files"],
            state["output_files"],
            state["current_used_tools"],
            state["current_available_output_files"]
        )
        if judge_result:
            yield from _append_many_and_yield(["🎉 Workflow is completed!"])
            state["workflow_done"] = True
        else:
            yield from _append_many_and_yield(["➡️ Click [Next Step] to continue with the workflow."])

    except Exception as e:
        yield from _append_many_and_yield([f"❌ Error occurred: {str(e)}"])
        state["try_count"] += 1
        if state["try_count"] <= 4:
            yield from _append_many_and_yield(["🛠️ Calling Debuger to fix the issue..."])
            debug_result = debuger.debug(
                state["tool_name"], state["tool_name"],
                state["analysis_goal"], state["current_used_tools"],
                state["selected_input_files"], state["expected_outputs_info"],
                state["installation_command"], state["execution_command"], str(e)
            )
            if debug_result:
                state["installation_command"] = debug_result.get("installation_command", state["installation_command"])
                state["execution_command"] = debug_result.get("execution_command", state["execution_command"])
                state["step_ready"] = True
                yield from _append_many_and_yield(["✅ Debuger generated new commands. Click [Run Command] to retry."])
            else:
                yield from _append_many_and_yield(["⚠️ Debuger did not return new commands. Retrying may continue..."])
        else:
            yield from _append_many_and_yield(["❌ Failed after multiple retries. Stopping workflow."])
            state["workflow_done"] = True
    finally:
        state["step_ready"] = False if state["try_count"] == 0 else True

def get_server_files():
    if not os.path.exists(DATA_DIR):
        return []
    return [f for f in os.listdir(DATA_DIR)]
with gr.Blocks() as demo:
    gr.Markdown("# 🧬 Bioinformatics Workflow Assistant 🧬")

    with gr.Row():
        with gr.Column():
            user_question = gr.Textbox(
    label="Enter your question",
    lines=5,   
    placeholder="Type your question here..."
)
            server_files = gr.CheckboxGroup(choices=[f for f in os.listdir(DATA_DIR)], label="Select server files")
            start_button = gr.Button("Start Analysis")
            next_button = gr.Button("Next Step")
            run_button = gr.Button("Run Command", variant="primary")

        with gr.Column():
            chatbox = gr.Chatbot(label="Execution Logs",height=750)
    start_button.click(
        process_question,
        inputs=[user_question, server_files],
        outputs=[chatbox, gr.State(state)]
    )

    next_button.click(
        next_step,
        inputs=[],
        outputs=[chatbox, gr.State(state)]
    )

    run_button.click(
        run_step,
        inputs=[],
        outputs=[chatbox, gr.State(state)]
    )


demo.launch(server_name="0.0.0.0", server_port=7861)
