from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import torch.nn as nn
import logging
import torch
import models.mfvilayers as mfvilayers


def finitialize(modules):
    logging.info("Initializing MSRA")
    for layer in modules:
        if isinstance(layer, (mfvilayers.Conv2dMFMC, mfvilayers.LinearMFMC, nn.Conv2d, nn.Linear)):
            nn.init.kaiming_normal_(layer.weight)
            if layer.bias is not None:
                nn.init.constant_(layer.bias, 0)


def conv3x3(in_planes, out_planes, stride=1, groups=1, dilation=1):
    """3x3 convolution with padding"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=dilation, groups=groups, bias=False, dilation=dilation)

def conv3x3Bayes(in_planes, out_planes, stride=1, groups=1, dilation=1, prior_precision=1e0, var_dampening=1e-4):
    """3x3 convolution with padding"""
    return mfvilayers.Conv2dMFMC(in_planes, out_planes, kernel_size=3, stride=stride,
         padding=dilation, groups=groups, bias=False, dilation=dilation,
         prior_precision=prior_precision, var_dampening=var_dampening)


def conv1x1(in_planes, out_planes, stride=1):
    """1x1 convolution"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)

def conv1x1Bayes(in_planes, out_planes, stride=1, prior_precision=1e0, var_dampening=1e-4):
    """1x1 convolution"""
    return mfvilayers.Conv2dMFMC(in_planes, out_planes, kernel_size=1, stride=stride, bias=False,
        prior_precision=prior_precision, var_dampening=var_dampening)

class BasicBlock(nn.Module):
    expansion = 1
    __constants__ = ['downsample']

    def __init__(self, inplanes, planes, stride=1, downsample=None, groups=1,
                 base_width=64, dilation=1, norm_layer=None, relu=True, l2_norm=False):
        super(BasicBlock, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        if groups != 1 or base_width != 64:
            raise ValueError('BasicBlock only supports groups=1 and base_width=64')
        if dilation > 1:
            raise NotImplementedError("Dilation > 1 not supported in BasicBlock")
        # Both self.conv1 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = norm_layer(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = norm_layer(planes)
        self.downsample = downsample
        self.stride = stride
        self._relu = relu
        self.l2_norm = l2_norm

    def forward(self, x):
        if self.l2_norm:
            x = x / torch.sqrt(x.pow(2).sum(dim=(1,2,3)) + 1e-5).reshape(-1,1,1,1)
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        if self._relu:
            out = self.relu(out)

        return out


class Bottleneck(nn.Module):
    expansion = 4
    __constants__ = ['downsample']

    def __init__(self, inplanes, planes, stride=1, downsample=None, groups=1,
                 base_width=64, dilation=1, norm_layer=None, relu=True, l2_norm=False):
        super(Bottleneck, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        width = int(planes * (base_width / 64.)) * groups
        # Both self.conv2 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv1x1(inplanes, width)
        self.bn1 = norm_layer(width)
        self.conv2 = conv3x3(width, width, stride, groups, dilation)
        self.bn2 = norm_layer(width)
        self.conv3 = conv1x1(width, planes * self.expansion)
        self.bn3 = norm_layer(planes * self.expansion)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride
        self._relu = relu
        self.l2_norm = l2_norm

    def forward(self, x):
        if self.l2_norm:
            x = x / torch.sqrt(x.pow(2).sum(dim=(1,2,3)) + 1e-5).reshape(-1,1,1,1)
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        if self._relu:
            out = self.relu(out)

        return out


class ResNetLLNCVIBN(nn.Module):

    def __init__(self, block, layers, num_classes=100, prior_precision=1e0, var_dampening=1e-4, groups=1,
                width_per_group=64, replace_stride_with_dilation=None, l2_norm=False, maxpool=False,
                initial_stride=1, initial_kernel_size=3, initial_padding=1):
        super(ResNetLLNCVIBN, self).__init__()
        self._norm_layer = nn.BatchNorm2d
        self._maxpool = maxpool

        self.inplanes = 64
        self.dilation = 1
        if replace_stride_with_dilation is None:
            # each element in the tuple indicates if we should replace
            # the 2x2 stride with a dilated convolution instead
            replace_stride_with_dilation = [False, False, False]
        if len(replace_stride_with_dilation) != 3:
            raise ValueError("replace_stride_with_dilation should be None "
                             "or a 3-element tuple, got {}".format(replace_stride_with_dilation))
        self.groups = groups
        self.base_width = width_per_group
        self.conv1 = nn.Conv2d(3, self.inplanes, kernel_size=initial_kernel_size, stride=initial_stride, 
                                padding=initial_padding, bias=False)
        self.bn1 = self._norm_layer(self.inplanes)
        self.relu = nn.ReLU()
        if self._maxpool:
            self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2 = self._make_layer(block, 128, layers[1], stride=2,
                                       dilate=replace_stride_with_dilation[0])
        self.layer3 = self._make_layer(block, 256, layers[2], stride=2,
                                       dilate=replace_stride_with_dilation[1])
        self.layer4 = self._make_layer(block, 512, layers[3], stride=2,
                                       dilate=replace_stride_with_dilation[2], relu=False, l2_norm=l2_norm)
        self.avgpool = nn.AdaptiveAvgPool2d((1,1))
        if block == 'BasicBlock':
            block = BasicBlock
        elif block == 'Bottleneck':
            block = Bottleneck
        self.decoder_fc = mfvilayers.LinearMFMC(512 * block.expansion, num_classes, 
            var_dampening=var_dampening, prior_precision=prior_precision)

        finitialize(self.modules())

    def _make_layer(self, block_name, planes, blocks, stride=1, dilate=False, relu=True, l2_norm=False):
        if block_name == 'BasicBlock':
            block = BasicBlock
        elif block_name == 'Bottleneck':
            block = Bottleneck
        norm_layer = self._norm_layer
        downsample = None
        previous_dilation = self.dilation
        if dilate:
            self.dilation *= stride
            stride = 1
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                conv1x1(self.inplanes, planes * block.expansion, stride),
                norm_layer(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample, self.groups,
                            self.base_width, previous_dilation, norm_layer))
        self.inplanes = planes * block.expansion
        for _ in range(1, blocks - 1):
            layers.append(block(self.inplanes, planes, groups=self.groups,
                                base_width=self.base_width, dilation=self.dilation,
                                norm_layer=norm_layer))

        layers.append(block(self.inplanes, planes, groups=self.groups,
                                base_width=self.base_width, dilation=self.dilation,
                                norm_layer=norm_layer, relu=relu, l2_norm=l2_norm))

        return nn.Sequential(*layers)

    def _forward_impl(self, x):
        # See note [TorchScript super()]
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        if self._maxpool:
            x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        z = self.layer4(x)
        z = self.avgpool(z)
        z = torch.flatten(z, 1)
        return z

    def forward(self, x):
        return self._forward_impl(x)

    def decoder(self, z):
        z = self.relu(z)
        z = self.decoder_fc(z)
        return z

    def kl_div(self):
        kl = 0.0
        for module in self.modules():
            if isinstance(module, (mfvilayers.LinearMFMC, mfvilayers.Conv2dMFMC)):
                kl += module.kl_div()
        return kl


class ResNet18LLNCVIBN(nn.Module):
    def __init__(self, num_classes=100, prior_precision=1e0, var_dampening=1e-4, l2_norm=True, training_samples=1, test_samples=16, gamma=1.0, maxpool=False, **kwargs):
        super(ResNet18LLNCVIBN, self).__init__()
        self._gamma = gamma
        self._training_samples = training_samples
        self._test_samples = test_samples
        self.resnet = ResNetLLNCVIBN('BasicBlock',  [2, 2, 2, 2], num_classes=num_classes, prior_precision=prior_precision,
            var_dampening=var_dampening, l2_norm=l2_norm, maxpool=maxpool, initial_stride=1, **kwargs)

    def forward(self, example_dict):
        if self.training:
            samples = self._training_samples
        else:
            samples = self._test_samples
            
        inputs = example_dict['input1']
        z = self.resnet(inputs)
        z_var =  torch.ones_like(z)
        latent_kld = 0.5 * (- torch.log(z_var + 1e-24) +  (z_var + z**2) - 1)
        vibn_loss = latent_kld.mean(dim=0).sum()

        z = z.repeat_interleave(samples, dim=0)
        z_var = z_var.repeat_interleave(samples, dim=0)

        normal_dist = torch.distributions.normal.Normal(torch.zeros_like(z), torch.ones_like(z))
        normals = normal_dist.sample()
        z_s = z + self._gamma * torch.sqrt(z_var + 1e-24) * normals
        prediction = self.decoder(z_s)

        normals = self._gamma * normal_dist.sample()
        l_noise = self.decoder(normals)

        return {'vibn_loss': vibn_loss, 'prediction': prediction, 'l_noise': l_noise, 'latent_kld': latent_kld, 'decoder_kld': self.kl_div()}

    def decoder(self, z):
        return self.resnet.decoder(z)

    def kl_div(self):
        return self.resnet.kl_div()


class ResNet50LLNCVIBN(nn.Module):
    def __init__(self, num_classes=100, prior_precision=1e0, var_dampening=1e-4,  l2_norm=True, training_samples=1, test_samples=16, gamma=1.0, maxpool=True, **kwargs):
        super(ResNet50LLNCVIBN, self).__init__()
        self._gamma = gamma
        self._training_samples = training_samples
        self._test_samples = test_samples
        self.resnet = ResNetLLNCVIBN('Bottleneck', [3, 4, 6, 3], num_classes=num_classes, prior_precision=prior_precision,
            var_dampening=var_dampening, l2_norm=l2_norm, maxpool=maxpool, initial_stride=2, initial_kernel_size=7, initial_padding=3, **kwargs)

    def forward(self, example_dict):
        if self.training:
            samples = self._training_samples
        else:
            samples = self._test_samples
            
        inputs = example_dict['input1']
        z = self.resnet(inputs)
        z_var =  torch.ones_like(z)
        latent_kld = 0.5 * (- torch.log(z_var + 1e-24) +  (z_var + z**2) - 1)
        vibn_loss = latent_kld.mean(dim=0).sum()

        z = z.repeat_interleave(samples, dim=0)
        z_var = z_var.repeat_interleave(samples, dim=0)

        normal_dist = torch.distributions.normal.Normal(torch.zeros_like(z), torch.ones_like(z))
        normals = normal_dist.sample()
        z_s = z + self._gamma * torch.sqrt(z_var + 1e-24) * normals
        prediction = self.decoder(z_s)

        normals = self._gamma * normal_dist.sample()
        l_noise = self.decoder(normals)

        return {'vibn_loss': vibn_loss, 'prediction': prediction, 'l_noise': l_noise, 'latent_kld': latent_kld, 'decoder_kld': self.kl_div()}

    def decoder(self, z):
        return self.resnet.decoder(z)

    def kl_div(self):
        return self.resnet.kl_div()