from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import torch
import torch.nn as nn

from models.resnet_vibn_base import ResNetVIBNBase


class ResNet18VIBN(nn.Module):
    def __init__(self, num_classes=100, l2_norm=True, training_samples=1, test_samples=16, gamma=1.0, maxpool=False, **kwargs):
        super(ResNet18VIBN, self).__init__()
        self._gamma = gamma
        self._training_samples = training_samples
        self._test_samples = test_samples
        self.resnet = ResNetVIBNBase('BasicBlock',  [2, 2, 2, 2], num_classes=num_classes, l2_norm=l2_norm, maxpool=maxpool, **kwargs)

    def forward(self, example_dict):
        if self.training:
            samples = self._training_samples
        else:
            samples = self._test_samples
            
        inputs = example_dict['input1']
        z, z_var = self.resnet(inputs)
        z_var = torch.exp(z_var)
        latent_kld = 0.5 * (- torch.log(z_var + 1e-24)  +  (z_var + z**2) - 1)
        vibn_loss = latent_kld.mean(dim=0).sum()

        z = z.repeat_interleave(samples, dim=0)
        z_var = z_var.repeat_interleave(samples, dim=0)

        normal_dist = torch.distributions.normal.Normal(torch.zeros_like(z), torch.ones_like(z))
        normals =  normal_dist.sample()
        z_s = z +  self._gamma * torch.sqrt(z_var + 1e-24) * normals
        prediction = self.decoder(z_s)

        normals = self._gamma  * normal_dist.sample()
        l_noise = self.decoder(normals)

        return {'vibn_loss': vibn_loss, 'prediction': prediction, 'l_noise': l_noise, 'latent_kld': latent_kld}

    def decoder(self, z):
        return self.resnet.decoder(z)