import matplotlib.pyplot as plt
import math
import numpy as np

def plot_regret(avg_regret, std_regret, T, exp_name='experiment', ylim=None):
    plt.figure(figsize=(10, 6))
    t_saved = np.arange(T)
    colors = ['b', 'g', 'r', 'c', 'm', 'y', 'k']
    styles = ['-', '--', '-.', ':']

    for i, alg_name in enumerate(avg_regret.keys()):
        avgR = avg_regret[alg_name]
        stdR = std_regret[alg_name]
        color = colors[i % len(colors)]
        style = styles[i % len(styles)]
        plt.plot(t_saved, avgR, linestyle=style, label=alg_name, color=color)

        low_bound = avgR - stdR
        low_bound[low_bound < 0] = 0 
        high_bound = avgR + stdR
        plt.fill_between(t_saved, low_bound, high_bound, alpha=0.15, color=color)

    plt.legend(loc='upper left', fontsize=13).get_frame().set_linewidth(0.5)
    plt.xlabel('Time', fontsize=16)
    plt.ylabel('Cumulative Regret', fontsize=16)
    if ylim is not None:
        plt.ylim(ylim)
    plt.title('Regret over Time', fontsize=18)
    plt.grid(True)
    plt.tight_layout()
    plt.savefig('fig/' + exp_name + '_regret' + '.pdf', format='pdf', bbox_inches='tight')
    plt.show()




def plot_times(avg_timings, exp_name='experiment'):
    """
    Plots the average running time per algorithm.
    """
    plt.figure(figsize=(8, 6))
    alg_names = list(avg_timings.keys())
    timings = [avg_timings[alg_name] for alg_name in alg_names]
    colors = ['b', 'g', 'r', 'c', 'm', 'y', 'k']

    bars = plt.bar(alg_names, timings, color=colors[:len(alg_names)])
    plt.xticks(rotation=30, ha='right')
    plt.ylabel('Average Time per Run (seconds)', fontsize=14)
    plt.title('Average Running Time per Algorithm', fontsize=16)

    for bar, timing in zip(bars, timings):
        height = bar.get_height()
        plt.text(bar.get_x() + bar.get_width() / 2, height, f'{timing:.2f}',
                 ha='center', va='bottom', fontsize=12)

    plt.tight_layout()
    plt.savefig('fig/' + exp_name + '_running_time' + '.pdf', format='pdf', bbox_inches='tight')
    plt.show()
