import math
import torch
from torch import nn

class TimeStepEmbedding(nn.Module):
    """
    Layer that embeds diffusion timesteps.
    
     Args:
        - dim (int): the dimension of the output.
        - max_period (int): controls the minimum frequency of the embeddings.
        - n_layers (int): number of dense layers
        - fourer (bool): whether to use random fourier features as embeddings
    """
    def __init__(
        self,
        dim: int,
        max_period: int = 10000,
        n_layers: int = 2,
        fourier: bool = False,
        scale=16,
    ):
        super().__init__()
        self.dim = dim
        self.max_period = max_period
        self.n_layers = n_layers
        self.fourier = fourier

        if dim % 2 != 0:
            raise ValueError(f"embedding dim must be even, got {dim}")

        if fourier:
            self.register_buffer("freqs", torch.randn(dim // 2) * scale)

        layers = []
        for i in range(n_layers - 1):
            layers.append(nn.Linear(dim, dim))
            layers.append(nn.SiLU())
        self.fc = nn.Sequential(*layers, nn.Linear(dim, dim))

    def forward(self, timesteps):
        if not self.fourier:
            d, T = self.dim, self.max_period
            mid = d // 2
            fs = torch.exp(-math.log(T) / mid * torch.arange(mid, dtype=torch.float32))
            fs = fs.to(timesteps.device)
            args = timesteps[:, None].float() * fs[None]
            emb = torch.cat([torch.cos(args), torch.sin(args)], dim=-1)
        else:
            x = timesteps.ger((2 * torch.pi * self.freqs).to(timesteps.dtype))
            emb = torch.cat([x.cos(), x.sin()], dim=1)

        return self.fc(emb)


class TabDDPM_MLP_Cont(nn.Module):
    """
    TabDDPM-like architecture for continuous features only.
    This is used for TabSyn as a score model for learned latents.
    """

    def __init__(self, num_features, emb_dim, n_layers, n_units, act="relu"):
        super().__init__()

        self.time_emb = TimeStepEmbedding(emb_dim, fourier=False)
        in_dims = [emb_dim] + (n_layers - 1) * [n_units]
        out_dims = n_layers * [n_units]
        layers = nn.ModuleList()
        for i in range(len(in_dims)):
            layers.append(nn.Linear(in_dims[i], out_dims[i]))
            layers.append(nn.ReLU() if act == "relu" else nn.SiLU())
        # add final layer
        layers.append(nn.Linear(out_dims[-1], num_features))
        self.fc = nn.Sequential(*layers)
        self.proj = nn.Linear(num_features, emb_dim)

    def forward(self, x, time):
        cond_emb = self.time_emb(time)
        x = self.proj(x) + cond_emb
        return self.fc(x)