#!/bin/bash

echo "Starting 3-seed training for Late Fusion model on fold 1..."

# Define experiment parameters
BATCH_SIZE=16
LR=0.0001
PATIENCE=10
EPOCHS=50
FOLD=1                     # Fixed fold 1
GPU=1                      # GPU device number
MATCHED=true
USE_DEMOGRAPHICS=false
CROSS_EVAL=""              # Set to "matched_to_full" or "full_to_matched" if needed

# Late Fusion specific parameters
EHR_ENCODER=transformer    # Options: 'lstm', 'transformer'
CXR_ENCODER=resnet50       # Options: 'resnet50', 'vit_b_16'
HIDDEN_SIZE=256            # Hidden dimension for both encoders
EHR_DROPOUT=0.2            # EHR encoder dropout
EHR_N_LAYERS=1             # Number of EHR encoder layers  
EHR_N_HEAD=4               # Number of attention heads for transformer
PRETRAINED=true            # Use pretrained CXR encoder
UNIMODAL_LOSS=false        # Whether to use unimodal loss

# Seeds configuration (including 42)
SEEDS=(42 123 1234)

# Tasks configuration
TASKS=("los")

# Function to determine log directory based on task and matched status
get_log_dir() {
    local TASK=$1
    local MATCHED=$2
    local MODEL="latefusion"  
    
    if [ "$MATCHED" = "true" ]; then
        echo "../experiments_${TASK}_matched/${MODEL}"
    else
        echo "../experiments_${TASK}_full/${MODEL}"
    fi
}

# Function to run training for a specific task
run_task_training() {
    local TASK=$1
    echo "Starting training for task: $TASK"
    
    # Determine log directory based on task and matched status
    LOG_DIR=$(get_log_dir $TASK $MATCHED)
    echo "Using log directory: $LOG_DIR"
    
    # Set task-specific parameters
    if [ "$TASK" = "phenotype" ]; then
        NUM_CLASSES=25
        INPUT_DIM=49
    elif [ "$TASK" = "mortality" ]; then
        NUM_CLASSES=1
        INPUT_DIM=49
    elif [ "$TASK" = "los" ]; then
        NUM_CLASSES=7
        INPUT_DIM=49
    else
        echo "Error: Unknown task $TASK"
        exit 1
    fi
    
    for SEED in "${SEEDS[@]}"
    do
        echo "Training with seed $SEED for task $TASK on fold $FOLD..."
        
        # Build base command
        CMD="python ../main.py \
            --model latefusion \
            --mode train \
            --task $TASK \
            --fold $FOLD \
            --batch_size $BATCH_SIZE \
            --lr $LR \
            --patience $PATIENCE \
            --epochs $EPOCHS \
            --seed $SEED \
            --ehr_encoder $EHR_ENCODER \
            --cxr_encoder $CXR_ENCODER \
            --hidden_size $HIDDEN_SIZE \
            --ehr_dropout $EHR_DROPOUT \
            --ehr_n_layers $EHR_N_LAYERS \
            --ehr_n_head $EHR_N_HEAD \
            --input_dim $INPUT_DIM \
            --num_classes $NUM_CLASSES \
            --log_dir $LOG_DIR \
            --gpu $GPU"
        
        # Add Late Fusion specific parameters
        if [ "$PRETRAINED" = "true" ]; then
            CMD="$CMD --pretrained"
        fi
        
        if [ "$UNIMODAL_LOSS" = "true" ]; then
            CMD="$CMD --unimodal_loss"
        fi
        
        # Add conditional parameters
        if [ "$MATCHED" = "true" ]; then
            CMD="$CMD --matched"
        fi
        
        if [ "$USE_DEMOGRAPHICS" = "true" ]; then
            CMD="$CMD --use_demographics"
        fi
        
        if [ -n "$CROSS_EVAL" ]; then
            CMD="$CMD --cross_eval $CROSS_EVAL"
        fi
        
        echo "Running command: $CMD"
        eval $CMD
        
        if [ $? -eq 0 ]; then
            echo "Seed $SEED for task $TASK completed successfully!"
        else
            echo "Error: Seed $SEED for task $TASK failed!"
            exit 1
        fi
    done
    
    echo "All 3 seeds completed for task $TASK on fold $FOLD!"
}

# Run training for each task
for TASK in "${TASKS[@]}"
do
    run_task_training $TASK
done

echo "All tasks completed successfully!"
echo "Training results are saved in the appropriate log directory based on task and matched status."