#!/bin/bash

echo "Starting 5-seed training for ShaSpec model on fold 1..."

# Define experiment parameters
BATCH_SIZE=16
LR=0.0001
PATIENCE=10
EPOCHS=50
FOLD=1                     # Fixed fold 1
GPU="4"                # Multiple GPU devices for parallel training
MATCHED=false
USE_DEMOGRAPHICS=false
CROSS_EVAL=""              # Set to "matched_to_full" or "full_to_matched" if needed

# ShaSpec specific parameters (based on best parameters from Bayesian search)
EHR_ENCODER=transformer    # Options: 'lstm', 'transformer'
CXR_ENCODER=resnet50       # Options: 'resnet50', 'vit_b_16'
DIM=256                    # Hidden feature dimension
DROPOUT=0.2              # Dropout rate
ALPHA=0.0539365721371763                # Shared feature consistency loss weight
BETA=0.02400870516077973                 # Domain classification loss weight
NHEAD=4                    # Number of attention heads for shared transformer
NUM_LAYERS=1               # Number of layers for shared transformer
MAX_SEQ_LEN=500            # Maximum sequence length

# EHR Encoder specific parameters
EHR_N_HEAD=4               # Number of attention heads for EHR transformer
EHR_N_LAYERS=1             # Number of layers for EHR transformer

# ShaSpec specific flags
WEIGHT_STD=true            # Use weight standardization
PRETRAINED=true            # Use pretrained CXR encoder

# Seeds configuration (including 42)
SEEDS=(42 123 1234)

# Tasks configuration
TASKS=("los")

for TASK in "${TASKS[@]}"
do
    echo "Training ShaSpec model for task: $TASK"
    
    # Set task-specific parameters
    if [ "$TASK" = "phenotype" ]; then
        NUM_CLASSES=25
        INPUT_DIM=49
    elif [ "$TASK" = "mortality" ]; then
        NUM_CLASSES=1
        INPUT_DIM=49
    elif [ "$TASK" = "los" ]; then
        NUM_CLASSES=7
        INPUT_DIM=49
    fi
    
    for SEED in "${SEEDS[@]}"
    do
        echo "Training with seed $SEED for task $TASK on fold $FOLD..."
        
        # Build base command
        CMD="python ../main.py \
            --model shaspec \
            --mode train \
            --task $TASK \
            --fold $FOLD \
            --batch_size $BATCH_SIZE \
            --lr $LR \
            --patience $PATIENCE \
            --epochs $EPOCHS \
            --seed $SEED \
            --ehr_encoder $EHR_ENCODER \
            --cxr_encoder $CXR_ENCODER \
            --dim $DIM \
            --dropout $DROPOUT \
            --alpha $ALPHA \
            --beta $BETA \
            --nhead $NHEAD \
            --num_layers $NUM_LAYERS \
            --max_seq_len $MAX_SEQ_LEN \
            --ehr_n_head $EHR_N_HEAD \
            --ehr_n_layers $EHR_N_LAYERS \
            --input_dim $INPUT_DIM \
            --num_classes $NUM_CLASSES \
            --use_label_weights \
            --gpu $GPU"
        
        # Add ShaSpec specific parameters
        if [ "$PRETRAINED" = "true" ]; then
            CMD="$CMD --pretrained"
        fi
        
        if [ "$WEIGHT_STD" = "true" ]; then
            CMD="$CMD --weight_std"
        fi
        
        # Add conditional parameters
        if [ "$MATCHED" = "true" ]; then
            CMD="$CMD --matched"
        fi
        
        if [ "$USE_DEMOGRAPHICS" = "true" ]; then
            CMD="$CMD --use_demographics"
        fi
        
        if [ -n "$CROSS_EVAL" ]; then
            CMD="$CMD --cross_eval $CROSS_EVAL"
        fi
        
        echo "Running command: $CMD"
        eval $CMD
        
        if [ $? -eq 0 ]; then
            echo "Seed $SEED for task $TASK completed successfully!"
        else
            echo "Error: Seed $SEED for task $TASK failed!"
            exit 1
        fi
    done
    
    echo "All 5 seeds completed for task $TASK on fold $FOLD!"
    echo "Collecting and aggregating statistics..."
    
    # Collect statistics for multi-seed experiment
    echo "Attempting statistics collection for multi-seed experiment..."
    python ../collect_seed_statistics.py \
        --experiment_dir ../experiments \
        --model shaspec \
        --task $TASK \
        --fold $FOLD \
        --seeds ${SEEDS[@]} \
        --batch_size $BATCH_SIZE \
        --lr $LR \
        --patience $PATIENCE \
        --epochs $EPOCHS \
        --ehr_encoder $EHR_ENCODER \
        --cxr_encoder $CXR_ENCODER \
        --dim $DIM \
        --dropout $DROPOUT \
        --alpha $ALPHA \
        --beta $BETA \
        --nhead $NHEAD \
        --num_layers $NUM_LAYERS \
        --max_seq_len $MAX_SEQ_LEN \
        --ehr_n_head $EHR_N_HEAD \
        --ehr_n_layers $EHR_N_LAYERS \
        --num_classes $NUM_CLASSES \
        --output experiments/shaspec/$TASK/shaspec_${TASK}_fold${FOLD}_5seeds_statistics.yaml
    
    if [ $? -eq 0 ]; then
        echo "Statistics collection completed successfully for task $TASK!"
    else
        echo "Warning: Statistics collection failed for task $TASK. Using alternative method..."
        
        # Fallback: Use the direct experiment directory statistics collection
        if [ -d "experiments/shaspec/$TASK/lightning_logs" ]; then
            echo "Using direct experiment directory statistics collection..."
            
            # Create a simple statistics aggregation for different seeds
            echo "Creating seed-based statistics summary..."
            
            # Calculate basic statistics across seeds
            python -c "
import os
import yaml
import pandas as pd
import numpy as np
from pathlib import Path

# Define paths and parameters
exp_dir = Path('experiments/shaspec/$TASK/lightning_logs')
seeds = [${SEEDS[@]}]
fold = $FOLD

# Collect metrics from each seed
all_metrics = []
for seed in seeds:
    # Look for experiment directories with this seed
    seed_dirs = list(exp_dir.glob(f'*seed_{seed}*'))
    if not seed_dirs:
        # Fallback: look for any directory and check version files
        seed_dirs = list(exp_dir.glob('*'))
    
    for seed_dir in seed_dirs:
        metrics_file = seed_dir / 'metrics.csv'
        if metrics_file.exists():
            try:
                df = pd.read_csv(metrics_file)
                if 'test_epoch' in df.columns:
                    test_metrics = df.dropna(subset=['test_epoch'])
                    if not test_metrics.empty:
                        last_test = test_metrics.iloc[-1]
                        all_metrics.append({
                            'seed': seed,
                            'metrics': last_test.to_dict()
                        })
                        break
            except Exception as e:
                print(f'Error reading {metrics_file}: {e}')

# Calculate statistics
if all_metrics:
    # Extract metric names (excluding non-numeric columns)
    metric_names = set()
    for m in all_metrics:
        for key in m['metrics'].keys():
            if key not in ['epoch', 'test_epoch', 'step'] and isinstance(m['metrics'][key], (int, float)):
                metric_names.add(key)
    
    # Calculate mean and std for each metric
    statistics = {}
    for metric in metric_names:
        values = []
        for m in all_metrics:
            if metric in m['metrics'] and not pd.isna(m['metrics'][metric]):
                values.append(m['metrics'][metric])
        
        if values:
            statistics[metric] = {
                'mean': float(np.mean(values)),
                'std': float(np.std(values)),
                'min': float(np.min(values)),
                'max': float(np.max(values)),
                'count': len(values)
            }
    
    # Save results
    output_file = 'experiments/shaspec/$TASK/shaspec_${TASK}_fold${FOLD}_5seeds_statistics.yaml'
    os.makedirs(os.path.dirname(output_file), exist_ok=True)
    
    with open(output_file, 'w') as f:
        yaml.dump({
            'experiment': 'shaspec_${TASK}_fold${FOLD}_5seeds',
            'fold': $FOLD,
            'seeds': seeds,
            'statistics': statistics
        }, f, default_flow_style=False)
    
    print(f'Statistics saved to {output_file}')
    print(f'Processed {len(all_metrics)} seed experiments')
else:
    print('No valid metrics found across seeds')
"
        fi
    fi
    
    echo "Task $TASK training and evaluation completed for fold $FOLD with 5 seeds!"
    echo "=================================="
done

echo "All tasks completed successfully!"
echo "Results can be found in experiments/shaspec/[task_name]/"
echo "Each seed experiment creates a separate checkpoint and log directory"
echo "Aggregated statistics are saved with '_5seeds_statistics.yaml' suffix"