#!/bin/bash

echo "Starting 5-seed training for Transformer model on fold 1..."

# Define experiment parameters
BATCH_SIZE=16
LR=0.0001
PATIENCE=10
EPOCHS=50
DROPOUT=0.2
FOLD=1                     # Fixed fold 1
GPU=1                      # GPU device number
MATCHED=false
USE_DEMOGRAPHICS=false
CROSS_EVAL=""              # Set to "matched_to_full" or "full_to_matched" if needed

# Transformer-specific parameters
D_MODEL=256                # Model dimension (equivalent to hidden_size)
N_HEAD=4                   # Number of attention heads
N_LAYERS=1                 # Number of transformer layers

# Seeds configuration (including 42)
SEEDS=(42 123 1234)

# Tasks configuration
TASKS=("los")

for TASK in "${TASKS[@]}"
do
    echo "Training Transformer model for task: $TASK"
    
    # Set task-specific parameters
    if [ "$TASK" = "phenotype" ]; then
        NUM_CLASSES=25
        VISION_NUM_CLASSES=25
        INPUT_DIM=498
        LABELS_SET=phenotype
    elif [ "$TASK" = "mortality" ]; then
        NUM_CLASSES=1
        VISION_NUM_CLASSES=1
        INPUT_DIM=498
        LABELS_SET=mortality
    elif [ "$TASK" = "los" ]; then
        NUM_CLASSES=7
        VISION_NUM_CLASSES=7
        INPUT_DIM=49
        LABELS_SET=los
    else
        echo "Error: Unknown task $TASK"
        exit 1
    fi
    
    for SEED in "${SEEDS[@]}"
    do
        echo "Training with seed $SEED for task $TASK on fold $FOLD..."
        
        # Build base command
        CMD="python ../main.py \
            --model transformer \
            --mode train \
            --task $TASK \
            --fold $FOLD \
            --batch_size $BATCH_SIZE \
            --lr $LR \
            --patience $PATIENCE \
            --epochs $EPOCHS \
            --dropout $DROPOUT \
            --seed $SEED \
            --d_model $D_MODEL \
            --n_head $N_HEAD \
            --n_layers $N_LAYERS \
            --input_dim $INPUT_DIM \
            --num_classes $NUM_CLASSES \
            --gpu $GPU"
        
        # Add conditional parameters
        if [ "$MATCHED" = "true" ]; then
            CMD="$CMD --matched"
        fi
        
        if [ "$USE_DEMOGRAPHICS" = "true" ]; then
            CMD="$CMD --use_demographics"
        fi
        
        if [ -n "$CROSS_EVAL" ]; then
            CMD="$CMD --cross_eval $CROSS_EVAL"
        fi
        
        echo "Running command: $CMD"
        eval $CMD
        
        if [ $? -eq 0 ]; then
            echo "Seed $SEED for task $TASK completed successfully!"
        else
            echo "Error: Seed $SEED for task $TASK failed!"
            exit 1
        fi
    done
    
    echo "All 5 seeds completed for task $TASK on fold $FOLD!"
    echo "Task $TASK training and evaluation completed for fold $FOLD with 5 seeds!"
    echo "=================================="
done

echo "All tasks completed successfully!"
echo "Results can be found in experiments/transformer/[task_name]/"
echo "Each seed experiment creates a separate checkpoint and log directory"