#!/bin/bash

echo "Starting MedFuse Model Testing..."

# Define experiment parameters (should match training parameters)
BATCH_SIZE=16
LR=0.0001
PATIENCE=10
EPOCHS=50
DROPOUT=0.2
FOLD=1                     # Fixed fold 1
GPU=1                      # GPU device number
PRETRAINED=true

# Testing configuration
USE_DEMOGRAPHICS=false
CROSS_EVAL="matched_to_full"              # Change this as needed

# MedFuse specific parameters (should match training)
EHR_ENCODER="lstm"           # Options: 'lstm', 'transformer', 'drfuse'
CXR_ENCODER="resnet50"       # Options: 'resnet50', 'vit_b_16', 'medfuse_cxr'
DIM=256                    # Feature dimension
LAYERS=1                   # Number of LSTM layers
VISION_BACKBONE="resnet50"   # Vision backbone
FUSION_TYPE="lstm"           # Fusion type: 'early', 'late', 'uni', 'lstm'
DRFUSE_ENCODER=false       # Use DrFuse encoder components

# Seeds configuration (should match training seeds)
SEEDS=(42)

# Tasks configuration
TASKS=("mortality")

# Function to find checkpoint file dynamically
find_checkpoint() {
    local checkpoint_dir="$1"
    local checkpoint_file=$(find "$checkpoint_dir" -name "*.ckpt" -type f | head -1)
    echo "$checkpoint_file"
}

# Function to determine training data config for checkpoint path
get_training_data_config() {
    local cross_eval="$1"
    if [ "$cross_eval" = "matched_to_full" ]; then
        echo "matched"
    elif [ "$cross_eval" = "full_to_matched" ]; then
        echo "full"
    else
        if [ "$MATCHED" = "true" ]; then
            echo "matched"
        else
            echo "full"
        fi
    fi
}

# 创建根目录名称：medfuse-{cross_eval}
ROOT_DIR="../experiments/medfuse-${CROSS_EVAL}"

echo "📁 All results will be saved under: $ROOT_DIR"

for TASK in "${TASKS[@]}"
do
    if [ -n "$CROSS_EVAL" ]; then
        echo "Testing MedFuse model for task: $TASK with cross evaluation: $CROSS_EVAL"
    else
        echo "Testing MedFuse model for task: $TASK (normal test)"
    fi
    
    # Set task-specific parameters
    if [ "$TASK" = "phenotype" ]; then
        NUM_CLASSES=25
        VISION_NUM_CLASSES=25
        INPUT_DIM=498
        LABELS_SET="phenotype"
    elif [ "$TASK" = "mortality" ]; then
        NUM_CLASSES=1
        VISION_NUM_CLASSES=1
        INPUT_DIM=498
        LABELS_SET="mortality"
    elif [ "$TASK" = "los" ]; then
        NUM_CLASSES=10
        VISION_NUM_CLASSES=10
        INPUT_DIM=498
        LABELS_SET="los"
    fi
    
    for SEED in "${SEEDS[@]}"
    do
        echo "Testing with seed $SEED for task $TASK on fold $FOLD..."
        
        # Determine the training data config for finding checkpoint
        TRAINING_DATA_CONFIG=$(get_training_data_config "$CROSS_EVAL")
        
        CHECKPOINT_DIR="../experiments/medfuse/$TASK/lightning_logs/MEDFUSE-model_medfuse-fusion_type_${FUSION_TYPE}-task_${TASK}-fold_${FOLD}-batch_size_${BATCH_SIZE}-lr_${LR}-patience_${PATIENCE}-epochs_${EPOCHS}-dropout_${DROPOUT}-seed_${SEED}-align_0.0-pretrained_True-data_config_${TRAINING_DATA_CONFIG}/checkpoints"
        
        # Find checkpoint file
        CHECKPOINT_FILE=$(find_checkpoint "$CHECKPOINT_DIR")
        
        if [ ! -f "$CHECKPOINT_FILE" ]; then
            echo "❌ Error: Checkpoint not found in $CHECKPOINT_DIR"
            echo "Available directories:"
            ls -la "../experiments/medfuse/$TASK/lightning_logs/" | grep "seed_${SEED}" || echo "No directories found for seed $SEED"
            continue
        fi
        
        echo "✅ Found checkpoint: $CHECKPOINT_FILE"
        
        # 创建自定义输出目录：root_dir/task_name
        CUSTOM_LOG_DIR="$ROOT_DIR/$TASK"
        
        # Build base command with custom log_dir
        CMD="python ../main.py \
            --model medfuse \
            --mode test \
            --task $TASK \
            --fold $FOLD \
            --batch_size $BATCH_SIZE \
            --lr $LR \
            --patience $PATIENCE \
            --epochs $EPOCHS \
            --dropout $DROPOUT \
            --seed $SEED \
            --ehr_encoder $EHR_ENCODER \
            --cxr_encoder $CXR_ENCODER \
            --dim $DIM \
            --layers $LAYERS \
            --vision_backbone $VISION_BACKBONE \
            --fusion_type $FUSION_TYPE \
            --input_dim $INPUT_DIM \
            --num_classes $NUM_CLASSES \
            --vision_num_classes $VISION_NUM_CLASSES \
            --labels_set $LABELS_SET \
            --gpu $GPU \
            --checkpoint_path $CHECKPOINT_FILE \
            --log_dir $CUSTOM_LOG_DIR"
        
        # Add conditional parameters
        if [ "$PRETRAINED" = "true" ]; then
            CMD="$CMD --pretrained"
        fi
        
        if [ "$DRFUSE_ENCODER" = "true" ]; then
            CMD="$CMD --drfuse_encoder"
        fi
        
        if [ "$USE_DEMOGRAPHICS" = "true" ]; then
            CMD="$CMD --use_demographics"
        fi
        
        # Add cross evaluation parameter if specified
        if [ -n "$CROSS_EVAL" ]; then
            CMD="$CMD --cross_eval $CROSS_EVAL"
        else
            if [ "$TRAINING_DATA_CONFIG" = "matched" ]; then
                CMD="$CMD --matched"
            fi
        fi

    done

done
