#!/bin/bash

echo "Starting SMIL Model Testing..."

# Define experiment parameters (should match training parameters)
BATCH_SIZE=16
LR=0.0001
PATIENCE=10
EPOCHS=50
DROPOUT=0.2
FOLD=1                     # Fixed fold 1
GPU=2                      # GPU device number
PRETRAINED=true

# Testing configuration
USE_DEMOGRAPHICS=false
# Set CROSS_EVAL to enable cross evaluation:
# - "" or empty: normal test (same data as training)
# - "matched_to_full": test models trained on matched data with full data
# - "full_to_matched": test models trained on full data with matched data
CROSS_EVAL="matched_to_full"              # Change this as needed

# SMIL-specific parameters (should match training)
EHR_ENCODER="transformer"         # Options: "lstm", "transformer" 
CXR_ENCODER="resnet50"     # Options: "resnet50", "vit_b_16"
FUSION_TYPE="smil"         # SMIL fusion type - REQUIRED for checkpoint path
HIDDEN_DIM=256             # Hidden dimension for SMIL
INNER_LOOP=2               # Number of inner loops for meta-learning
LR_INNER=0.0005106366481617694     # Inner learning rate
MC_SIZE=10                 # Monte Carlo size
ALPHA=0.1460970829212307                 # Feature distillation weight
BETA=0.1829896904844842   # EHR mean distillation weight
TEMPERATURE=2.1659838411477903            # Knowledge distillation temperature
N_CLUSTERS=10              # Number of clusters for CXR k-means

# EHR LSTM parameters (used when EHR_ENCODER="lstm")
EHR_NUM_LAYERS=1
EHR_BIDIRECTIONAL=true

# EHR Transformer parameters (used when EHR_ENCODER="transformer")
EHR_N_HEAD=4
EHR_N_LAYERS=1
MAX_LEN=500

# Seeds configuration (should match training seeds)
SEEDS=(42 123 1234)

# Tasks configuration
TASKS=("mortality")

# Function to find checkpoint file dynamically
find_checkpoint() {
    local checkpoint_dir="$1"
    local checkpoint_file=$(find "$checkpoint_dir" -name "*.ckpt" -type f | head -1)
    echo "$checkpoint_file"
}

# Function to determine training data config for checkpoint path
get_training_data_config() {
    local cross_eval="$1"
    if [ "$cross_eval" = "matched_to_full" ]; then
        echo "matched"
    elif [ "$cross_eval" = "full_to_matched" ]; then
        echo "full"
    else
        # For normal testing, determine based on current settings
        if [ "$MATCHED" = "true" ]; then
            echo "matched"
        else
            echo "full"
        fi
    fi
}

# 创建根目录名称：smil-{cross_eval}
ROOT_DIR="../experiments/smil-${CROSS_EVAL}"

echo "📁 All results will be saved under: $ROOT_DIR"

for TASK in "${TASKS[@]}"
do
    if [ -n "$CROSS_EVAL" ]; then
        echo "Testing SMIL model for task: $TASK with cross evaluation: $CROSS_EVAL"
    else
        echo "Testing SMIL model for task: $TASK (normal test)"
    fi
    
    # Set task-specific parameters
    if [ "$TASK" = "phenotype" ]; then
        NUM_CLASSES=25
        INPUT_DIM=49          # Note: SMIL uses 49 for this config
    elif [ "$TASK" = "mortality" ]; then
        NUM_CLASSES=1
        INPUT_DIM=49
    elif [ "$TASK" = "los" ]; then
        NUM_CLASSES=10
        INPUT_DIM=49
    fi
    
    # Check if CXR k-means centers are available for cross evaluation
    if [ -n "$CROSS_EVAL" ]; then
        if [ "$CROSS_EVAL" = "matched_to_full" ]; then
            TEST_DATA_TYPE="full"
        elif [ "$CROSS_EVAL" = "full_to_matched" ]; then
            TEST_DATA_TYPE="matched"
        fi
    else
        TEST_DATA_TYPE="matched"
        if [ "$MATCHED" = "false" ]; then
            TEST_DATA_TYPE="full"
        fi
    fi
    
    echo "Checking CXR k-means centers availability for testing (data type: $TEST_DATA_TYPE)..."
    CXR_MEAN_FILE="../models/smil/cxr_mean/cxr_mean_fold${FOLD}_${TEST_DATA_TYPE}_${CXR_ENCODER}_${N_CLUSTERS}clusters.npy"
    if [ ! -f "$CXR_MEAN_FILE" ]; then
        echo "Warning: CXR k-means file not found for testing: $CXR_MEAN_FILE"
        echo "Please run CXR k-means computation for test data type:"
        echo "cd ../models/smil && ./compute_cxr_kmeans.sh --task $TASK --folds $FOLD --data_type $TEST_DATA_TYPE --cxr_encoder $CXR_ENCODER --n_clusters $N_CLUSTERS --gpu $GPU"
        echo "Continuing without k-means check (assuming file will be created during testing)..."
    else
        echo "CXR k-means centers are available for testing!"
    fi
    
    for SEED in "${SEEDS[@]}"
    do
        echo "Testing with seed $SEED for task $TASK on fold $FOLD..."
        
        # Determine the training data config for finding checkpoint
        TRAINING_DATA_CONFIG=$(get_training_data_config "$CROSS_EVAL")
        
        # Build checkpoint directory path with fusion_type parameter
        CHECKPOINT_DIR="../experiments/smil/$TASK/lightning_logs/SMIL-model_smil-fusion_type_${FUSION_TYPE}-task_${TASK}-fold_${FOLD}-batch_size_${BATCH_SIZE}-lr_${LR}-patience_${PATIENCE}-epochs_${EPOCHS}-dropout_${DROPOUT}-seed_${SEED}-pretrained_True-data_config_${TRAINING_DATA_CONFIG}/checkpoints"
        
        # Find checkpoint file
        CHECKPOINT_FILE=$(find_checkpoint "$CHECKPOINT_DIR")
        
        if [ ! -f "$CHECKPOINT_FILE" ]; then
            echo "❌ Error: Checkpoint not found in $CHECKPOINT_DIR"
            echo "Available directories:"
            ls -la "../experiments/smil/$TASK/lightning_logs/" | grep "seed_${SEED}" || echo "No directories found for seed $SEED"
            continue
        fi
        
        echo "✅ Found checkpoint: $CHECKPOINT_FILE"
        
        # 创建自定义输出目录：root_dir/task_name
        CUSTOM_LOG_DIR="$ROOT_DIR/$TASK"
        
        # Build base command with custom log_dir
        CMD="python ../main.py \
            --model smil \
            --fusion_type $FUSION_TYPE \
            --mode test \
            --task $TASK \
            --fold $FOLD \
            --batch_size $BATCH_SIZE \
            --lr $LR \
            --patience $PATIENCE \
            --epochs $EPOCHS \
            --dropout $DROPOUT \
            --seed $SEED \
            --hidden_dim $HIDDEN_DIM \
            --input_dim $INPUT_DIM \
            --num_classes $NUM_CLASSES \
            --ehr_encoder $EHR_ENCODER \
            --cxr_encoder $CXR_ENCODER \
            --inner_loop $INNER_LOOP \
            --lr_inner $LR_INNER \
            --mc_size $MC_SIZE \
            --alpha $ALPHA \
            --beta $BETA \
            --temperature $TEMPERATURE \
            --n_clusters $N_CLUSTERS \
            --gpu $GPU \
            --checkpoint_path $CHECKPOINT_FILE \
            --log_dir $CUSTOM_LOG_DIR"
        
        # Add encoder-specific parameters
        if [ "$EHR_ENCODER" = "lstm" ]; then
            CMD="$CMD --ehr_num_layers $EHR_NUM_LAYERS"
            if [ "$EHR_BIDIRECTIONAL" = "true" ]; then
                CMD="$CMD --ehr_bidirectional"
            fi
        elif [ "$EHR_ENCODER" = "transformer" ]; then
            CMD="$CMD --ehr_n_head $EHR_N_HEAD --ehr_n_layers $EHR_N_LAYERS --max_len $MAX_LEN"
        fi
        
        # Add conditional parameters
        if [ "$PRETRAINED" = "true" ]; then
            CMD="$CMD --pretrained"
        fi
        
        if [ "$USE_DEMOGRAPHICS" = "true" ]; then
            CMD="$CMD --use_demographics"
        fi
        
        # Add cross evaluation parameter if specified
        if [ -n "$CROSS_EVAL" ]; then
            CMD="$CMD --cross_eval $CROSS_EVAL"
        else
            # For normal testing, add matched parameter if needed
            if [ "$TRAINING_DATA_CONFIG" = "matched" ]; then
                CMD="$CMD --matched"
            fi
        fi
        

    done

done
