#!/bin/bash

echo "Starting UMSE Model Testing..."

# Define experiment parameters (should match training parameters)
BATCH_SIZE=16
LR=0.0001
PATIENCE=10
EPOCHS=50
DROPOUT=0.2
FOLD=1                     # Fixed fold 1
GPU=0                      # GPU device number

# Testing configuration
USE_DEMOGRAPHICS=false
CROSS_EVAL="matched_to_full"              # Change this as needed

# UMSE-specific parameters (should match training)
D_MODEL=256                # Model dimension
VARIABLES_NUM=49           # Number of variables (from config)
NUM_LAYERS=1               # Number of transformer layers
NUM_HEADS=4                # Number of attention heads  
N_MODALITY=2               # Number of modalities (EHR, CXR)
BOTTLENECKS_N=1            # Number of bottlenecks for MBT
MAX_EHR_LEN=500            # Maximum EHR sequence length

# Seeds configuration (should match training seeds)
SEEDS=(42 123 1234)

# Tasks configuration
TASKS=("phenotype")

# Function to find checkpoint file dynamically
find_checkpoint() {
    local checkpoint_dir="$1"
    local checkpoint_file=$(find "$checkpoint_dir" -name "*.ckpt" -type f | head -1)
    echo "$checkpoint_file"
}

# Function to determine training data config for checkpoint path
get_training_data_config() {
    local cross_eval="$1"
    if [ "$cross_eval" = "matched_to_full" ]; then
        echo "matched"
    elif [ "$cross_eval" = "full_to_matched" ]; then
        echo "full"
    else
        if [ "$MATCHED" = "true" ]; then
            echo "matched"
        else
            echo "full"
        fi
    fi
}

# 创建根目录名称：umse-{cross_eval}
ROOT_DIR="../experiments/umse-${CROSS_EVAL}"

echo " All results will be saved under: $ROOT_DIR"

for TASK in "${TASKS[@]}"
do
    if [ -n "$CROSS_EVAL" ]; then
        echo "Testing UMSE model for task: $TASK with cross evaluation: $CROSS_EVAL"
    else
        echo "Testing UMSE model for task: $TASK (normal test)"
    fi
    
    # Set task-specific parameters
    if [ "$TASK" = "phenotype" ]; then
        NUM_CLASSES=25
        INPUT_DIM=49
    elif [ "$TASK" = "mortality" ]; then
        NUM_CLASSES=1
        INPUT_DIM=49
    elif [ "$TASK" = "los" ]; then
        NUM_CLASSES=10
        INPUT_DIM=49
    fi
    
    for SEED in "${SEEDS[@]}"
    do
        echo "Testing with seed $SEED for task $TASK on fold $FOLD..."
        
        # Determine the training data config for finding checkpoint
        TRAINING_DATA_CONFIG=$(get_training_data_config "$CROSS_EVAL")
        
        # Build checkpoint directory path
        CHECKPOINT_DIR="../experiments/umse/$TASK/lightning_logs/UMSE-model_umse-task_${TASK}-fold_${FOLD}-batch_size_${BATCH_SIZE}-lr_${LR}-patience_${PATIENCE}-epochs_${EPOCHS}-dropout_${DROPOUT}-seed_${SEED}-data_config_${TRAINING_DATA_CONFIG}/checkpoints"
        
        # Find checkpoint file
        CHECKPOINT_FILE=$(find_checkpoint "$CHECKPOINT_DIR")
        
        if [ ! -f "$CHECKPOINT_FILE" ]; then
            echo "❌ Error: Checkpoint not found in $CHECKPOINT_DIR"
            echo "Available directories:"
            ls -la "../experiments/umse/$TASK/lightning_logs/" | grep "seed_${SEED}" || echo "No directories found for seed $SEED"
            continue
        fi
        
        echo "✅ Found checkpoint: $CHECKPOINT_FILE"
        
        # 创建自定义输出目录：root_dir/task_name
        CUSTOM_LOG_DIR="$ROOT_DIR/$TASK"
        
        # Build base command with custom log_dir
        CMD="python ../main.py \
            --model umse \
            --mode test \
            --task $TASK \
            --fold $FOLD \
            --batch_size $BATCH_SIZE \
            --lr $LR \
            --patience $PATIENCE \
            --epochs $EPOCHS \
            --dropout $DROPOUT \
            --seed $SEED \
            --d_model $D_MODEL \
            --variables_num $VARIABLES_NUM \
            --num_layers $NUM_LAYERS \
            --num_heads $NUM_HEADS \
            --n_modality $N_MODALITY \
            --bottlenecks_n $BOTTLENECKS_N \
            --max_ehr_len $MAX_EHR_LEN \
            --num_classes $NUM_CLASSES \
            --gpu $GPU \
            --checkpoint_path $CHECKPOINT_FILE \
            --log_dir $CUSTOM_LOG_DIR"
        
        if [ "$USE_DEMOGRAPHICS" = "true" ]; then
            CMD="$CMD --use_demographics"
        fi
        
        # Add cross evaluation parameter if specified
        if [ -n "$CROSS_EVAL" ]; then
            CMD="$CMD --cross_eval $CROSS_EVAL"
        else
            if [ "$TRAINING_DATA_CONFIG" = "matched" ]; then
                CMD="$CMD --matched"
            fi
        fi
        

    done

done
