#!/bin/bash

echo "Starting 5-seed training for Late Fusion model on fold 1..."

# Define experiment parameters
BATCH_SIZE=16
LR=0.0001
PATIENCE=10
EPOCHS=50
FOLD=1                     # Fixed fold 1
GPU=2                      # GPU device number
MATCHED=false
USE_DEMOGRAPHICS=false
CROSS_EVAL=""              # Set to "matched_to_full" or "full_to_matched" if needed

# Late Fusion specific parameters
EHR_ENCODER=transformer    # Options: 'lstm', 'transformer'
CXR_ENCODER=resnet50       # Options: 'resnet50', 'vit_b_16'
HIDDEN_SIZE=256            # Hidden dimension for both encoders
EHR_DROPOUT=0.2            # EHR encoder dropout
EHR_N_LAYERS=1             # Number of EHR encoder layers  
EHR_N_HEAD=4               # Number of attention heads for transformer
PRETRAINED=true            # Use pretrained CXR encoder
UNIMODAL_LOSS=false        # Whether to use unimodal loss

# Seeds configuration (including 42)
SEEDS=(42 123 1234)

# Tasks configuration
TASKS=("phenotype")

for TASK in "${TASKS[@]}"
do
    echo "Training Late Fusion model for task: $TASK"
    
    # Set task-specific parameters
    if [ "$TASK" = "phenotype" ]; then
        NUM_CLASSES=25
        INPUT_DIM=498
    elif [ "$TASK" = "mortality" ]; then
        NUM_CLASSES=1
        INPUT_DIM=498
    fi
    
    for SEED in "${SEEDS[@]}"
    do
        echo "Training with seed $SEED for task $TASK on fold $FOLD..."
        
        # Build base command
        CMD="python ../main.py \
            --model latefusion \
            --mode train \
            --task $TASK \
            --fold $FOLD \
            --batch_size $BATCH_SIZE \
            --lr $LR \
            --patience $PATIENCE \
            --epochs $EPOCHS \
            --seed $SEED \
            --ehr_encoder $EHR_ENCODER \
            --cxr_encoder $CXR_ENCODER \
            --hidden_size $HIDDEN_SIZE \
            --ehr_dropout $EHR_DROPOUT \
            --ehr_n_layers $EHR_N_LAYERS \
            --ehr_n_head $EHR_N_HEAD \
            --input_dim $INPUT_DIM \
            --num_classes $NUM_CLASSES \
            --gpu $GPU"
        
        # Add Late Fusion specific parameters
        if [ "$PRETRAINED" = "true" ]; then
            CMD="$CMD --pretrained"
        fi
        
        if [ "$UNIMODAL_LOSS" = "true" ]; then
            CMD="$CMD --unimodal_loss"
        fi
        
        # Add conditional parameters
        if [ "$MATCHED" = "true" ]; then
            CMD="$CMD --matched"
        fi
        
        if [ "$USE_DEMOGRAPHICS" = "true" ]; then
            CMD="$CMD --use_demographics"
        fi
        
        if [ -n "$CROSS_EVAL" ]; then
            CMD="$CMD --cross_eval $CROSS_EVAL"
        fi
        
        echo "Running command: $CMD"
        eval $CMD
        
        if [ $? -eq 0 ]; then
            echo "Seed $SEED for task $TASK completed successfully!"
        else
            echo "Error: Seed $SEED for task $TASK failed!"
            exit 1
        fi
    done
    
    echo "All 5 seeds completed for task $TASK on fold $FOLD!"
    echo "Task $TASK training and evaluation completed for fold $FOLD with 5 seeds!"
    echo "=================================="
done

echo "All tasks completed successfully!"
echo "Results can be found in experiments/latefusion/[task_name]/"
echo "Each seed experiment creates a separate checkpoint and log directory"