#!/bin/bash

echo "Starting FlexMoE Model Testing..."

# Define experiment parameters (should match training parameters)
BATCH_SIZE=16
LR=0.0001
PATIENCE=10
EPOCHS=50
DROPOUT=0.2
FOLD=1                     # Fixed fold 1
GPU=1                      # GPU device number
PRETRAINED=true

# Testing configuration
USE_DEMOGRAPHICS=false
# Set CROSS_EVAL to enable cross evaluation:
# - "" or empty: normal test (same data as training)
# - "matched_to_full": test models trained on matched data with full data
# - "full_to_matched": test models trained on full data with matched data
CROSS_EVAL="matched_to_full"              # Change this as needed

# FlexMoE-specific parameters (should match training)
EHR_ENCODER="transformer"  # Options: "lstm", "transformer"
CXR_ENCODER="resnet50"     # Options: "resnet50", "vit_b_16"
HIDDEN_DIM=256             # Hidden dimension for FlexMoE
NUM_PATCHES=16             # Number of patches for input tokenization
NUM_LAYERS=1               # Number of transformer layers
NUM_LAYERS_PRED=1          # Number of prediction layers
NUM_EXPERTS=16             # Number of experts in MoE
NUM_ROUTERS=1              # Number of routers
TOP_K=8                    # Number of experts to route to
NUM_HEADS=4                # Number of attention heads
GATE_LOSS_WEIGHT=0.059157941793120575      # Gate loss weight

# EHR LSTM parameters (used when EHR_ENCODER="lstm")
EHR_NUM_LAYERS=1
EHR_BIDIRECTIONAL=true

# EHR Transformer parameters (used when EHR_ENCODER="transformer")
EHR_N_HEAD=4
EHR_N_LAYERS=1

# Seeds configuration (should match training seeds)
SEEDS=(42 123 1234)

# Tasks configuration
TASKS=("phenotype")

# Function to find checkpoint file dynamically
find_checkpoint() {
    local checkpoint_dir="$1"
    local checkpoint_file=$(find "$checkpoint_dir" -name "*.ckpt" -type f | head -1)
    echo "$checkpoint_file"
}

# Function to determine training data config for checkpoint path
get_training_data_config() {
    local cross_eval="$1"
    if [ "$cross_eval" = "matched_to_full" ]; then
        echo "matched"
    elif [ "$cross_eval" = "full_to_matched" ]; then
        echo "full"
    else
        # For normal testing, determine based on current settings
        if [ "$MATCHED" = "true" ]; then
            echo "matched"
        else
            echo "full"
        fi
    fi
}

# 创建根目录名称：flexmoe-{cross_eval}
ROOT_DIR="../experiments/flexmoe-${CROSS_EVAL}"

echo "📁 All results will be saved under: $ROOT_DIR"

for TASK in "${TASKS[@]}"
do
    if [ -n "$CROSS_EVAL" ]; then
        echo "Testing FlexMoE model for task: $TASK with cross evaluation: $CROSS_EVAL"
    else
        echo "Testing FlexMoE model for task: $TASK (normal test)"
    fi
    
    # Set task-specific parameters
    if [ "$TASK" = "phenotype" ]; then
        NUM_CLASSES=25
        INPUT_DIM=498
    elif [ "$TASK" = "mortality" ]; then
        NUM_CLASSES=1
        INPUT_DIM=498
    elif [ "$TASK" = "los" ]; then
        NUM_CLASSES=10
        INPUT_DIM=498
    fi
    
    for SEED in "${SEEDS[@]}"
    do
        echo "Testing with seed $SEED for task $TASK on fold $FOLD..."
        
        # Determine the training data config for finding checkpoint
        TRAINING_DATA_CONFIG=$(get_training_data_config "$CROSS_EVAL")
        
        # Build checkpoint directory path (still use the full path to find existing checkpoints)
        CHECKPOINT_DIR="../experiments-m-m/flexmoe/$TASK/lightning_logs/FLEXMOE-model_flexmoe-task_${TASK}-fold_${FOLD}-batch_size_${BATCH_SIZE}-lr_${LR}-patience_${PATIENCE}-epochs_${EPOCHS}-dropout_${DROPOUT}-seed_${SEED}-pretrained_True-data_config_${TRAINING_DATA_CONFIG}/checkpoints"
        
        # Find checkpoint file
        CHECKPOINT_FILE=$(find_checkpoint "$CHECKPOINT_DIR")
        
        if [ ! -f "$CHECKPOINT_FILE" ]; then
            echo "❌ Error: Checkpoint not found in $CHECKPOINT_DIR"
            echo "Available directories:"
            ls -la "../experiments/flexmoe/$TASK/lightning_logs/" | grep "seed_${SEED}" || echo "No directories found for seed $SEED"
            continue
        fi
        
        echo "✅ Found checkpoint: $CHECKPOINT_FILE"
        
        # 创建自定义输出目录：root_dir/task_name
        CUSTOM_LOG_DIR="$ROOT_DIR/$TASK"
        
        # Build base command with custom log_dir
        CMD="python ../main.py \
            --model flexmoe \
            --mode test \
            --task $TASK \
            --fold $FOLD \
            --batch_size $BATCH_SIZE \
            --lr $LR \
            --patience $PATIENCE \
            --epochs $EPOCHS \
            --dropout $DROPOUT \
            --seed $SEED \
            --hidden_dim $HIDDEN_DIM \
            --input_dim $INPUT_DIM \
            --num_classes $NUM_CLASSES \
            --ehr_encoder $EHR_ENCODER \
            --cxr_encoder $CXR_ENCODER \
            --num_patches $NUM_PATCHES \
            --num_layers $NUM_LAYERS \
            --num_layers_pred $NUM_LAYERS_PRED \
            --num_experts $NUM_EXPERTS \
            --num_routers $NUM_ROUTERS \
            --top_k $TOP_K \
            --num_heads $NUM_HEADS \
            --gate_loss_weight $GATE_LOSS_WEIGHT \
            --gpu $GPU \
            --checkpoint_path $CHECKPOINT_FILE \
            --log_dir $CUSTOM_LOG_DIR"
        
        # Add encoder-specific parameters
        if [ "$EHR_ENCODER" = "lstm" ]; then
            CMD="$CMD --ehr_num_layers $EHR_NUM_LAYERS"
            if [ "$EHR_BIDIRECTIONAL" = "true" ]; then
                CMD="$CMD --ehr_bidirectional"
            fi
        elif [ "$EHR_ENCODER" = "transformer" ]; then
            CMD="$CMD --ehr_n_head $EHR_N_HEAD --ehr_n_layers $EHR_N_LAYERS"
        fi
        
        # Add conditional parameters
        if [ "$PRETRAINED" = "true" ]; then
            CMD="$CMD --pretrained"
        fi
        
        if [ "$USE_DEMOGRAPHICS" = "true" ]; then
            CMD="$CMD --use_demographics"
        fi
        
        # Add cross evaluation parameter if specified
        if [ -n "$CROSS_EVAL" ]; then
            CMD="$CMD --cross_eval $CROSS_EVAL"
        else
            # For normal testing, add matched parameter if needed
            if [ "$TRAINING_DATA_CONFIG" = "matched" ]; then
                CMD="$CMD --matched"
            fi
        fi
        
    done
    
done
